package com.af.v4.system.common.datasource.dialects;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.StringTokenizer;

/**
 * String Utilities usually used inside of framework
 *
 * @since 1.0.0
 */
public class StrUtils {
    public static final String WHITESPACE = " \n\r\f\t";
    private static final SecureRandom random = new SecureRandom();
    private static final char[] ALPHABET = "0123456789abcdefghijklmnopqrstuvwxyz".toCharArray();

    private StrUtils() {
        // default constructor
    }

    /**
     * Check whether the given String is empty.
     */
    public static boolean isEmpty(Object str) {
        return str == null || "".equals(str);
    }

    /**
     * Judge if is or not white space characters
     */
    public static boolean isBlank(String str) {
        int strLen;
        if (str == null || (strLen = str.length()) == 0) {
            return true;
        }
        for (int i = 0; i < strLen; i++) {
            if ((!Character.isWhitespace(str.charAt(i)))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Return true if is null object or blank string
     */
    public static boolean isBlankObject(Object obj) {
        if (obj == null) return true;
        if (obj instanceof String) return isBlank((String) obj);
        return false;
    }

    /**
     * Check that the given CharSequence is neither {@code null} nor of length 0.
     * Note: Will return {@code true} for a CharSequence that purely consists of
     * whitespace.
     * <p>
     *
     * <pre class="code">
     * StrUtils.hasLength(null) = false
     * StrUtils.hasLength("") = false
     * StrUtils.hasLength(" ") = true
     * StrUtils.hasLength("Hello") = true
     * </pre>
     *
     * @param str the CharSequence to check (may be {@code null})
     * @return {@code true} if the CharSequence is not null and has length
     */
    public static boolean hasLength(CharSequence str) {
        return str != null && !str.isEmpty();
    }

    /**
     * Check that the given String is neither {@code null} nor of length 0. Note:
     * Will return {@code true} for a String that purely consists of whitespace.
     *
     * @param str the String to check (may be {@code null})
     * @return {@code true} if the String is not null and has length
     * @see #hasLength(CharSequence)
     */
    public static boolean isEmpty(String str) {
        return !hasLength(str);
    }

    /**
     * Check whether the given CharSequence contains any whitespace characters.
     *
     * @param str the CharSequence to check (may be {@code null})
     * @return {@code true} if the CharSequence is not empty and contains at least 1
     * whitespace character
     * @see Character#isWhitespace
     */
    public static boolean containsWhitespace(CharSequence str) {
        if (!hasLength(str)) {
            return false;
        }
        int strLen = str.length();
        for (int i = 0; i < strLen; i++) {
            if (Character.isWhitespace(str.charAt(i))) {
                return true;
            }
        }
        return false;
    }

    /**
     * Check whether the given String contains any whitespace characters.
     *
     * @param str the String to check (may be {@code null})
     * @return {@code true} if the String is not empty and contains at least 1
     * whitespace character
     * @see #containsWhitespace(CharSequence)
     */
    public static boolean containsWhitespace(String str) {
        return containsWhitespace((CharSequence) str);
    }

    /**
     * Trim leading and trailing whitespace from the given String.
     *
     * @param str the String to check
     * @return the trimmed String
     * @see Character#isWhitespace
     */
    public static String trimWhitespace(String str) {
        if (!isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        while (!sb.isEmpty() && Character.isWhitespace(sb.charAt(0))) {
            sb.deleteCharAt(0);
        }
        while (!sb.isEmpty() && Character.isWhitespace(sb.charAt(sb.length() - 1))) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    /**
     * Trim <i>all</i> whitespace from the given String: leading, trailing, and in
     * between characters.
     *
     * @param str the String to check
     * @return the trimmed String
     * @see Character#isWhitespace
     */
    public static String trimAllWhitespace(String str) {
        if (isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        int index = 0;
        while (sb.length() > index) {
            if (Character.isWhitespace(sb.charAt(index))) {
                sb.deleteCharAt(index);
            } else {
                index++;
            }
        }
        return sb.toString();
    }

    /**
     * Trim leading whitespace from the given String.
     *
     * @param str the String to check
     * @return the trimmed String
     * @see Character#isWhitespace
     */
    public static String trimLeadingWhitespace(String str) {
        if (isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        while (!sb.isEmpty() && Character.isWhitespace(sb.charAt(0))) {
            sb.deleteCharAt(0);
        }
        return sb.toString();
    }

    /**
     * Trim trailing whitespace from the given String.
     *
     * @param str the String to check
     * @return the trimmed String
     * @see Character#isWhitespace
     */
    public static String trimTrailingWhitespace(String str) {
        if (isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        while (!sb.isEmpty() && Character.isWhitespace(sb.charAt(sb.length() - 1))) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    /**
     * Trim all occurrences of the supplied leading character from the given String.
     *
     * @param str              the String to check
     * @param leadingCharacter the leading character to be trimmed
     * @return the trimmed String
     */
    public static String trimLeadingCharacter(String str, char leadingCharacter) {
        if (isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        while (!sb.isEmpty() && sb.charAt(0) == leadingCharacter) {
            sb.deleteCharAt(0);
        }
        return sb.toString();
    }

    /**
     * Trim all occurrences of the supplied trailing character from the given
     * String.
     *
     * @param str               the String to check
     * @param trailingCharacter the trailing character to be trimmed
     * @return the trimmed String
     */
    public static String trimTrailingCharacter(String str, char trailingCharacter) {
        if (isEmpty(str)) {
            return str;
        }
        StringBuilder sb = new StringBuilder(str);
        while (!sb.isEmpty() && sb.charAt(sb.length() - 1) == trailingCharacter) {
            sb.deleteCharAt(sb.length() - 1);
        }
        return sb.toString();
    }

    /**
     * Test if the given String starts with the specified prefix, ignoring
     * upper/lower case.
     *
     * @param str    the String to check
     * @param prefix the prefix to look for
     * @see String#startsWith
     */
    public static boolean startsWithIgnoreCase(String str, String prefix) {
        if (str == null || prefix == null) {
            return false;
        }
        if (str.startsWith(prefix)) {
            return true;
        }
        if (str.length() < prefix.length()) {
            return false;
        }
        String lcStr = str.substring(0, prefix.length()).toLowerCase();
        String lcPrefix = prefix.toLowerCase();
        return lcStr.equals(lcPrefix);
    }

    /**
     * Find if exist searchStr in str ignore case
     */
    public static boolean containsIgnoreCase(String str, String searchStr) {
        if (str == null || searchStr == null) return false;
        final int length = searchStr.length();
        if (length == 0) return true;
        for (int i = str.length() - length; i >= 0; i--) {
            if (str.regionMatches(true, i, searchStr, 0, length)) return true;
        }
        return false;
    }

    /**
     * Replace all occurrences of a substring within a string with another string.
     *
     * @param originString The original String
     * @param oldPattern   old String Pattern to replace
     * @param newPattern   new String pattern to insert
     * @return a String with the replacements
     */
    public static String replace(String originString, String oldPattern, String newPattern) {
        if (isEmpty(originString) || isEmpty(oldPattern) || newPattern == null) {
            return originString;
        }
        StringBuilder sb = new StringBuilder();
        int pos = 0;
        int index = originString.indexOf(oldPattern);
        int patLen = oldPattern.length();
        while (index >= 0) {
            sb.append(originString, pos, index);
            sb.append(newPattern);
            pos = index + patLen;
            index = originString.indexOf(oldPattern, pos);
        }
        sb.append(originString.substring(pos));
        return sb.toString();
    }

    /**
     * Replace first occurrences of a substring within a string with another string.
     *
     * @param originString The original String
     * @param oldPattern   old String Pattern to replace
     * @param newPattern   new String pattern to insert
     * @return a String with the replacements
     */
    public static String replaceFirst(String originString, String oldPattern, String newPattern) {
        if (isEmpty(originString) || isEmpty(oldPattern) || newPattern == null) {
            return originString;
        }
        StringBuilder sb = new StringBuilder();
        int pos = 0;
        int index = originString.indexOf(oldPattern);
        int patLen = oldPattern.length();
        if (index >= 0) {
            sb.append(originString, pos, index);
            sb.append(newPattern);
            pos = index + patLen;
        }
        sb.append(originString.substring(pos));
        return sb.toString();
    }

    /**
     * Replace all sub strings ignore case <br/>
     * replaceIgnoreCase("AbcDECd", "Cd", "FF") = "AbFFEFF"
     */
    public static String replaceIgnoreCase(String text, String findtxt, String replacetxt) {
        if (text == null) return null;
        String str = text;
        if (findtxt == null || findtxt.isEmpty()) {
            return str;
        }
        if (findtxt.length() > str.length()) {
            return str;
        }
        int counter = 0;
        String thesubstr;
        while ((counter < str.length()) && (str.substring(counter).length() >= findtxt.length())) {
            thesubstr = str.substring(counter, counter + findtxt.length());
            if (thesubstr.equalsIgnoreCase(findtxt)) {
                str = str.substring(0, counter) + replacetxt + str.substring(counter + findtxt.length());
                counter += replacetxt.length();
            } else {
                counter++;
            }
        }
        return str;
    }

    /**
     * Return first postion ignore case, return -1 if not found
     */
    public static int indexOfIgnoreCase(final String str, final String searchStr) {
        if (searchStr.isEmpty() || str.isEmpty()) {
            return str.indexOf(searchStr);
        }
        for (int i = 0; i < str.length(); ++i) {
            if (i + searchStr.length() > str.length()) {
                return -1;
            }
            int j = 0;
            int ii = i;
            while (ii < str.length() && j < searchStr.length()) {
                char c = Character.toLowerCase(str.charAt(ii));
                char c2 = Character.toLowerCase(searchStr.charAt(j));
                if (c != c2) {
                    break;
                }
                j++;
                ii++;
            }
            if (j == searchStr.length()) {
                return i;
            }
        }
        return -1;
    }

    /**
     * Return last sub-String position ignore case, return -1 if not found
     */
    public static int lastIndexOfIgnoreCase(String str, String searchStr) {
        if (searchStr.isEmpty() || str.isEmpty()) return -1;
        return str.toLowerCase().lastIndexOf(searchStr.toLowerCase());
    }

    /**
     * <p>
     * Splits the provided text into an array, separator specified. This is an
     * alternative to using StringTokenizer.
     * </p>
     *
     * <p>
     * The separator is not included in the returned String array. Adjacent
     * separators are treated as one separator. For more control over the split use
     * the StrTokenizer class.
     * </p>
     *
     * <p>
     * A {@code null} input String returns {@code null}.
     * </p>
     *
     * <pre>
     * StringUtils.split(null, *)         = null
     * StringUtils.split("", *)           = []
     * StringUtils.split("a.b.c", '.')    = ["a", "b", "c"]
     * StringUtils.split("a..b.c", '.')   = ["a", "b", "c"]
     * StringUtils.split("a:b:c", '.')    = ["a:b:c"]
     * StringUtils.split("a b c", ' ')    = ["a", "b", "c"]
     * </pre>
     *
     * @param str           the String to parse, may be null
     * @param separatorChar the character used as the delimiter
     * @return an array of parsed Strings, {@code null} if null String input
     * @since 2.0
     */
    public static String[] split(final String str, final char separatorChar) {
        return splitWorker(str, separatorChar, false);
    }

    /**
     * Performs the logic for the {@code split} and {@code splitPreserveAllTokens}
     * methods that do not return a maximum array length.
     *
     * @param str               the String to parse, may be {@code null}
     * @param separatorChar     the separate character
     * @param preserveAllTokens if {@code true}, adjacent separators are treated as empty token
     *                          separators; if {@code false}, adjacent separators are treated as
     *                          one separator.
     * @return an array of parsed Strings, {@code null} if null String input
     */
    private static String[] splitWorker(final String str, final char separatorChar, final boolean preserveAllTokens) {
        // Performance tuned for 2.0 (JDK1.4)

        if (str == null) {
            return new String[0];
        }
        final int len = str.length();
        if (len == 0) {
            return new String[0];
        }
        final List<String> list = new ArrayList<>();
        int i = 0;
        int start = 0;
        boolean match = false;
        while (i < len) {
            if (str.charAt(i) == separatorChar) {
                if (match || preserveAllTokens) {
                    list.add(str.substring(start, i));
                    match = false;
                }
                start = ++i;
                continue;
            }
            match = true;
            i++;
        }
        if (match || preserveAllTokens) {
            list.add(str.substring(start, i));
        }
        return list.toArray(new String[0]);
    }

    public static String[] split(String separators, String list) {
        return split(separators, list, false);
    }

    public static String[] split(String separators, String list, boolean include) {
        StringTokenizer tokens = new StringTokenizer(list, separators, include);
        String[] result = new String[tokens.countTokens()];
        int i = 0;
        while (tokens.hasMoreTokens()) {
            result[i++] = tokens.nextToken();
        }
        return result;
    }

    /**
     * <p>
     * Searches a String for substrings delimited by a start and end tag, returning
     * all matching substrings in an array.
     * </p>
     *
     * <p>
     * A {@code null} input String returns {@code null}. A {@code null} open/close
     * returns {@code null} (no match). An empty ("") open/close returns
     * {@code null} (no match).
     * </p>
     *
     * <pre>
     * StringUtils.substringsBetween("[a][b][c]", "[", "]") = ["a","b","c"]
     * StringUtils.substringsBetween(null, *, *)            = null
     * StringUtils.substringsBetween(*, null, *)            = null
     * StringUtils.substringsBetween(*, *, null)            = null
     * StringUtils.substringsBetween("", "[", "]")          = []
     * </pre>
     *
     * @param str   the String containing the substrings, null returns null, empty
     *              returns empty
     * @param open  the String identifying the start of the substring, empty returns
     *              null
     * @param close the String identifying the end of the substring, empty returns
     *              null
     * @return a String Array of substrings, or {@code null} if no match
     * @since 2.3
     */
    public static String[] substringsBetween(final String str, final String open, final String close) {
        if (str == null || isEmpty(open) || isEmpty(close)) {
            return new String[0];
        }
        final int strLen = str.length();
        if (strLen == 0) {
            return new String[0];
        }
        final int closeLen = close.length();
        final int openLen = open.length();
        final List<String> list = new ArrayList<>();
        int pos = 0;
        while (pos < strLen - closeLen) {
            int start = str.indexOf(open, pos);
            if (start < 0) {
                break;
            }
            start += openLen;
            final int end = str.indexOf(close, start);
            if (end < 0) {
                break;
            }
            list.add(str.substring(start, end));
            pos = end + closeLen;
        }
        if (list.isEmpty()) {
            return new String[0];
        }
        return list.toArray(new String[0]);
    }

    /**
     * Gets the String that is nested in between two Strings. Only the first match
     * is returned.
     * <p>
     * A <code>null</code> input String returns <code>null</code>. A
     * <code>null</code> open/close returns <code>null</code> (no match). An empty
     * ("") open and close returns an empty string.
     *
     * <pre>
     * StringUtils.substringBetween("wx[b]yz", "[", "]") = "b"
     * StringUtils.substringBetween(null, *, *)          = null
     * StringUtils.substringBetween(*, null, *)          = null
     * StringUtils.substringBetween(*, *, null)          = null
     * StringUtils.substringBetween("", "", "")          = ""
     * StringUtils.substringBetween("", "", "]")         = null
     * StringUtils.substringBetween("", "[", "]")        = null
     * StringUtils.substringBetween("yabcz", "", "")     = ""
     * StringUtils.substringBetween("yabcz", "y", "z")   = "abc"
     * StringUtils.substringBetween("yabczyabcz", "y", "z")   = "abc"
     * </pre>
     *
     * @param str   the String containing the substring, may be null
     * @param open  the String before the substring, may be null
     * @param close the String after the substring, may be null
     * @return the substring, <code>null</code> if no match
     * @since 2.0
     */
    public static String substringBetween(String str, String open, String close) {
        if (str == null || open == null || close == null) {
            return null;
        }
        int start = str.indexOf(open);
        if (start != -1) {
            int end = str.indexOf(close, start + open.length());
            if (end != -1) {
                return str.substring(start + open.length(), end);
            }
        }
        return null;
    }

    /**
     * <p>
     * Gets the substring before the first occurrence of a separator. The separator
     * is not returned.
     * </p>
     *
     * <p>
     * A {@code null} string input will return {@code null}. An empty ("") string
     * input will return the empty string. A {@code null} separator will return the
     * input string.
     * </p>
     *
     * <p>
     * If nothing is found, the string input is returned.
     * </p>
     *
     * <pre>
     * StringUtils.substringBefore(null, *)      = null
     * StringUtils.substringBefore("", *)        = ""
     * StringUtils.substringBefore("abc", "a")   = ""
     * StringUtils.substringBefore("abcba", "b") = "a"
     * StringUtils.substringBefore("abc", "c")   = "ab"
     * StringUtils.substringBefore("abc", "d")   = "abc"
     * StringUtils.substringBefore("abc", "")    = ""
     * StringUtils.substringBefore("abc", null)  = "abc"
     * </pre>
     *
     * @param str       the String to get a substring from, may be null
     * @param separator the String to search for, may be null
     * @return the substring before the first occurrence of the separator,
     * {@code null} if null String input
     */
    public static String substringBefore(final String str, final String separator) {
        if (isEmpty(str) || separator == null) {
            return str;
        }
        if (separator.isEmpty()) {
            return "";
        }
        final int pos = str.indexOf(separator);
        if (pos == -1) {
            return str;
        }
        return str.substring(0, pos);
    }

    /**
     * <p>
     * Gets the substring after the first occurrence of a separator. The separator
     * is not returned.
     * </p>
     *
     * <p>
     * A {@code null} string input will return {@code null}. An empty ("") string
     * input will return the empty string. A {@code null} separator will return the
     * empty string if the input string is not {@code null}.
     * </p>
     *
     * <p>
     * If nothing is found, the empty string is returned.
     * </p>
     *
     * <pre>
     * StringUtils.substringAfter(null, *)      = null
     * StringUtils.substringAfter("", *)        = ""
     * StringUtils.substringAfter(*, null)      = ""
     * StringUtils.substringAfter("abc", "a")   = "bc"
     * StringUtils.substringAfter("abcba", "b") = "cba"
     * StringUtils.substringAfter("abc", "c")   = ""
     * StringUtils.substringAfter("abc", "d")   = ""
     * StringUtils.substringAfter("abc", "")    = "abc"
     * </pre>
     *
     * @param str       the String to get a substring from, may be null
     * @param separator the String to search for, may be null
     * @return the substring after the first occurrence of the separator,
     * {@code null} if null String input
     */
    public static String substringAfter(final String str, final String separator) {
        if (isEmpty(str)) {
            return str;
        }
        if (separator == null) {
            return "";
        }
        final int pos = str.indexOf(separator);
        if (pos == -1) {
            return "";
        }
        return str.substring(pos + separator.length());
    }

    /**
     * <p>
     * Gets the substring after the last occurrence of a separator. The separator is
     * not returned.
     * </p>
     *
     * <p>
     * A {@code null} string input will return {@code null}. An empty ("") string
     * input will return the empty string. An empty or {@code null} separator will
     * return the empty string if the input string is not {@code null}.
     * </p>
     *
     * <p>
     * If nothing is found, the empty string is returned.
     * </p>
     *
     * <pre>
     * StringUtils.substringAfterLast(null, *)      = null
     * StringUtils.substringAfterLast("", *)        = ""
     * StringUtils.substringAfterLast(*, "")        = ""
     * StringUtils.substringAfterLast(*, null)      = ""
     * StringUtils.substringAfterLast("abc", "a")   = "bc"
     * StringUtils.substringAfterLast("abcba", "b") = "a"
     * StringUtils.substringAfterLast("abc", "c")   = ""
     * StringUtils.substringAfterLast("a", "a")     = ""
     * StringUtils.substringAfterLast("a", "z")     = ""
     * </pre>
     *
     * @param str       the String to get a substring from, may be null
     * @param separator the String to search for, may be null
     * @return the substring after the last occurrence of the separator,
     * {@code null} if null String input
     * @since 2.0
     */
    public static String substringAfterLast(final String str, final String separator) {
        if (isEmpty(str)) {
            return str;
        }
        if (isEmpty(separator)) {
            return "";
        }
        final int pos = str.lastIndexOf(separator);
        if (pos == -1 || pos == str.length() - separator.length()) {
            return "";
        }
        return str.substring(pos + separator.length());
    }

    /**
     * <p>
     * Gets the substring before the last occurrence of a separator. The separator
     * is not returned.
     * </p>
     *
     * <p>
     * A {@code null} string input will return {@code null}. An empty ("") string
     * input will return the empty string. An empty or {@code null} separator will
     * return the input string.
     * </p>
     *
     * <p>
     * If nothing is found, the string input is returned.
     * </p>
     *
     * <pre>
     * StringUtils.substringBeforeLast(null, *)      = null
     * StringUtils.substringBeforeLast("", *)        = ""
     * StringUtils.substringBeforeLast("abcba", "b") = "abc"
     * StringUtils.substringBeforeLast("abc", "c")   = "ab"
     * StringUtils.substringBeforeLast("a", "a")     = ""
     * StringUtils.substringBeforeLast("a", "z")     = "a"
     * StringUtils.substringBeforeLast("a", null)    = "a"
     * StringUtils.substringBeforeLast("a", "")      = "a"
     * </pre>
     *
     * @param str       the String to get a substring from, may be null
     * @param separator the String to search for, may be null
     * @return the substring before the last occurrence of the separator,
     * {@code null} if null String input
     * @since 2.0
     */
    public static String substringBeforeLast(final String str, final String separator) {
        if (isEmpty(str) || isEmpty(separator)) return str;
        final int pos = str.lastIndexOf(separator);
        if (pos == -1) return str;
        return str.substring(0, pos);
    }

    /**
     * <p>
     * Counts how many times the char appears in the given string.
     * </p>
     *
     * <p>
     * A {@code null} or empty ("") String input returns {@code 0}.
     * </p>
     *
     * <pre>
     * StringUtils.countMatches(null, *)       = 0
     * StringUtils.countMatches("", *)         = 0
     * StringUtils.countMatches("abba", 0)  = 0
     * StringUtils.countMatches("abba", 'a')   = 2
     * StringUtils.countMatches("abba", 'b')  = 2
     * StringUtils.countMatches("abba", 'x') = 0
     * </pre>
     *
     * @param str the CharSequence to check, may be null
     * @param ch  the char to count
     * @return the number of occurrences, 0 if the CharSequence is {@code null}
     * @since 3.4
     */
    public static int countMatches(final CharSequence str, final char ch) {
        if (isEmpty(str)) {
            return 0;
        }
        int count = 0;
        // We could also call str.toCharArray() for faster look ups but that
        // would generate more garbage.
        for (int i = 0; i < str.length(); i++) {
            if (ch == str.charAt(i)) {
                count++;
            }
        }
        return count;
    }

    public static char getRandomChar() {
        return ALPHABET[random.nextInt(32)];
    }

    public static String getRandomString(int length) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < length; i++) {
            sb.append(ALPHABET[random.nextInt(32)]);
        }
        return sb.toString();
    }

    /**
     * Compare 2 array
     *
     * @return true if each item equal
     */
    public static boolean arraysEqual(Object[] array1, Object[] array2) {
        if (array1 == null || array1.length == 0 || array2 == null || array2.length == 0)
            DialectException.throwEX("StrUtils arraysEqual() method can not compare empty arrays");
        for (int i = 0; i < array1.length; i++)
            if (!array1[i].equals(array2[i])) return false;
        return true;
    }

    /**
     * Change a Object array to "obj1,obj2...,objn" String
     */
    public static String arrayToStringButSkipFirst(Object[] array) {
        if (array == null) DialectException.throwEX("StrUtils arrayToString() method do not accept null parameter");
        StringBuilder sb = new StringBuilder();
        int i = 1;
        for (Object object : array) {
            if (i++ != 1) sb.append(object).append(",");

        }
        if (!sb.isEmpty()) sb.setLength(sb.length() - 1);
        return sb.toString();
    }

    /**
     * Change a Object array to "obj1,obj2...,objn" String
     */
    public static String arrayToString(Object[] array) {
        if (array == null) DialectException.throwEX("StrUtils arrayToString() method do not accept null parameter");
        StringBuilder sb = new StringBuilder();
        for (Object object : array)
            sb.append(object).append(",");
        if (!sb.isEmpty()) sb.setLength(sb.length() - 1);
        return sb.toString();
    }

    /**
     * Change a Object array to connected string by given seperateString
     */
    public static String arrayToString(Object[] array, String seperateString) {
        if (array == null) DialectException.throwEX("StrUtils arrayToString() method do not accept null parameter");
        StringBuilder sb = new StringBuilder();
        for (Object object : array)
            sb.append(object).append(seperateString);
        if (!sb.isEmpty()) sb.setLength(sb.length() - seperateString.length());
        return sb.toString();
    }

    /**
     * Change a Object List to "obj1,obj2...,objn" String
     */
    public static String listToString(List<?> lst) {
        if (lst == null) DialectException.throwEX("StrUtils listToString() method do not accept null parameter");
        StringBuilder sb = new StringBuilder();
        for (Object object : lst)
            sb.append(object).append(",");
        if (!sb.isEmpty()) sb.setLength(sb.length() - 1);
        return sb.toString();
    }

    /**
     * Join 2 String array into one
     */
    public static String[] joinStringArray(String[] array1, String[] array2) {
        List<String> l = new ArrayList<>();
        Collections.addAll(l, array1);
        Collections.addAll(l, array2);
        return l.toArray(new String[0]);
    }

    /**
     * Return true if first letter is Capitalised
     */
    public static boolean isCapitalizedString(String str) {
        char c = str.substring(0, 1).toCharArray()[0];
        return c >= 'A' && c <= 'Z';
    }

    /**
     * First letter change to lower
     */
    public static String toLowerCaseFirstOne(String s) {
        if (Character.isLowerCase(s.charAt(0))) return s;
        else return Character.toLowerCase(s.charAt(0)) + s.substring(1);
    }

    /**
     * First letter change to capitalised
     */
    public static String toUpperCaseFirstOne(String s) {
        if (Character.isUpperCase(s.charAt(0))) return s;
        else return Character.toUpperCase(s.charAt(0)) + s.substring(1);
    }


    /**
     * underScore String convert to camel format, for example: HELLO_WORLD->HelloWorld
     */
    public static String underScoreToCamel(String name) {
        StringBuilder result = new StringBuilder();
        if (name == null || name.isEmpty()) {
            return "";
        } else if (!name.contains("_")) // 不含下划线，仅将首字母小写
            return name.substring(0, 1).toLowerCase() + name.substring(1);
        String[] camels = name.split("_");
        for (String camel : camels) {
            if (camel.isEmpty()) // 跳过原始字符串中开头、结尾的下换线或双重下划线
                continue;
            if (result.isEmpty()) {// 处理真正的驼峰片段
                result.append(camel.toLowerCase());// 第一个驼峰片段，全部字母都小写
            } else {
                result.append(camel.substring(0, 1).toUpperCase());// 其他的驼峰片段，首字母大写
                result.append(camel.substring(1).toLowerCase());
            }
        }
        return result.toString();
    }

    /**
     * Camel String convert to lower case underScore, for example: HelloWorld -> hello_world
     */
    public static String camelToLowerCaseUnderScore(String name) {
        char[] chars = toLowerCaseFirstOne(name).toCharArray();
        StringBuilder sb = new StringBuilder();
        for (char aChar : chars) {
            if (Character.isUpperCase(aChar)) sb.append("_");
            sb.append(Character.toLowerCase(aChar));
        }
        return sb.toString();
    }


    /**
     * Check if a String only have a-z,A-Z,0-9,"_" characters
     */
    public static boolean isNormalLetters(char c) {
        return (c >= 'a' && c <= 'z') || (c >= '0' && c <= '9') || (c >= 'A' && c <= 'Z') || c == '_';
    }

    /**
     * Return true if is an invisible Char like space, tab, return... char
     */
    public static boolean isInvisibleChar(char c) {
        return c <= ' ';
    }

    /**
     * Format all " ", \t, \r... , to " ",
     */
    public static String formatSQL(String sql) {
        if (sql == null || sql.isEmpty()) return sql;
        StringBuilder sb = new StringBuilder();
        char[] chars = sql.toCharArray();
        boolean addedSpace = false;
        for (char c : chars) {
            if (isInvisibleChar(c)) {
                if (!addedSpace) {
                    sb.append(" ");
                    addedSpace = true;
                }
            } else {
                sb.append(c);
                addedSpace = false;
            }
        }
        sb.append(" ");
        return sb.toString();
    }

    /**
     * Some column has quote chars, like `someCol` or "someCol" or [someCol] use
     * this method to clear quote chars
     */
    public static String clearQuote(String columnName) {
        if (StrUtils.isEmpty(columnName)) return columnName;
        String s = StrUtils.replace(columnName, "`", "");
        s = StrUtils.replace(s, "\"", "");
        s = StrUtils.replace(s, "[", "");
        s = StrUtils.replace(s, "]", "");
        return s;
    }

    /**
     * Simple replace danderous chars in String to avoid SQL injection attack
     */
    public static String simpleReplaceDangerous(String str) {
        str = str.replace(";", "").replace("&", "&amp;").replace("<", "&lt;").replace(">", "&gt;").replace("'", "''").replace("--", "").replace("/", "").replace("%", "");
        return str;
    }

    /**
     * Build a ('xxx',xxx,xxx, ..., xxx) format String based on a object array,
     * usually used for an "in" condition SQL String, for example: <br/>
     * <p>
     * array("a","b",1,2) return String ('a','b',1,2)
     * array() return String (null)
     */
    public static String array(Object... arr) {
        if (arr == null || arr.length == 0) {
            return "(null)";
        }
        StringBuilder builder = new StringBuilder(200);
        builder.append("(");
        for (Object obj : arr) {
            if (obj == null) continue;
            Class<?> c = obj.getClass();
            if (String.class == c) {
                builder.append("'").append(simpleReplaceDangerous((String) obj)).append("',");
            } else {
                if (Integer.class == c || Long.class == c || Short.class == c || Byte.class == c)
                    builder.append(obj).append(",");
            }
        }
        if (builder.length() > 1) {
            builder.setLength(builder.length() - 1);
            return builder.append(")").toString();
        }
        return "(null)";
    }

    public static String sqlServerConvertTran(Dialect dialect, String... args) {
        // 判断是否是 转换时间的函数
        if (args.length == 3) {
            // 第一项 varchar 开头 varchar (100), 第二项 日期, 第三项 格式 21/23/120
            if (args[0].trim().toLowerCase().startsWith("varchar") && List.of("21", "23", "111", "120").contains(args[2].trim())) {
                if (dialect.isMySqlFamily()) {
                    return switch (args[2].trim()) {
                        case "21" -> "DATE_FORMAT(" + args[1] + ", '%Y-%m-%d %H:%i:%s.%f')";
                        case "23" -> "DATE_FORMAT(" + args[1] + ", '%Y-%m-%d')";
                        case "111" -> "DATE_FORMAT(" + args[1] + ", '%Y/%m/%d')";
                        // 120
                        default -> "DATE_FORMAT(" + args[1] + ", '%Y-%m-%d %H:%i:%s')";
                    };
                } else if (dialect.isOracleFamily()) {
                    return switch (args[2]) {
                        case "21" -> "TO_CHAR(" + args[1] + ", 'YYYY-MM-DD HH24:MI:SS.FF3')";
                        case "23" -> "TO_CHAR(" + args[1] + ", 'YYYY-MM-DD')";
                        case "111" -> "TO_CHAR(" + args[1] + ", 'YYYY/MM/DD')";
                        // 120
                        default -> "TO_CHAR(" + args[1] + ", 'YYYY-MM-DD HH24:MI:SS')";
                    };
                }
            }
        } else if (args.length == 2) {
            //  convert(numeric(19, 2), a )
            return sqlServerCastTran(dialect, "$args[1].trim()$ AS " + args[0].trim()).replace("$args[1].trim()$", args[1].trim());
        }
        return "CONVERT(" + String.join(", ", args) + ")";
    }

    //    sqlserver
    //    cast(filed as NUMERIC)
    //    cast(filed as varchar)
    //    cast(filed as numeric(19,2))
    //    cast(filed as decimal(19,2))
    //      mysql
    //    CAST(field AS DECIMAL)
    //    CAST(field AS CHAR)
    //    CAST(field AS DECIMAL(19,2))
    //    CAST(field AS DECIMAL(19,2))
    //      oracle
    //    CAST(field AS NUMBER)
    //    CAST(field AS VARCHAR2(255))
    //    CAST(field AS NUMBER(19,2))
    //    CAST(field AS NUMBER(19,2))
    public static String sqlServerCastTran(Dialect dialect, String... args) {
        if (args.length == 1) {
            String trimStr = args[0].trim().toLowerCase();
            // 如果三层说明是这种简单的cast使用 field AS NUMBER
            if (dialect.isMySqlFamily()) {
                if (trimStr.contains("numeric")) {
                    return "CAST(" + trimStr.replace("numeric", "decimal") + ")";
                }
                if (trimStr.contains("varchar")) {
                    return "CAST(" + trimStr.replace("varchar", "char") + ")";
                }
            } else if (dialect.isOracleFamily()) {
                if (dialect.isMySqlFamily()) {
                    if (trimStr.contains("numeric")) {
                        return "CAST(" + trimStr.replace("numeric", "number") + ")";
                    }
                    if (trimStr.contains("varchar")) {
                        return "CAST(" + trimStr.replace("varchar", "varchar2") + ")";
                    }
                }
                return "CAST(" + trimStr + ")";
            }
        }
        return "cast(" + String.join(", ", args) + ")";
    }

    public static String sqlServerDATEDIFFTran(Dialect dialect, String... args) {
        if (args.length == 3) {
            String trimStr = args[0].trim().toLowerCase();
            // 如果三层说明是这种简单的cast使用 field AS NUMBER
            if (dialect.isMySqlFamily()) {
                return "TIMESTAMPDIFF(" + dialect.typeConvertMapping.getOrDefault(trimStr, trimStr) + ", " + args[1] + ", " + args[2] + ")";
            }
            // PostgreSQL的DATEDIFF转换
            if (dialect.isPostgresFamily()) {
                return "(" + args[2] + "::date - " + args[1] + "::date)";
            }
        }
        return "datediff(" + String.join(", ", args) + ")";
    }

    /**
     * Handle WITH RECURSIVE SQL compatibility across different databases
     *
     * @param dialect target database dialect
     * @param args    SQL arguments containing WITH clause
     * @return processed WITH clause or empty string
     */
    public static String handleWithRecursive(Dialect dialect, String... args) {
        if (args == null || args.length == 0) {
            return "";
        }

        String sql = String.join(" ", args);

        // Check if this is a recursive CTE (contains UNION ALL and self-reference pattern)
        boolean isRecursive = isRecursiveCTE(sql);

        if (dialect.isPostgresFamily()) {
            // PostgreSQL requires RECURSIVE keyword for recursive CTEs
            if (isRecursive && !containsIgnoreCase(sql, "RECURSIVE")) {
                // Add RECURSIVE keyword after WITH
                return replaceIgnoreCase(sql, "WITH", "WITH RECURSIVE");
            }
            // Keep existing RECURSIVE if present
        } else {
            // Other databases: remove RECURSIVE keyword as they don't support it in older versions
            if (containsIgnoreCase(sql, "RECURSIVE")) {
                return replaceIgnoreCase(sql, "WITH RECURSIVE", "WITH");
            }
        }
        return sql;
    }

    /**
     * Detect if a SQL contains recursive CTE pattern
     * Basic detection: looks for UNION ALL and self-reference patterns
     */
    private static boolean isRecursiveCTE(String sql) {
        if (sql == null || sql.isEmpty()) {
            return false;
        }

        String sqlLower = sql.toLowerCase();

        // Must contain UNION ALL for recursion
        if (!sqlLower.contains("union all")) {
            return false;
        }

        // Try to extract CTE name(s) and check for self-reference
        // Simple pattern: WITH [RECURSIVE] name AS (...UNION ALL... SELECT ... FROM name ...)
        String[] parts = sqlLower.split("\\s+");
        String cteName = null;

        for (int i = 0; i < parts.length - 2; i++) {
            if ("with".equals(parts[i]) || "recursive".equals(parts[i])) {
                // Next non-keyword token should be CTE name
                int nameIndex = i + 1;
                if ("recursive".equals(parts[i]) && i + 1 < parts.length && !"as".equals(parts[i + 1])) {
                    nameIndex = i + 1; // WITH RECURSIVE name
                }

                if (nameIndex < parts.length && !"as".equals(parts[nameIndex])) {
                    cteName = parts[nameIndex];
                    break;
                }
            }
        }

        // Check if CTE name is referenced after UNION ALL
        if (cteName != null) {
            int unionIndex = sqlLower.indexOf("union all");
            if (unionIndex > 0) {
                String afterUnion = sqlLower.substring(unionIndex);
                return afterUnion.contains(" " + cteName + " ") ||
                        afterUnion.contains(" " + cteName + ",") ||
                        afterUnion.contains("from " + cteName) ||
                        afterUnion.contains("join " + cteName);
            }
        }

        return false;
    }
}
