package com.af.v4.system.common.datasource.dialects.model;

import com.af.v4.system.common.datasource.dialects.DebugUtils;
import com.af.v4.system.common.datasource.dialects.DialectException;
import com.af.v4.system.common.datasource.dialects.StrUtils;
import com.af.v4.system.common.datasource.dialects.enums.GenerationType;
import com.af.v4.system.common.datasource.dialects.id.*;

import java.util.ArrayList;
import java.util.List;

/**
 * A TableModel definition represents a platform dependent Database Table, from
 * 1.0.5 this class name changed from "Table" to "TableModel" to avoid naming
 * conflict to JPA's "@Table" annotation
 *
 * @since 1.0.2
 */
public class TableModel {
    /**
     * The table tableName in database
     */
    private String tableName;

    /**
     * check constraint for table
     */
    private String check;

    /**
     * comment for table
     */
    private String comment;

    /**
     * Optional, If support engine like MySQL or MariaDB, add engineTail at the end
     * of "create table... engine=xxx" DDL, usually used to set encode String like " DEFAULT
     * CHARSET=utf8" for MySQL
     * engineTail only effect when dialect support engine
     */
    private String engineTail;

    /**
     * if set not null, tableTail will override engine and engineTail
     * and if not null, tableTail always be added at end of DDL no matter if dialect support engine
     * Tips: if set tableTail="" can disable engine and engineTail
     */
    private String tableTail = null;

    /**
     * Columns in this table
     */
    private List<ColumnModel> columns = new ArrayList<>();

    /**
     * IdGenerators
     */
    private List<IdGenerator> idGenerators;

    /**
     * Foreign Keys
     */
    private List<FKeyModel> fkeyConstraints;

    /**
     * Indexes
     */
    private List<IndexModel> indexConsts;

    /**
     * Unique constraints
     */
    private List<UniqueModel> uniqueConsts;

    /**
     * Map to which entityClass, this field is designed to ORM tool to use
     */
    private Class<?> entityClass;

    private Boolean readOnly = false;

    public TableModel() {
        super();
    }

    public TableModel(String tableName) {
        this.tableName = tableName;
    }

    public static void sortColumns(List<ColumnModel> lst) {
        if (lst == null || lst.isEmpty())
            return;
        lst.sort((a, b) -> {
            if (a == null || b == null || a.getColumnName() == null)
                return -1;
            return a.getColumnName().compareTo(b.getColumnName());
        });
    }

    /**
     * Get one of these IdGenerator instance by generationType:
     * IDENTITY,AUTO,UUID25,UUID32,UUID36,TIMESTAMP, if not found , return null;
     */
    private static IdGenerator getIdGeneratorByType(TableModel model, GenerationType generationType) {
        if (generationType == null)
            return null;
        return switch (generationType) {
            case IDENTITY -> {
                for (IdGenerator idGen : model.getIdGenerators())
                    if (generationType.equals(idGen.getGenerationType()))
                        yield idGen;
                yield null;
            }
            case AUTO -> AutoIdGenerator.INSTANCE;
            case UUID -> UUIDGenerator.INSTANCE;
            case UUID25 -> UUID25Generator.INSTANCE;
            case UUID26 -> UUID26Generator.INSTANCE;
            case UUID32 -> UUID32Generator.INSTANCE;
            case UUID36 -> UUID36Generator.INSTANCE;
            case TIMESTAMP -> TimeStampIdGenerator.INSTANCE;
            case SNOWFLAKE -> SnowflakeGenerator.INSTANCE;
            default -> null; //identity and sequence related to tableModel, by type search can only return null
        };
    }

    /**
     * Get a IdGenerator by type, if not found, search by name
     */
    private static IdGenerator getIdGenerator(TableModel model, GenerationType generationType, String name,
                                              List<IdGenerator> idGeneratorList) {
        // fixed idGenerators
        IdGenerator idGen = getIdGeneratorByType(model, generationType);
        if (idGen != null)
            return idGen;
        if (StrUtils.isEmpty(name))
            return null;
        for (IdGenerator idGenerator : idGeneratorList) {
            if (generationType != null && name.equalsIgnoreCase(idGenerator.getIdGenName()))
                return idGenerator;
            if ((generationType == null || GenerationType.OTHER.equals(generationType))
                    && name.equalsIgnoreCase(idGenerator.getIdGenName()))
                return idGenerator;
        }
        return null;
    }

    /**
     * @return an editable copy of current TableModel
     */
    public TableModel newCopy() {
        TableModel tb = new TableModel();
        tb.tableName = this.tableName;
        tb.check = this.check;
        tb.comment = this.comment;
        tb.engineTail = this.engineTail;
        tb.tableTail = this.tableTail;
        tb.entityClass = this.entityClass;
        if (!columns.isEmpty())
            for (ColumnModel item : columns) {
                ColumnModel newItem = item.newCopy();
                newItem.setTableModel(tb);
                tb.columns.add(newItem);
            }

        if (idGenerators != null && !idGenerators.isEmpty())
            for (IdGenerator item : idGenerators)
                tb.getIdGenerators().add(item.newCopy());

        if (fkeyConstraints != null && !fkeyConstraints.isEmpty())
            for (FKeyModel item : fkeyConstraints)
                tb.getFkeyConstraints().add(item.newCopy());

        if (indexConsts != null && !indexConsts.isEmpty())
            for (IndexModel item : indexConsts)
                tb.getIndexConsts().add(item.newCopy());

        if (uniqueConsts != null && !uniqueConsts.isEmpty())
            for (UniqueModel item : uniqueConsts)
                tb.getUniqueConsts().add(item.newCopy());
        return tb;
    }

    /**
     * Add a TableGenerator
     */
    public void tableGenerator(String name, String tableName, String pkColumnName, String valueColumnName,
                               String pkColumnValue, Integer initialValue, Integer allocationSize) {
        checkReadOnly();
        addGenerator(new TableIdGenerator(name, tableName, pkColumnName, valueColumnName, pkColumnValue, initialValue,
                allocationSize));
    }

    /**
     * Add a UUIDAnyGenerator
     */
    public void uuidAny(String name, Integer length) {
        checkReadOnly();
        addGenerator(new UUIDAnyGenerator(name, length));
    }

    /**
     * Add a "create table..." DDL to generate ID, similar like JPA's TableGen
     */
    public void addGenerator(IdGenerator generator) {
        checkReadOnly();
        DialectException.assureNotNull(generator);
        DialectException.assureNotNull(generator.getGenerationType());
        DialectException.assureNotEmpty(generator.getIdGenName(), "IdGenerator name can not be empty");
        getIdGenerators().add(generator);
    }

    /**
     * Add a sequence definition DDL, note: some dialects do not support sequence
     *
     * @param name           The name of sequence Java object itself
     * @param sequenceName   the name of the sequence will created in database
     * @param initialValue   The initial value
     * @param allocationSize The allocationSize
     */
    public void sequenceGenerator(String name, String sequenceName, Integer initialValue, Integer allocationSize) {
        checkReadOnly();
        this.addGenerator(new SequenceIdGenerator(name, sequenceName, initialValue, allocationSize));
    }

    public void identityGenerator(String column) {
        checkReadOnly();
        DialectException.assureNotEmpty(tableName, "IdGenerator tableName can not be empty");
        DialectException.assureNotEmpty(column, "IdGenerator column can not be empty");
        this.addGenerator(new IdentityIdGenerator(tableName, column));
    }

    /**
     * Add a Sequence Generator, note: not all database support sequence
     */
    public void sortedUUIDGenerator(String name, Integer sortedLength, Integer uuidLength) {
        checkReadOnly();
        DialectException.assureNotNull(name);
        if (this.getIdGenerator(GenerationType.SORTED_UUID, name) != null)
            throw new DialectException(STR."Duplicated sortedUUIDGenerator name '\{name}'");
        idGenerators.add(new SortedUUIDGenerator(name, sortedLength, uuidLength));
    }

    /**
     * Add the table check, note: not all database support table check
     */
    public TableModel check(String check) {
        checkReadOnly();
        this.check = check;
        return this;
    }

    /**
     * Add the table comment, note: not all database support table comment
     */
    public TableModel comment(String comment) {
        checkReadOnly();
        this.comment = comment;
        return this;
    }

    /**
     * Add a ColumnModel
     */
    public TableModel addColumn(ColumnModel column) {
        checkReadOnly();
        DialectException.assureNotNull(column);
        DialectException.assureNotEmpty(column.getColumnName(), "Column's columnName can not be empty");
        column.setTableModel(this);
        columns.add(column);
        return this;
    }

    /**
     * Remove a ColumnModel by given columnName
     */
    public TableModel removeColumn(String columnName) {
        checkReadOnly();
        List<ColumnModel> oldColumns = this.getColumns();
        oldColumns.removeIf(columnModel -> columnModel.getColumnName().equalsIgnoreCase(columnName));
        return this;
    }

    /**
     * Remove a FKey by given fkeyName
     */
    public TableModel removeFKey(String fkeyName) {
        checkReadOnly();
        List<FKeyModel> fkeys = getFkeyConstraints();
        fkeys.removeIf(fKeyModel -> fKeyModel.getFkeyName().equalsIgnoreCase(fkeyName));
        return this;
    }

    /**
     * find column in tableModel by given columnName, if not found, add a new column
     * with columnName
     */
    public ColumnModel column(String columnName) {
        for (ColumnModel columnModel : columns) {
            if (columnModel.getColumnName() != null && columnModel.getColumnName().equalsIgnoreCase(columnName))
                return columnModel;
            if (columnModel.getEntityField() != null && columnModel.getEntityField().equalsIgnoreCase(columnName))
                return columnModel;
        }
        return addColumn(columnName);
    }

    /**
     * Add a column with given columnName to tableModel
     *
     * @param columnName columnName
     * @return the Column object
     */
    public ColumnModel addColumn(String columnName) {
        checkReadOnly();
        DialectException.assureNotEmpty(columnName, "columnName can not be empty");
        for (ColumnModel columnModel : columns)
            if (columnName.equalsIgnoreCase(columnModel.getColumnName()))
                throw new DialectException(STR."ColumnModel name '\{columnName}' already existed");
        ColumnModel column = new ColumnModel(columnName);
        addColumn(column);
        return column;
    }

    /**
     * Get ColumnModel by column Name or field name ignore case, if not found,
     * return null
     */
    public ColumnModel getColumn(String colOrFieldName) {
        for (ColumnModel columnModel : columns) {
            if (columnModel.getColumnName() != null && columnModel.getColumnName().equalsIgnoreCase(colOrFieldName))
                return columnModel;
            if (columnModel.getEntityField() != null && columnModel.getEntityField().equalsIgnoreCase(colOrFieldName))
                return columnModel;
        }
        return null;
    }

    /**
     * Get ColumnModel by columnName ignore case, if not found, return null
     */
    public ColumnModel getColumnByColName(String colName) {
        for (ColumnModel columnModel : columns) {
            if (columnModel.getColumnName() != null && (columnModel.getColumnName().equalsIgnoreCase(colName)
                    || columnModel.getColumnName().equalsIgnoreCase(STR."`\{colName}`")
                    || columnModel.getColumnName().equalsIgnoreCase(STR."\"\{colName}\"")
                    || columnModel.getColumnName().equalsIgnoreCase(STR."[\{colName}]")
            )
            )
                return columnModel;
        }
        return null;
    }

    /**
     * Get ColumnModel by entity field name ignore case, if not found, return null
     */
    public ColumnModel getColumnByFieldName(String fieldName) {
        for (ColumnModel columnModel : columns)
            if (columnModel.getEntityField() != null && columnModel.getEntityField().equalsIgnoreCase(fieldName))
                return columnModel;
        return null;
    }

    /**
     * @return First found ShardTable Column , if not found , return null
     */
    public ColumnModel getShardTableColumn() {
        for (ColumnModel columnModel : columns)
            if (columnModel.getShardTable() != null)
                return columnModel;// return first found only
        return null;
    }

    /**
     * @return First found ShardDatabase Column , if not found , return null
     */
    public ColumnModel getShardDatabaseColumn() {
        for (ColumnModel columnModel : columns)
            if (columnModel.getShardDatabase() != null)
                return columnModel;// return first found only
        return null;
    }

    /**
     * Start add a foreign key definition in DDL, detail usage see demo
     */
    public FKeyModel fkey() {
        checkReadOnly();
        FKeyModel fkey = new FKeyModel();
        fkey.setTableName(this.tableName);
        fkey.setTableModel(this);
        getFkeyConstraints().add(fkey);
        return fkey;
    }

    /**
     * Start add a foreign key definition in DDL, detail usage see demo
     */
    public FKeyModel fkey(String fkeyName) {
        checkReadOnly();
        FKeyModel fkey = new FKeyModel();
        fkey.setTableName(this.tableName);
        fkey.setFkeyName(fkeyName);
        fkey.setTableModel(this);
        getFkeyConstraints().add(fkey);
        return fkey;
    }

    /**
     * Get a FKeyModel by given fkeyName
     */
    public FKeyModel getFkey(String fkeyName) {
        if (fkeyConstraints == null)
            return null;
        for (FKeyModel fkey : fkeyConstraints)
            if (!StrUtils.isEmpty(fkeyName) && fkeyName.equalsIgnoreCase(fkey.getFkeyName()))
                return fkey;
        return null;
    }

    /**
     * Start add a Index in DDL, detail usage see demo
     */
    public IndexModel index() {
        checkReadOnly();
        IndexModel index = new IndexModel();
        index.setTableModel(this);
        getIndexConsts().add(index);
        return index;
    }

    /**
     * Start add a Index in DDL, detail usage see demo
     */
    public IndexModel index(String indexName) {
        checkReadOnly();
        IndexModel index = new IndexModel();
        index.setName(indexName);
        index.setTableModel(this);
        getIndexConsts().add(index);
        return index;
    }

    /**
     * Start add a unique constraint in DDL, detail usage see demo
     */
    public UniqueModel unique() {
        checkReadOnly();
        UniqueModel unique = new UniqueModel();
        unique.setTableModel(this);
        getUniqueConsts().add(unique);
        return unique;
    }

    /**
     * Start add a unique constraint in DDL, detail usage see demo
     */
    public UniqueModel unique(String uniqueName) {
        checkReadOnly();
        UniqueModel unique = new UniqueModel();
        unique.setName(uniqueName);
        unique.setTableModel(this);
        getUniqueConsts().add(unique);
        return unique;
    }

    /**
     * If support engine like MySQL or MariaDB, add engineTail at the end of "create
     * table..." DDL, usually used to set encode String like " DEFAULT CHARSET=utf8"
     * for MySQL
     */
    public TableModel engineTail(String engineTail) {
        checkReadOnly();
        this.engineTail = engineTail;
        return this;
    }

    /**
     * if not null, no matter if support engine, use this tableTail override engine and engineTail
     */
    public TableModel tableTail(String tableTail) {
        checkReadOnly();
        this.tableTail = tableTail;
        return this;
    }

    /**
     * Search and return the IdGenerator in this TableModel by its generationType
     * and name
     */
    public IdGenerator getIdGenerator(GenerationType generationType, String name) {
        return getIdGenerator(this, generationType, name, getIdGenerators());
    }

    /**
     * Get one of these IdGenerator instance by generationType:
     * IDENTITY,AUTO, UUID25,UUID26,UUID,UUID32,UUID36,TIMESTAMP
     */
    public IdGenerator getIdGenerator(GenerationType generationType) {
        return getIdGeneratorByType(this, generationType);
    }

    /**
     * Search and return the IdGenerator in this TableModel by its name
     */
    public IdGenerator getIdGenerator(String name) {
        return getIdGenerator(this, null, name, getIdGenerators());
    }

    public int getPKeyCount() {
        int pkeyCount = 0;
        for (ColumnModel col : columns)
            if (col.getPkey() && !col.getTransientable())
                pkeyCount++;
        return pkeyCount;
    }

    public ColumnModel getFirstPKeyColumn() {
        for (ColumnModel col : columns)
            if (col.getPkey() && !col.getTransientable())
                return col;
        return null;
    }

    /**
     * Get pkey columns sorted by column name
     */
    public List<ColumnModel> getPKeyColumns() {
        List<ColumnModel> pkeyCols = new ArrayList<>();
        for (ColumnModel col : columns)
            if (col.getPkey() && !col.getTransientable())
                pkeyCols.add(col);
        return pkeyCols;
    }

    public String getDebugInfo() {
        return DebugUtils.getTableModelDebugInfo(this);
    }

    private void checkReadOnly() {
        if (readOnly)
            throw new DialectException(STR."TableModel '\{tableName}' is readOnly, can not be modified.");
    }

    public String getTableName() {
        return tableName;
    }

    public void setTableName(String tableName) {
        checkReadOnly();
        this.tableName = tableName;
    }

    public String getCheck() {
        return check;
    }

    public void setCheck(String check) {
        checkReadOnly();
        this.check = check;
    }

    public String getComment() {
        return comment;
    }

    public void setComment(String comment) {
        checkReadOnly();
        this.comment = comment;
    }

    public List<ColumnModel> getColumns() {
        return columns;
    }

    public void setColumns(List<ColumnModel> columns) {
        checkReadOnly();
        this.columns = columns;
    }

    public List<FKeyModel> getFkeyConstraints() {
        if (fkeyConstraints == null)
            fkeyConstraints = new ArrayList<>();
        return fkeyConstraints;
    }

    public void setFkeyConstraints(List<FKeyModel> fkeyConstraints) {
        checkReadOnly();
        this.fkeyConstraints = fkeyConstraints;
    }

    public String getEngineTail() {
        return engineTail;
    }

    public void setEngineTail(String engineTail) {
        checkReadOnly();
        this.engineTail = engineTail;
    }

    public String getTableTail() {
        return tableTail;
    }

    public void setTableTail(String tableTail) {
        checkReadOnly();
        this.tableTail = tableTail;
    }

    public List<IndexModel> getIndexConsts() {
        if (indexConsts == null)
            indexConsts = new ArrayList<>();
        return indexConsts;
    }

    public void setIndexConsts(List<IndexModel> indexConsts) {
        checkReadOnly();
        this.indexConsts = indexConsts;
    }

    public List<UniqueModel> getUniqueConsts() {
        if (uniqueConsts == null)
            uniqueConsts = new ArrayList<>();
        return uniqueConsts;
    }

    public void setUniqueConsts(List<UniqueModel> uniqueConsts) {
        checkReadOnly();
        this.uniqueConsts = uniqueConsts;
    }

    public List<IdGenerator> getIdGenerators() {
        if (idGenerators == null)
            idGenerators = new ArrayList<>();
        return idGenerators;
    }

    public void setIdGenerators(List<IdGenerator> idGenerators) {
        checkReadOnly();
        this.idGenerators = idGenerators;
    }

    public Class<?> getEntityClass() {
        return entityClass;
    }

    public void setEntityClass(Class<?> entityClass) {
        checkReadOnly();
        this.entityClass = entityClass;
    }

    public Boolean getReadOnly() {
        return readOnly;
    }

    public void setReadOnly(Boolean readOnly) {
        this.readOnly = readOnly;
    }

}
