package com.af.v4.system.common.datasource.dialects;


import javax.sql.DataSource;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.SQLException;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@SuppressWarnings("all")
public class GuessDialectUtils {
    private static final Map<DataSource, Dialect> dataSourceDialectCache = new ConcurrentHashMap<DataSource, Dialect>();

    /**
     * Guess dialect based on given JDBC connection instance, Note: this method does
     * not close connection
     *
     * @param jdbcConnection The connection
     * @return dialect or null if can not guess out which dialect
     */
    public static Dialect guessDialect(Connection jdbcConnection) {
        String databaseName;
        String driverName;
        int majorVersion;
        int minorVersion;
        try {
            DatabaseMetaData meta = jdbcConnection.getMetaData();
            driverName = meta.getDriverName();
            databaseName = meta.getDatabaseProductName();
            majorVersion = meta.getDatabaseMajorVersion();
            minorVersion = meta.getDatabaseMinorVersion();
        } catch (SQLException e) {
            return (Dialect) DialectException.throwEX(e);
        }
        return guessDialect(driverName, databaseName, majorVersion, minorVersion);
    }

    /**
     * Guess dialect based on given dataSource
     *
     * @param datasource The dataSource
     * @return dialect or null if can not guess out which dialect
     */
    public static Dialect guessDialect(DataSource dataSource) {
        Dialect result = dataSourceDialectCache.get(dataSource);
        if (result != null)
            return result;
        Connection con = null;
        try {
            con = dataSource.getConnection();
            result = guessDialect(con);
            if (result == null)
                return (Dialect) DialectException
                        .throwEX("Can not get dialect from DataSource, please submit this bug.");
            dataSourceDialectCache.put(dataSource, result);
            return result;
        } catch (SQLException e) {
            return (Dialect) DialectException.throwEX(e);
        } finally {
            try {
                if (con != null && !con.isClosed()) {
                    try {
                        con.close();
                    } catch (SQLException e) {
                        DialectException.throwEX(e);
                    }
                }
            } catch (SQLException e) {
                DialectException.throwEX(e);
            }
        }
    }

    public static Dialect guessDialect(String driverName) {
        return guessDialect(driverName, null, 0, 0);
    }

    /**
     * @param databaseName             The database name
     * @param majorVersionMinorVersion The major version,The minor version, Optional optional
     * @return dialect or null if not found
     */
    public static Dialect guessDialect(String driverName, String databaseName, Object... majorVersionMinorVersion) {
        int majorVersion = 0;
        int minorVersion = 0;
        for (int i = 0; i < majorVersionMinorVersion.length; i++) {
            if (i == 0)
                majorVersion = (Integer) majorVersionMinorVersion[i];
            if (i == 1)
                minorVersion = (Integer) majorVersionMinorVersion[i];
        }
        if ("MySQL".equals(databaseName)) {
            if (majorVersion < 5)
                return Dialect.MySQLDialect;
            else if (majorVersion == 5) {
                if (minorVersion < 5)
                    return Dialect.MySQL5Dialect;
                else if (minorVersion < 7)
                    return Dialect.MySQL55Dialect;
                else
                    return Dialect.MySQL57Dialect;
            } else if (majorVersion == 8) {
                return Dialect.MySQL8Dialect;
            }
            return Dialect.MySQLDialect;
        }
        if ("PostgreSQL".equals(databaseName)) {
            if (majorVersion == 9) {
                if (minorVersion >= 4) {
                    return Dialect.PostgreSQL94Dialect;
                } else if (minorVersion >= 2) {
                    return Dialect.PostgreSQL92Dialect;
                }
                return Dialect.PostgreSQL9Dialect;
            }
            if (majorVersion == 8 && minorVersion >= 2) {
                return Dialect.PostgreSQL82Dialect;
            }
            return Dialect.PostgreSQL81Dialect;
        }
        if (databaseName.startsWith("Microsoft SQL Server")) {
            switch (majorVersion) {
                case 8:
                    return Dialect.SQLServerDialect;
                case 9:
                    return Dialect.SQLServer2005Dialect;
                case 10:
                    return Dialect.SQLServer2008Dialect;
                case 11:
                case 12:
                case 13:
                    return Dialect.SQLServer2012Dialect;
                default:
                    if (majorVersion < 8)
                        return Dialect.SQLServerDialect;
                    else
                        return Dialect.SQLServer2012Dialect;
            }
        }
        if (StrUtils.containsIgnoreCase(databaseName, "Oracle")) {
            switch (majorVersion) {
                case 12:
                    return Dialect.Oracle12cDialect;
                case 11:
                case 10:
                    return Dialect.Oracle10gDialect;
                case 9:
                    return Dialect.Oracle9iDialect;
                case 8:
                    return Dialect.Oracle8iDialect;
                default:
            }
            return Dialect.Oracle12cDialect;
        }
        if (StrUtils.containsIgnoreCase(databaseName, "sqlite"))
            return Dialect.SQLiteDialect;

        if (StrUtils.startsWithIgnoreCase(databaseName, "DM"))
            return Dialect.DamengDialect;
        return null;
    }
}

