package com.af.v4.system.common.datasource.dialects;

import java.util.regex.Pattern;

/**
 * WITH RECURSIVE语法处理器
 * 处理不同数据库对递归CTE的语法差异
 *
 * @author framework
 * @since 2.12.55
 */
public class WithRecursiveProcessor implements SqlSyntaxProcessor {

    // 匹配真正的WITH CTE语句的正则表达式
    // 支持各种常见的人工编写格式：
    // - WITH CTE AS (
    // - WITH RECURSIVE CTE AS (
    // - WITH\n  CTE AS (
    // - WITH/*comment*/CTE AS (
    // - WITH    RECURSIVE    CTE    AS    (
    // 支持SQL开头就是WITH的情况
    private static final Pattern WITH_CTE_PATTERN =
            Pattern.compile("(?:^|\\s|\\n)WITH\\s*(?:/\\*.*?\\*/\\s*)?(?:RECURSIVE\\s*(?:/\\*.*?\\*/\\s*)?)?\\s*\\w+\\s*(?:/\\*.*?\\*/\\s*)?AS\\s*\\(",
                    Pattern.CASE_INSENSITIVE | Pattern.DOTALL);

    // 匹配UNION ALL的模式，用于检测递归CTE
    private static final Pattern UNION_ALL_PATTERN =
            Pattern.compile("UNION\\s+ALL", Pattern.CASE_INSENSITIVE);

    @Override
    public String process(Dialect dialect, String sql) {
        if (!canHandle(sql)) {
            return sql;
        }

        // 检查是否是递归CTE（包含UNION ALL和自引用模式）
        boolean isRecursive = isRecursiveCTE(sql);
        boolean hasRecursive = StrUtils.containsIgnoreCase(sql, "RECURSIVE");

        if (dialect.isPostgresFamily() || dialect.isMySqlFamily()) {
            // PostgreSQL Mysql 需要RECURSIVE关键字用于递归CTE
            if (isRecursive && !hasRecursive) {
                // 在WITH后添加RECURSIVE关键字，支持各种格式
                // 匹配: WITH [空格/注释] CTE_NAME，确保RECURSIVE前后有空格
                // 支持SQL开头就是WITH的情况
                return sql.replaceAll("(?i)((?:^|\\s|\\n)WITH)(\\s*(?:/\\*.*?\\*/\\s*)?)(\\w+)",
                        "$1 RECURSIVE $3");
            }
            // 如果已有RECURSIVE则保持不变
            return sql;
        } else {
            // 其他数据库：移除RECURSIVE关键字（老版本不支持）
            if (hasRecursive) {
                // 移除RECURSIVE关键字，支持各种格式
                // 匹配: WITH [空格/注释] RECURSIVE [空格/注释]
                // 支持SQL开头就是WITH的情况
                return sql.replaceAll("(?i)((?:^|\\s|\\n)WITH)(\\s*(?:/\\*.*?\\*/\\s*)?)RECURSIVE(\\s*(?:/\\*.*?\\*/\\s*)?)",
                        "$1$2$3");
            }
            return sql;
        }
    }

    @Override
    public boolean canHandle(String sql) {
        if (sql == null || sql.trim().isEmpty()) {
            return false;
        }

        // 使用正则表达式精确匹配WITH CTE语句
        // 只处理真正的WITH CTE，不处理table hint等其他with用法
        boolean hasWithCte = WITH_CTE_PATTERN.matcher(sql).find();
        if (hasWithCte) {
            // 额外检查：确保不是table hint (FROM table with(...))
            Pattern tableHintPattern = Pattern.compile("FROM\\s+\\w+\\s+with\\s*\\(", Pattern.CASE_INSENSITIVE);
            boolean hasTableHint = tableHintPattern.matcher(sql).find();
            return !hasTableHint;
        }

        return false;
    }

    @Override
    public int getPriority() {
        // 高优先级，因为WITH语句通常在SQL开头，需要早期处理
        return 10;
    }

    @Override
    public String getName() {
        return "WithRecursiveProcessor";
    }

    /**
     * 检测SQL是否包含递归CTE模式
     * 基本检测：寻找UNION ALL和自引用模式
     */
    private boolean isRecursiveCTE(String sql) {
        if (sql == null || sql.isEmpty()) {
            return false;
        }

        String sqlLower = sql.toLowerCase();

        // 必须包含UNION ALL才可能是递归
        if (!UNION_ALL_PATTERN.matcher(sql).find()) {
            return false;
        }

        // 尝试提取CTE名称并检查自引用
        // 简单模式：WITH [RECURSIVE] name AS (...UNION ALL... SELECT ... FROM name ...)
        String[] parts = sqlLower.split("\\s+");
        String cteName = null;

        for (int i = 0; i < parts.length - 2; i++) {
            if ("with".equals(parts[i])) {
                int nameIndex = i + 1;
                if (nameIndex < parts.length && "recursive".equals(parts[nameIndex])) {
                    nameIndex = i + 2; // WITH RECURSIVE name
                }
                if (nameIndex < parts.length && !"as".equals(parts[nameIndex])) {
                    cteName = parts[nameIndex];
                    break;
                }
            }
        }

        // 检查CTE名称是否在UNION ALL之后被引用
        if (cteName != null) {
            int unionIndex = sqlLower.indexOf("union all");
            if (unionIndex > 0) {
                String afterUnion = sqlLower.substring(unionIndex);
                return afterUnion.contains(STR." \{cteName} ") ||
                        afterUnion.contains(STR." \{cteName},") ||
                        afterUnion.contains(STR."from \{cteName}") ||
                        afterUnion.contains(STR."join \{cteName}");
            }
        }

        return false;
    }
}
