package com.af.v4.system.common.datasource.dialects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Dialect function Utils, detail see render method
 *
 * @since 1.0.2
 */
public class DialectFunctionUtils {
    private static final Logger logger = LoggerFactory.getLogger(DialectFunctionUtils.class);

    /**
     * The render method translate function template to real SQL piece
     *
     * <pre>
     * Template can be:
     * "*": standard SQL function, identical to abc($Params)
     * "abc($Params)": template with special parameter format:
     * "$P1, $P2, $P3, $P4, $P5, $P6..."="$Params"
     * "$P1,$P2,$P3,$P4,$P5,$P6..."="$Compact_Params"
     * "$P1||$P2||$P3||$P4||$P5||$P6..."="$Lined_Params"
     * "$P1+$P2+$P3+$P4+$P5+$P6..."="$Add_Params");
     * "$P1 in $P2 in $P3 in $P4 in $P5 in $P6..."="$IN_Params"
     * "$P1%pattern$P2%pattern$P3%pattern$P4%pattern$P5%pattern$P6..."="$Pattern_Params"
     * "11%startswith$P2%startswith$P3%startswith$P4%startswith$P5%startswith$P6..."= "$Startswith_Params");
     * "nvl($P1, nvl($P2, nvl($P3, nvl($P4, nvl($P5, $P6...)))))"="$NVL_Params");
     *
     * "0=abc()": function do not support parameter
     * "1=abc($P1)": function only support 1 parameter
     * "2=abc($P1,$P2)": function only support 2 parameters
     * "0=abc()|1=abc($P1)|3=abc($P1,$P2,$P3)": function support 0 or 1 or 3 parameters
     *
     * </pre>
     *
     * @param functionName function name
     * @param args         function parameters
     * @return A SQL function piece
     */
    protected static String render(Dialect d, String functionName, String... args) {
        String template = d.functions.get(functionName.toLowerCase());
        if (template == null || template.isEmpty()) {
            template = "*";
        }
        if ("*".equals(template))
            template = STR."\{functionName}($Params)";
        char c = template.charAt(1);
        if (c != '=') {
            if (template.contains("$Params")) {
                return StrUtils.replace(template, "$Params", StrUtils.arrayToString(args, ", "));
            }
            if (template.contains("$Compact_Params")) {
                return StrUtils.replace(template, "$Compact_Params", StrUtils.arrayToString(args, ","));
            }
            if (template.contains("$Lined_Params")) {
                return StrUtils.replace(template, "$Lined_Params", StrUtils.arrayToString(args, "||"));
            }
            if (template.contains("$Add_Params")) {
                return StrUtils.replace(template, "$Add_Params", StrUtils.arrayToString(args, "+"));
            }
            if (template.contains("$IN_Params")) {
                return StrUtils.replace(template, "$IN_Params", StrUtils.arrayToString(args, " in "));
            }
            if (template.contains("$Pattern_Params")) {
                return StrUtils.replace(template, "$Pattern_Params", StrUtils.arrayToString(args, "%pattern"));
            }
            if (template.contains("$Startswith_Params")) {
                return StrUtils.replace(template, "$Startswith_Params", StrUtils.arrayToString(args, "%startswith"));
            }
            if (template.contains("$DelFunc")) {
                return StrUtils.replace(template, "$DelFunc", "");
            }
            if (template.contains("$WITH_RECURSIVE")) {
                return StrUtils.handleWithRecursive(d, args);
            }
            // todo 自定义转换 项目上着急先写了判断 后期看是否要 向上面一样写通用的格式转换和类型转换
            if (template.contains("$CONVERT")) {
                return StrUtils.sqlServerConvertTran(d, args);
            }
            if (template.contains("$CAST")) {
                return StrUtils.sqlServerCastTran(d, args);
            }
            if (template.contains("$DATEDIFF")) {
                return StrUtils.sqlServerDATEDIFFTran(d, args);
            }
            if (template.contains("$NVL_Params")) {
                if (args == null || args.length < 2)
                    DialectException.throwEX("Nvl function require at least 2 parameters");
                else {
                    StringBuilder s = new StringBuilder(STR."nvl(\{args[args.length - 2]}, \{args[args.length - 1]})");
                    for (int i = args.length - 3; i > -1; i--)
                        s = new StringBuilder(STR."nvl(\{args[i]}, \{s})");
                    return StrUtils.replace(template, "$NVL_Params", s.toString());
                }
            }
            return (String) DialectException.throwEX("jDialect found a template bug error, please submit this bug");
        } else {
            int argsCount = 0;
            if (args != null)
                argsCount = args.length;
            String searchStr = STR."\{argsCount}=";
            if (!template.contains(searchStr)) {
                logger.warn("Dialect {}'s function \"{}\" only support {} parameters", d, functionName, allowedParameterQTY(template));
            }
            String result = StrUtils.substringBetween(STR."\{template}|", searchStr, "|");
            for (int i = 0; args != null && i < args.length; i++) {
                result = StrUtils.replace(result, STR."$P\{i + 1}", args[i]);
            }
            return result;
        }
    }

    private static String allowedParameterQTY(String template) {
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < 5; i++) {
            if (template.contains(STR."\{i}="))
                sb.append(i).append(" or ");
        }
        if (!sb.isEmpty())
            sb.setLength(sb.length() - 4);
        return sb.toString();
    }
}
