package com.af.v4.system.common.datasource.dialects;

import com.af.v4.system.common.datasource.dialects.model.ColumnModel;
import com.af.v4.system.common.datasource.dialects.model.TableModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;

/**
 * jDialects is a small Java tool collect all databases' dialect, most data are
 * extracted from Hibernate, usually jDialects is used for build pagination SQL
 * and DDL SQL for cross-databases developing. Currently jDialects support ~70
 * database dialects. It has no any 3rd party dependency, run on JDK1.6 or
 * above.
 *
 * @since 1.7.0
 */
@SuppressWarnings("all")
public class Dialect {

    public static final Dialect OracleDialect = new Dialect("OracleDialect");
    public static final Dialect Oracle9Dialect = new Dialect("Oracle9Dialect");
    public static final Dialect Oracle10gDialect = new Dialect("Oracle10gDialect");
    public static final Dialect Oracle12cDialect = new Dialect("Oracle12cDialect");
    public static final Dialect Oracle8iDialect = new Dialect("Oracle8iDialect");
    public static final Dialect Oracle9iDialect = new Dialect("Oracle9iDialect");

    public static final Dialect SQLiteDialect = new Dialect("SQLiteDialect");

    public static final Dialect MySQLDialect = new Dialect("MySQLDialect");
    public static final Dialect MySQL5InnoDBDialect = new Dialect("MySQL5InnoDBDialect");
    public static final Dialect MySQL57InnoDBDialect = new Dialect("MySQL57InnoDBDialect");
    public static final Dialect MySQLInnoDBDialect = new Dialect("MySQLInnoDBDialect");
    public static final Dialect MySQLMyISAMDialect = new Dialect("MySQLMyISAMDialect");
    public static final Dialect MySQL55Dialect = new Dialect("MySQL55Dialect");
    public static final Dialect MySQL57Dialect = new Dialect("MySQL57Dialect");
    public static final Dialect MySQL5Dialect = new Dialect("MySQL5Dialect");
    public static final Dialect MySQL8Dialect = new Dialect("MySQL8Dialect");

    public static final Dialect PostgreSQLDialect = new Dialect("PostgreSQLDialect");
    public static final Dialect PostgresPlusDialect = new Dialect("PostgresPlusDialect");
    public static final Dialect PostgreSQL9Dialect = new Dialect("PostgreSQL9Dialect");
    public static final Dialect PostgreSQL10Dialect = new Dialect("PostgreSQL10Dialect");
    public static final Dialect PostgreSQL81Dialect = new Dialect("PostgreSQL81Dialect");
    public static final Dialect PostgreSQL82Dialect = new Dialect("PostgreSQL82Dialect");
    public static final Dialect PostgreSQL91Dialect = new Dialect("PostgreSQL91Dialect");
    public static final Dialect PostgreSQL92Dialect = new Dialect("PostgreSQL92Dialect");
    public static final Dialect PostgreSQL93Dialect = new Dialect("PostgreSQL93Dialect");
    public static final Dialect PostgreSQL94Dialect = new Dialect("PostgreSQL94Dialect");
    public static final Dialect PostgreSQL95Dialect = new Dialect("PostgreSQL95Dialect");

    public static final Dialect SQLServerDialect = new Dialect("SQLServerDialect");
    public static final Dialect SQLServer2005Dialect = new Dialect("SQLServer2005Dialect");
    public static final Dialect SQLServer2008Dialect = new Dialect("SQLServer2008Dialect");
    public static final Dialect SQLServer2012Dialect = new Dialect("SQLServer2012Dialect");
    public static final Dialect SQLServer2016Dialect = new Dialect("SQLServer2016Dialect");

    public static final Dialect DamengDialect = new Dialect("DamengDialect");
    public static final Dialect ClickhouseDialect = new Dialect("ClickhouseDialect");

    public static Dialect[] dialects = new Dialect[]{OracleDialect, Oracle9Dialect, MySQL55Dialect, MySQL57Dialect, MySQL57InnoDBDialect, MySQL5Dialect, MySQL5InnoDBDialect, MySQL8Dialect, MySQLDialect, MySQLInnoDBDialect, MySQLMyISAMDialect, Oracle10gDialect, Oracle12cDialect, Oracle8iDialect, Oracle9iDialect, PostgresPlusDialect, PostgreSQL81Dialect, PostgreSQL82Dialect, PostgreSQL91Dialect, PostgreSQL92Dialect, PostgreSQL93Dialect, PostgreSQL94Dialect, PostgreSQL95Dialect, PostgreSQL9Dialect, PostgreSQLDialect, SQLServer2005Dialect, SQLServer2008Dialect, SQLServer2012Dialect, SQLServer2016Dialect, SQLServerDialect, SQLiteDialect, DamengDialect};

    /**
     * If set true will allow use reserved words in DDL, default value is false
     */
    private static Boolean globalAllowReservedWords = false;

    private static final Logger logger = LoggerFactory.getLogger(Dialect.class);

    /**
     * If set true will output log for each paginate, translate, paginAndTranslate,
     * toCreateDDL, toDropAndCreateDDL, toDropDDL method call, default value is
     * false
     */
    private static Boolean globalAllowShowSql = false;

    /**
     * The SQL function prefix String, default value is null
     */
    private static String globalSqlFunctionPrefix = null;

    /**
     * If disable, will use same SqlTemplate for first page pagination query
     */
    public static final String NOT_SUPPORT = "NOT_SUPPORT";
    private static final String SKIP_ROWS = "$SKIP_ROWS";
    private static final String PAGESIZE = "$PAGESIZE";
    private static final String TOTAL_ROWS = "$TOTAL_ROWS";
    private static final String DISTINCT_TAG = "($DISTINCT)";
    public String sqlTemplate;
    public String topLimitTemplate;
    public String name;
    public Map<Type, String> typeMappings = new EnumMap<Type, String>(Type.class);
    public Map<String, String> typeConvertMapping = new HashMap<String, String>();
    public Map<String, String> functions = new HashMap<String, String>();
    public DDLFeatures ddlFeatures = new DDLFeatures();

    static {//Initialize all dialects templates at one time
        DialectTypeMappingTemplate.initTypeMappingTemplates();
        DialectTypeMappingTemplate.initTypeConvertMapping();
        DialectFunctionTemplate.initFunctionTemplates();
        DialectPaginationTemplate.initPaginTemplates();
        DDLFeatures.initDDLFeatures();

        //=================Manual register extra functions in templates ================
        DialectFunctionTemplate.initExtraFunctionTemplates();
    }

    public Dialect() {
        this.name = this.getClass().getSimpleName();
    }

    public Dialect(String name) {
        this.name = name;
    }

    /**
     * Use Dialect.dialects instead
     */
    @Deprecated
    public static Dialect[] values() {
        return dialects;
    }

    /**
     * Check if is current dialect or ANSI reserved word, if yes throw exception. if
     * is other database's reserved word, log output a warning.
     */
    private void checkIfReservedWord(String word, String... tableName) {
        if (ReservedDBWords.isReservedWord(word)) {
            String inTable = tableName.length > 0 ? "In table " + tableName[0] + ", " : "";
            String reservedForDB = ReservedDBWords.reservedForDB(word);
            if (ReservedDBWords.isReservedWord(this, word)) {
                if (Dialect.globalAllowReservedWords)
                    logger.warn(inTable + "\"" + word + "\" is a reserved word of \"" + reservedForDB + "\", should not use it as table, column, unique or index name");
                else
                    DialectException.throwEX(inTable + "\"" + word + "\" is a reserved word of \"" + reservedForDB + "\", should not use it as table, column, unique or index name. " + "if you really want use this reserved word, call Dialect.setGlobalAllowReservedWords() at application starting.");
            } else {
                logger.warn(inTable + "\"" + word + "\" is a reserved word of other database \"" + reservedForDB + "\", not recommend be used as table, column, unique or index name");
            }
        }
    }

    /**
     * Check if a word or word array include current dialect or ANSI-SQL's reserved
     * word, if yes throw exception. if belong to other database's reserved word,
     * log output a warning. Otherwise return word itself or first word if is array
     */
    public String checkReservedWords(String... words) {
        if (words == null || words.length == 0) return null;
        for (String word : words)
            checkIfReservedWord(word);
        return words[0];
    }

    /**
     * Check if a word is current dialect or ANSI-SQL's reserved word, if yes throw
     * exception. if is other database's reserved word, log output a warning.
     * Otherwise return word itself.
     */
    public String checkNotEmptyReservedWords(String word, String type, String tableName) {
        if (StrUtils.isEmpty(word)) DialectException.throwEX(type + " can not be empty");
        checkIfReservedWord(word, tableName);
        return word;
    }

    /**
     * Transfer columnModel to a real dialect's DDL definition String, lengths is
     * optional for some types
     */
    public String translateToDDLType(ColumnModel col) {
        Type type = col.getColumnType();
        String value = this.typeMappings.get(type);
        if (StrUtils.isEmpty(value) || "N/A".equals(value) || "n/a".equals(value))
            DialectException.throwEX("Type \"" + type + "\" is not supported by dialect \"" + this + "\"");

        if (value.contains("|")) {
            // format example: varchar($l)<255|lvarchar($l)<32739|varchar($l)
            String[] typeTempls = StrUtils.split("|", value);
            for (String templ : typeTempls) {
                if (templ.contains("<")) {// varchar($l)<255
                    String[] limitType = StrUtils.split("<", templ);
                    if (col.getLength() > 0 && col.getLength() <= Integer.parseInt(limitType[1]))
                        return replacePlaceHolders(type, limitType[0], col);
                } else {// varchar($l)
                    return replacePlaceHolders(type, templ, col);
                }
            }
            return (String) DialectException.throwEX("Type \"" + type + "\" is not supported by dialect \"" + this + "\" of template:" + value);
        } else if (value.contains("$")) return replacePlaceHolders(type, value, col);
        else return value;
    }

    /**
     * inside function
     */
    private String replacePlaceHolders(Type type, String value, ColumnModel col) {
        String newValue = StrUtils.replace(value, "$l", String.valueOf(col.getLength()));
        if (newValue.contains("$p")) newValue = StrUtils.replace(newValue, "$p", String.valueOf(col.getPrecision()));
        if (newValue.contains("$s")) newValue = StrUtils.replace(newValue, "$s", String.valueOf(col.getScale()));
        return newValue;
    }

    /**
     * An example tell users how to use a top limit SQL for a dialect
     */
    private static String aTopLimitSqlExample(String template) {
        String result = StrUtils.replaceIgnoreCase(template, "$SQL", "select * from users order by userid");
        result = StrUtils.replaceIgnoreCase(result, "$BODY", "* from users order by userid");
        result = StrUtils.replaceIgnoreCase(result, " " + DISTINCT_TAG, "");
        result = StrUtils.replaceIgnoreCase(result, SKIP_ROWS, "0");
        result = StrUtils.replaceIgnoreCase(result, PAGESIZE, "10");
        result = StrUtils.replaceIgnoreCase(result, TOTAL_ROWS, "10");
        return result;
    }

    // ====================================================
    // ====================================================

    public String trans(String... sql) {
        StringBuilder sb = new StringBuilder();
        for (String str : sql)
            sb.append(str);
        return DialectFunctionTranslator.instance.doTranslate(this, sb.toString());
    }

    @Override
    public String toString() {
        return name;
    }

    @Override
    public boolean equals(Object obj) {
        return name.equals(((Dialect) obj).name);
    }

    /**
     * @return true if is MySql family
     */
    public boolean isFamily(String databaseName) {
        return StrUtils.startsWithIgnoreCase(this.toString(), databaseName);
    }

    /**
     * @return true if is MySql family
     */
    public boolean isMySqlFamily() {
        return isFamily("MySQL");
    }

    /**
     * @return true if is Oracle family
     */
    public boolean isOracleFamily() {
        return isFamily("Oracle");
    }

    /**
     * @return true if is SQL Server family
     */
    public boolean isSQLServerFamily() {
        return isFamily("SQLServer");
    }

    /**
     * @return true if is Postgres family
     */
    public boolean isPostgresFamily() {
        return isFamily("Postgres");
    }

    // ===============================================
    // Below are new DDL methods
    // ===============================================

    /**
     * Transfer tables to create DDL
     */
    public String[] toCreateDDL(TableModel... tables) {
        return DDLCreateUtils.toCreateDDL(this, tables);
    }

    /**
     * Transfer tables to drop DDL
     */
    public String[] toDropDDL(TableModel... tables) {
        return DDLDropUtils.toDropDDL(this, tables);
    }

    /**
     * Transfer columnModels to add column DDL String array
     */
    public String[] toAddColumnDDL(ColumnModel... columnModels) {
        return DDLCreateUtils.toAddColumnDDL(this, columnModels);
    }

    /**
     * Transfer columnModels to modify column DDL String array
     */
    public String[] toModifyColumnDDL(ColumnModel... columnModels) {
        return DDLCreateUtils.toModifyColumnDDL(this, columnModels);
    }

    /**
     * Transfer columnModels to drop column DDL String array
     */
    public String[] toDropColumnDDL(ColumnModel... columnModels) {
        return DDLDropUtils.toDropColumnDDL(this, columnModels);
    }

    /**
     * Transfer tables to drop and create DDL String array
     */
    public String[] toDropAndCreateDDL(TableModel... tables) {
        String[] drop = DDLDropUtils.toDropDDL(this, tables);
        String[] create = DDLCreateUtils.toCreateDDL(this, tables);
        return StrUtils.joinStringArray(drop, create);
    }

    /**
     * Build a "drop table xxxx " like DDL String according this dialect
     */
    public String dropTableDDL(String tableName) {
        return ddlFeatures.dropTableString.replaceFirst("_TABLENAME", tableName);
    }

    /**
     * Build a "drop sequence xxxx " like DDL String according this dialect
     */
    public String dropSequenceDDL(String sequenceName) {
        if (DDLFeatures.isValidDDLTemplate(ddlFeatures.dropSequenceStrings))
            return StrUtils.replace(ddlFeatures.dropSequenceStrings, "_SEQNAME", sequenceName);
        else
            return (String) DialectException.throwEX("Dialect \"" + this + "\" does not support drop sequence ddl, on sequence \"" + sequenceName + "\"");
    }

    /**
     * Build a "alter table tableName drop foreign key fkeyName " like DDL String
     * according this dialect
     */
    public String dropFKeyDDL(String tableName, String fkeyName) {
        if (DDLFeatures.isValidDDLTemplate(ddlFeatures.dropForeignKeyString))
            return "alter table " + tableName + " " + ddlFeatures.dropForeignKeyString + " " + fkeyName;
        else
            return (String) DialectException.throwEX("Dialect \"" + this + "\" does not support drop foreign key, on foreign key \"" + fkeyName + "\"");
    }

    // getter & setter====
    public String getName() {
        return name;
    }

    public void setName(String name) {
        this.name = name;
    }

    public DDLFeatures getDdlFeatures() {
        return ddlFeatures;
    }

    public static Boolean getGlobalAllowShowSql() {
        return globalAllowShowSql;
    }


    public static String getGlobalSqlFunctionPrefix() {
        return globalSqlFunctionPrefix;
    }

}
