package com.af.v4.system.api;

import com.af.v4.system.api.domain.SysUser;
import com.af.v4.system.api.factory.RemoteUserFallbackFactory;
import com.af.v4.system.api.model.LoginUser;
import com.af.v4.system.common.core.constant.SecurityConstants;
import com.af.v4.system.common.core.constant.ServiceNameConstants;
import com.af.v4.system.common.core.domain.R;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.Map;

/**
 * 用户服务
 */
@FeignClient(contextId = "remoteUserService", value = ServiceNameConstants.SYSTEM_SERVICE, fallbackFactory = RemoteUserFallbackFactory.class)
public interface RemoteUserService {
        /**
         * 通过用户名查询用户信息
         *
         * @param username 用户名
         * @param source   请求来源
         * @return 结果
         */
        @GetMapping("/user/info/{username}")
        R<LoginUser> getUserInfo(
                        @PathVariable("username") String username,
                        @RequestParam("resourceName") String resourceName,
                        @RequestParam("isMobile") boolean isMobile,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 通过微信openid查询用户信息
         *
         * @param openid 微信openid
         * @param source 请求来源
         * @return 结果
         */
        @GetMapping("/user/wxinfo/{openid}")
        R<LoginUser> getWxUserInfo(
                        @PathVariable("openid") String openid,
                        @RequestParam("resourceName") String resourceName,
                        @RequestParam("isMobile") boolean isMobile,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 登出
         *
         * @param source 请求来源
         * @return 结果
         */
        @GetMapping("/user/logout")
        R<Map<String, Object>> logout(@RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 注册用户信息
         *
         * @param sysUser 用户信息
         * @param source  请求来源
         * @return 结果
         */
        @PostMapping("/user/register")
        R<Boolean> registerUserInfo(@RequestBody SysUser sysUser,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 通过微信code登录查询用户信息
         *
         * @param data   请求参数
         * @param source 请求来源
         * @return 结果
         */
        @PostMapping("/user/unifiedLogin")
        R<LoginUser> unifiedLogin(@RequestBody String data,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * SSO单点登录
         *
         * @param data   请求参数（包含token和config）
         * @param source 请求来源
         * @return 结果
         */
        @PostMapping("/user/ssoLogin")
        R<LoginUser> ssoLogin(@RequestBody String data, @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 外部用户信息查询 - 智能路由核心
         *
         * @param platformType   平台类型（参见 PlatformType 枚举）
         * @param platformUserId 平台用户ID（如微信的openid）
         * @param resourceName   资源名称
         * @param isMobile       是否移动端
         * @param source         请求来源
         * @return 结果（系统用户或外部用户的LoginUser）
         */
        @GetMapping("/user/external-user/info")
        R<LoginUser> getExternalUserInfo(
                        @RequestParam("platformType") String platformType,
                        @RequestParam("platformUserId") String platformUserId,
                        @RequestParam("tenantName") String tenantName,
                        @RequestParam(value = "resourceName", required = false) String resourceName,
                        @RequestParam(value = "isMobile", required = false) boolean isMobile,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 外部用户自动注册
         *
         * @param registerData 注册数据JSON（包含platformType等字段）
         * @param source       请求来源
         * @return 结果（新注册的外部用户LoginUser）
         */
        @PostMapping("/user/external-user/auto-register")
        R<LoginUser> autoRegisterExternalUser(
                        @RequestBody String registerData,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 第三方平台认证接口 - 纯第三方认证
         * 
         * <p>
         * 此接口只负责第三方平台认证，获取第三方用户原始数据，不做任何系统登录处理
         * </p>
         * 
         * @param authData 认证数据（包含loginMode, tenantName, code等）
         * @param source   请求来源
         * @return 第三方平台的原始用户数据（JSONObject）
         */
        @PostMapping("/user/authInfo")
        R<Map<String, Object>> authInfo(
                        @RequestBody String authData,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);

        /**
         * 记录用户会话
         * 
         * @param sessionData 会话数据JSON字符串
         * @param source      请求来源
         * @return 结果
         */
        @PostMapping("/session/record")
        R<Boolean> recordSession(
                        @RequestBody String sessionData,
                        @RequestHeader(SecurityConstants.FROM_SOURCE) String source);
}
