package com.aote.util.sxb;

import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.Base64;
import java.util.HashMap;
import jodd.util.StringUtil;
import org.bouncycastle.crypto.AsymmetricCipherKeyPair;
import org.bouncycastle.crypto.CipherParameters;
import org.bouncycastle.crypto.engines.SM2Engine;
import org.bouncycastle.crypto.generators.ECKeyPairGenerator;
import org.bouncycastle.crypto.params.ECDomainParameters;
import org.bouncycastle.crypto.params.ECKeyGenerationParameters;
import org.bouncycastle.crypto.params.ECKeyParameters;
import org.bouncycastle.crypto.params.ECPrivateKeyParameters;
import org.bouncycastle.crypto.params.ECPublicKeyParameters;
import org.bouncycastle.crypto.params.ParametersWithRandom;
import org.bouncycastle.math.ec.ECCurve;
import org.bouncycastle.math.ec.ECPoint;
import org.bouncycastle.util.test.TestRandomBigInteger;

public class Sm2Tool {
    private SM2Engine sm2Engine;
    private static ECCurve curve;
    private static ECDomainParameters domainParams;
    private String pubk;
    private String prik;
    private static final String PUB = "_pub";
    private static HashMap<String, ECKeyParameters> keys = new HashMap();
    private static HashMap<String, String> pubkeys = new HashMap();

    static {
        BigInteger SM2_ECC_P = new BigInteger("FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFF", 16);
        BigInteger SM2_ECC_A = new BigInteger("FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF00000000FFFFFFFFFFFFFFFC", 16);
        BigInteger SM2_ECC_B = new BigInteger("28E9FA9E9D9F5E344D5A9E4BCF6509A7F39789F515AB8F92DDBCBD414D940E93", 16);
        BigInteger SM2_ECC_N = new BigInteger("FFFFFFFEFFFFFFFFFFFFFFFFFFFFFFFF7203DF6B21C6052B53BBF40939D54123", 16);
        BigInteger SM2_ECC_GX = new BigInteger("32C4AE2C1F1981195F9904466A39C9948FE30BBFF2660BE1715A4589334C74C7", 16);
        BigInteger SM2_ECC_GY = new BigInteger("BC3736A2F4F6779C59BDCEE36B692153D0A9877CC62A474002DF32E52139F0A0", 16);
        curve = new ECCurve.Fp(SM2_ECC_P, SM2_ECC_A, SM2_ECC_B);
        ECPoint g = curve.createPoint(SM2_ECC_GX, SM2_ECC_GY);
        domainParams = new ECDomainParameters(curve, g, SM2_ECC_N);
    }

    public Sm2Tool() {
        this((String)null, (String)null);
    }

    public Sm2Tool(String pubk, String prik) {
        ECKeyPairGenerator keyPairGenerator = new ECKeyPairGenerator();
        if (StringUtil.isBlank(prik) && StringUtil.isBlank(pubk)) {
            prik = "646AF6FC57D918E01C90DBF67C2BE0445078CEADDD4B692313D6DA9D0F811D99";
        }

        if (StringUtil.isNotBlank(prik)) {
            if (!keys.containsKey(prik)) {
                ECKeyGenerationParameters aKeyGenParams = new ECKeyGenerationParameters(domainParams, new TestRandomBigInteger(prik, 16));
                keyPairGenerator.init(aKeyGenParams);
                AsymmetricCipherKeyPair aKp = keyPairGenerator.generateKeyPair();
                ECPublicKeyParameters aPub = (ECPublicKeyParameters)aKp.getPublic();
                ECPrivateKeyParameters aPriv = (ECPrivateKeyParameters)aKp.getPrivate();
                String genPub = FormartUtil.byteToHex(aPub.getQ().getEncoded(false));
                pubkeys.put(prik, genPub);
                pubkeys.put(genPub, prik);
                keys.put(genPub, aPub);
                keys.put(prik, aPriv);
                keys.put(prik + "_pub", aPub);
            }

            this.prik = prik;
            this.pubk = (String)pubkeys.get(prik);
        } else if (StringUtil.isNotBlank(pubk)) {
            if (!keys.containsKey(pubk)) {
                byte[] publicKey = FormartUtil.hexToByte(pubk);
                ECPublicKeyParameters aPub = new ECPublicKeyParameters(curve.decodePoint(publicKey), domainParams);
                keys.put(pubk, aPub);
            }

            this.pubk = pubk;
            this.prik = (String)pubkeys.get(pubk);
        }

        this.sm2Engine = new SM2Engine();
    }

    public String encrypt(String text) throws Exception {
        byte[] m = Base64.getEncoder().encode(text.getBytes("utf-8"));
        String key = this.pubk;
        if (StringUtil.isBlank(key)) {
            key = this.prik + "_pub";
        }

        this.sm2Engine.init(true, new ParametersWithRandom((CipherParameters)keys.get(key), new SecureRandom()));
        byte[] enc = this.sm2Engine.processBlock(m, 0, m.length);
        return FormartUtil.byteToHex(enc);
    }

    public String decrypt(String enc) throws Exception {
        this.sm2Engine.init(false, (CipherParameters)keys.get(this.prik));
        byte[] eec = FormartUtil.hexToByte(enc);
        byte[] dec = this.sm2Engine.processBlock(eec, 0, eec.length);
        byte[] eee = Base64.getDecoder().decode(dec);
        return new String(eee, "utf-8");
    }

    public String getPubk() {
        return this.pubk;
    }

    public String getPrik() {
        return this.prik;
    }

    public static String generatorSM2Key(String aPrik) {
        Sm2Tool util = new Sm2Tool((String)null, aPrik);
        return util.getPubk() + "_" + util.getPrik();
    }

    public static void main(String[] args) throws Exception {
        String text = (new Sm2Tool()).encrypt("123");
        System.out.println(text);
        System.out.println((new Sm2Tool()).decrypt(text));
        System.out.println((new Sm2Tool()).decrypt("0406F778C5AC4DF0DF8C883E78F6A0201638303989B4E1C93E9193406E7A16B09A1D57C367AF6E28BE369054200F69D25124FC1AB57698E2ADF9FCD6358B8839CFD91039712C670B847197C65566F866D17913598F579E7C35FF618BC4AF1A98AE758B306A"));
    }
}
