package com.aote.util.pdf;

import cn.hutool.core.io.FileUtil;
import cn.hutool.core.util.StrUtil;
import com.af.plugins.DateTools;
import com.itextpdf.awt.geom.Rectangle2D;
import com.itextpdf.text.DocumentException;
import com.itextpdf.text.Image;
import com.itextpdf.text.Rectangle;
import com.itextpdf.text.pdf.*;
import com.itextpdf.text.pdf.parser.*;
import lombok.extern.slf4j.Slf4j;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.rendering.PDFRenderer;
import org.json.JSONArray;
import org.json.JSONObject;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;


@Slf4j
public class PdfUtils {



    /**
     * 根据PDF模版生成PDF文件
     * @param templateFilePath PDF模版文件路径
     * @param data 表单数据
     * @param imageData 图片数据 VALUE为图片文件路径
     * @param formFlattening false：生成后的PDF文件表单域仍然可编辑 true：生成后的PDF文件表单域不可编辑
     * @param pdfFilePath 生成PDF的文件路径
     */
    public static String createPDF(String templateFilePath, JSONObject data, JSONObject imageData,Boolean formFlattening,String pdfFilePath) throws Exception{

        File dir = new File(pdfFilePath + File.separator + DateTools.getNow("yyyy-MM-dd"));
        if (!dir.exists()) {
            dir.mkdir();
        }
        File file = null;
        PdfReader reader = null;
        ByteArrayOutputStream bos = null;
        PdfStamper pdfStamper = null;
        FileOutputStream fos = null;
        String filenaeme = null;
        try{
            // 读取PDF模版文件
            reader = new PdfReader(templateFilePath);
            // 输出流
            bos = new ByteArrayOutputStream();
            // 构建PDF对象
            pdfStamper = new PdfStamper(reader, bos);

            // 获取表单数据
            AcroFields form = pdfStamper.getAcroFields();

            // 使用中文字体 使用 AcroFields填充值的不需要在程序中设置字体，在模板文件中设置字体为中文字体 Adobe 宋体 std L
            BaseFont bfChinese = BaseFont.createFont("STSongStd-Light", "UniGB-UCS2-H", BaseFont.NOT_EMBEDDED);
            form.addSubstitutionFont(bfChinese);

            // 表单赋值
            for(String key : data.keySet()){
                form.setField(key,data.getString(key));
                // 也可以指定字体
                form.setFieldProperty(key, "textfont", bfChinese, null);
            }

            // 添加图片
            if(null != imageData && imageData.length() > 0){
                for(String key : imageData.keySet()){
                    int pageNo = form.getFieldPositions(key).get(0).page;
                    Rectangle signRect = form.getFieldPositions(key).get(0).position;
                    float x = signRect.getLeft();
                    float y = signRect.getBottom();
                    // 读图片
                    Image image = Image.getInstance(imageData.getString(key));
                    // 获取操作的页面
                    PdfContentByte under = pdfStamper.getOverContent(pageNo);
                    // 根据域的大小缩放图片
                    image.scaleToFit(signRect.getWidth(), signRect.getHeight());
                    // 添加图片
                    image.setAbsolutePosition(x, y);
                    under.addImage(image);
                }
            }

            // 如果为false那么生成的PDF文件还能编辑，一定要设为true
            pdfStamper.setFormFlattening(formFlattening);
            pdfStamper.close();


            file = new File(dir.getAbsolutePath() + File.separator + DateTools.getNow("yyyy_MM_dd_hh_mm_ss") + "资料真实性声明.pdf");

            // 保存文件
            fos = new FileOutputStream(file);
            fos.write(bos.toByteArray());
            fos.flush();
            filenaeme = file.getAbsolutePath();
        }finally {
            if(null != fos){
                try {fos.close(); }catch (Exception e){e.printStackTrace();}
            }

            if(null != bos){
                try {bos.close(); }catch (Exception e){e.printStackTrace();}
            }

            if(null != reader){
                try {reader.close(); }catch (Exception e){e.printStackTrace();}
            }
        }
        return filenaeme;
    }


    /**
     * 查找pdf中关键字的位置
     *
     * @param filePath pdf文件路径
     * @param key      关键字
     * @param offset 是否偏移(向右偏移出关键字,指关键字后面的位置)
     * @return 坐标点数组
     */
    public static JSONArray findPostions(String filePath, String key,Boolean offset) {
        JSONArray result = new JSONArray();
        try {
            // 读取文件
            File pdfFile = FileUtil.file(filePath);
            byte[] pdfData = FileUtil.readBytes(pdfFile);
            // 调用方法，给定关键字和文件
            List<float[]> positions = findKeywordPostions(pdfData, key);
            if (positions != null && positions.size() > 0) {
                for (float[] position : positions) {
                    JSONObject postion = new JSONObject();
                    postion.put("page", (int) position[0]);
                    if (offset) {
                        postion.put("x", position[1] + (key.length() * position[3]));
                        postion.put("y", position[2] - position[3] - position[3]- position[3]);
//                        postion.put("x", position[1] + (key.length() * position[3]));
//                        postion.put("y", position[2] - position[3]);
                    } else {
                        postion.put("x", position[1]);
                        postion.put("y", position[2]);
                    }
                    result.put(postion);
                }
            }
        } catch (IOException e) {
            log.debug("查找pdf中关键字的位置异常", e);
            throw new RuntimeException("查找pdf中关键字的位置异常", e);
        }
        return result;
    }


    /**
     * 在pdf指定位置插入图片
     * @param pdfPath pdf文件路径
     * @param imagePath 插入图片文件路径
     * @param postions 坐标点{page: 页码,x: x坐标,y: y坐标}
     * @return 添加后的pdf文件相关信息
     */
    public static String pdfAddImage(String pdfPath, String imagePath, JSONObject postions, String savePdfPath) {
        String resultpath = null;
        JSONObject result = new JSONObject();
        try {
            BufferedInputStream inputStream = FileUtil.getInputStream(pdfPath);
            // 签名后的文件路径
            String outFilePath = StrUtil.removeSuffix(savePdfPath, ".pdf") + "_已签.pdf";
            BufferedOutputStream outputStream = FileUtil.getOutputStream(outFilePath);
            int page = postions.getInt("page");
            Double x = postions.getDouble("x");
            Double y = postions.getDouble("y");
            // 读取pdf文件
            PdfReader reader = new PdfReader(inputStream);
            PdfStamper stamper = new PdfStamper(reader, outputStream);
            // 获取要操作的页面
            PdfContentByte under = stamper.getOverContent(page);
            // 要添加的图片
            Image image = Image.getInstance(imagePath);
            // 缩放定位图片
            image.scaleToFit(100, 50);
            image.setAbsolutePosition(x.floatValue(), y.floatValue());
            // 添加图片
            under.addImage(image);
            // 释放资源
            stamper.close();
            reader.close();
            result.put("outFilePath", outFilePath);
            resultpath = outFilePath;
        } catch (Exception e) {
            log.debug("pdf添加图片异常", e);
            throw new RuntimeException("pdf添加图片异常", e);
        }
        return resultpath;
    }


    /**
     * 在pdf指定位置插入图片多张图片
     * @param pdfPath pdf文件路径
     * @param postions 坐标点{page: 页码,x: x坐标,y: y坐标, imagePath: 图片路径}
     * @return 添加后的pdf文件相关信息
     */
    public static JSONObject pdfAddImages(String pdfPath, JSONArray postions, String savePdfPath) {
        JSONObject result = new JSONObject();
        PdfReader reader = null;
        PdfStamper stamper = null;
        try {
            BufferedInputStream inputStream = FileUtil.getInputStream(pdfPath);
            // 签名后的文件路径
            String outFilePath = StrUtil.removeSuffix(savePdfPath, ".pdf") + "_已签.pdf";
            BufferedOutputStream outputStream = FileUtil.getOutputStream(outFilePath);
            // 读取pdf文件
            reader = new PdfReader(inputStream);
            stamper = new PdfStamper(reader, outputStream);
            for(Object o : postions) {
                JSONObject item = (JSONObject) o;
                int page = item.getInt("page");
                Double x = item.getDouble("x");
                Double y = item.getDouble("y");
                String imagePath = item.getString("imagePath");
                // 获取要操作的页面
                PdfContentByte under = stamper.getOverContent(page);
                // 要添加的图片
                Image image = Image.getInstance(imagePath);
                // 缩放定位图片
                image.scaleToFit(100, 200);
                image.setAbsolutePosition(x.floatValue(), y.floatValue());
                // 添加图片
                under.addImage(image);
            }
            result.put("outFilePath", outFilePath);
        } catch (Exception e) {
            log.debug("pdf添加图片异常", e);
            throw new RuntimeException("pdf添加图片异常", e);
        } finally {
            // 释放资源
            if(stamper != null){
                try {
                    stamper.close();
                } catch (DocumentException | IOException e) {
                    e.printStackTrace();
                }
            }
            if(reader != null) {
                reader.close();
            }
        }
        return result;
    }


    public static String textMergePdf (String oldPdfPath,JSONObject data, String newPdfPath, String size) {
        try {
        // 初始化itext
        // 设置编码
        // BaseFont baseFont = BaseFont.createFont("STSong-Light", "UniGB-UCS2-H", BaseFont.NOT_EMBEDDED);
        // 仿宋体
        BaseFont baseFont = BaseFont.createFont("c://windows//fonts//SIMFANG.TTF", BaseFont.IDENTITY_H, BaseFont.NOT_EMBEDDED);
        PdfReader pdfReader = new PdfReader(oldPdfPath);
        PdfStamper pdfStamper = new PdfStamper(pdfReader, new FileOutputStream(newPdfPath));
        AcroFields form = pdfStamper.getAcroFields();
        form.addSubstitutionFont(baseFont);
        // 写入数据
        for(String key : data.keySet()){
            String value = data.get(key).toString();
            // key对应模板数据域的名称
            float fontSize = Float.parseFloat(size);
            form.setFieldProperty(key,"textsize", fontSize,null);
            form.setField(key, value, true);
        }
        // 设置不可编辑
        pdfStamper.setFormFlattening(true);
        pdfStamper.close();
        pdfReader.close();
        } catch (DocumentException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }
        return newPdfPath;
    }



    public static void main(String[] args) {
        // 要插入的数据
        JSONObject data = new JSONObject();
        data.put("gasConsumption", "测试");
        data.put("idnumber", "123");
        System.out.println(data);
        String pdfPath = "d:/handplanImage/民用供气合同.pdf";
        String newPdfPath = "d:/handplanImage/2022-02-22/396516f7-1bef-47f4-a82b-a6e588c3cf37_gasSupplyContract.pdf";
        String size = "10f";
        textMergePdf(pdfPath, data, newPdfPath, size);
//        JSONArray row = new JSONArray();
//        row.put(new JSONObject().put("str","测试").put("key","用气方："));
//        row.put(new JSONObject().put("str","1234567").put("key","身份证号："));
//        AtomicReference<String> pdfPath = new AtomicReference<>("d:/handplanImage/民用供气合同.pdf");
//        JSONArray postions = new JSONArray();
//        row.forEach((item)->{
//            JSONObject ob = ((JSONObject) item);
//            String imagePath = "d:/handplanImage/fontToImg/"+ob.getString("str")+".png";
//            String imgPath = null;
//            try {
//                imgPath = FontToImage.createImage(ob.getString("str"), "微软雅黑", 32, imagePath, 200, 64);
//            } catch (Exception e) {
//                e.printStackTrace();
//            }
//            log.debug("==================>"+imgPath);
//            // 生成需要追加到签名pdf的位置和图片路径信息
//            JSONObject postion = PdfUtils.findPostions(pdfPath.get(), ob.getString("key"), true).getJSONObject(0);
//            postion.put("imagePath", imgPath);
//            postions.put(postion);
//            log.debug("postions ==================>"+postions);
//            System.out.println(postions);
//        });
//        String file = DateTools.getNow("yyyy-MM-dd");
//        String signPdfFilePath = "d:/handplanImage/"+file+"/396516f7-1bef-47f4-a82b-a6e588c3cf37_gasSupplyContract.pdf";
//        JSONObject signPdf = PdfUtils.pdfAddImages(pdfPath.get(), postions, signPdfFilePath);
//        log.debug("signPdf签名==================>"+signPdf);
//        System.out.println(signPdf);
    }

    /**
     * findKeywordPostions
     *
     * @param pdfData 通过IO流 PDF文件转化的byte数组
     * @param keyword 关键字
     * @return List<float[]>: float[0]:pageNum float[1]:x float[2]:y
     * @throws IOException
     */
    public static List<float[]> findKeywordPostions(byte[] pdfData, String keyword) throws IOException {
        List<float[]> result = new ArrayList<>();
        List<PdfPageContentPositions> pdfPageContentPositions = getPdfContentPostionsList(pdfData);


        for (PdfPageContentPositions pdfPageContentPosition : pdfPageContentPositions) {
            List<float[]> charPositions = findPositions(keyword, pdfPageContentPosition);
            if (charPositions == null || charPositions.size() < 1) {
                continue;
            }
            result.addAll(charPositions);
        }
        return result;
    }


    private static List<PdfPageContentPositions> getPdfContentPostionsList(byte[] pdfData) throws IOException {
        PdfReader reader = new PdfReader(pdfData);


        List<PdfPageContentPositions> result = new ArrayList<>();


        int pages = reader.getNumberOfPages();
        for (int pageNum = 1; pageNum <= pages; pageNum++) {
            float width = reader.getPageSize(pageNum).getWidth();
            float height = reader.getPageSize(pageNum).getHeight();


            PdfRenderListener pdfRenderListener = new PdfRenderListener(pageNum, width, height);


            //解析pdf，定位位置
            PdfContentStreamProcessor processor = new PdfContentStreamProcessor(pdfRenderListener);
            PdfDictionary pageDic = reader.getPageN(pageNum);
            PdfDictionary resourcesDic = pageDic.getAsDict(PdfName.RESOURCES);
            try {
                processor.processContent(ContentByteUtils.getContentBytesForPage(reader, pageNum), resourcesDic);
            } catch (IOException e) {
                reader.close();
                throw e;
            }


            String content = pdfRenderListener.getContent();
            List<CharPosition> charPositions = pdfRenderListener.getcharPositions();


            List<float[]> positionsList = new ArrayList<>();
            for (CharPosition charPosition : charPositions) {
                float[] positions = new float[]{charPosition.getPageNum(), charPosition.getX(), charPosition.getY(), charPosition.getCharWidth()};
                positionsList.add(positions);
            }


            PdfPageContentPositions pdfPageContentPositions = new PdfPageContentPositions();
            pdfPageContentPositions.setContent(content);
            pdfPageContentPositions.setPostions(positionsList);


            result.add(pdfPageContentPositions);
        }
        reader.close();
        return result;
    }


    private static List<float[]> findPositions(String keyword, PdfPageContentPositions pdfPageContentPositions) {


        List<float[]> result = new ArrayList<>();


        String content = pdfPageContentPositions.getContent();
        List<float[]> charPositions = pdfPageContentPositions.getPositions();


        for (int pos = 0; pos < content.length(); ) {
            int positionIndex = content.indexOf(keyword, pos);
            if (positionIndex == -1) {
                break;
            }
            float[] postions = charPositions.get(positionIndex);
            result.add(postions);
            pos = positionIndex + 1;
        }
        return result;
    }


    private static class PdfPageContentPositions {
        private String content;
        private List<float[]> positions;


        public String getContent() {
            return content;
        }


        public void setContent(String content) {
            this.content = content;
        }


        public List<float[]> getPositions() {
            return positions;
        }


        public void setPostions(List<float[]> positions) {
            this.positions = positions;
        }
    }


    private static class PdfRenderListener implements RenderListener {
        private final int pageNum;
        private final float pageWidth;
        private final float pageHeight;
        private final StringBuilder contentBuilder = new StringBuilder();
        private final List<CharPosition> charPositions = new ArrayList<>();


        public PdfRenderListener(int pageNum, float pageWidth, float pageHeight) {
            this.pageNum = pageNum;
            this.pageWidth = pageWidth;
            this.pageHeight = pageHeight;
        }


        @Override
        public void beginTextBlock() {
        }


        @Override
        public void renderText(TextRenderInfo renderInfo) {
            List<TextRenderInfo> characterRenderInfos = renderInfo.getCharacterRenderInfos();
            for (TextRenderInfo textRenderInfo : characterRenderInfos) {
                String word = textRenderInfo.getText();
                if (word.length() > 1) {
                    word = word.substring(word.length() - 1);
                }
                Rectangle2D.Float rectangle = textRenderInfo.getAscentLine().getBoundingRectange();

                float x = (float) rectangle.getX();
                float y = (float) rectangle.getY();
//                float x = (float)rectangle.getCenterX();
//                float y = (float)rectangle.getCenterY();
//                double x = rectangle.getMinX();
//                double y = rectangle.getMaxY();


                //这两个是关键字在所在页面的XY轴的百分比
                float xPercent = Math.round(x / pageWidth * 10000) / 10000f;
                float yPercent = Math.round((1 - y / pageHeight) * 10000) / 10000f;

                float charWidth = (float) (rectangle.getMaxX() - rectangle.getMinX());
                CharPosition charPosition = new CharPosition(pageNum, x, y, charWidth);
                charPositions.add(charPosition);
                contentBuilder.append(word);
            }
        }


        @Override
        public void endTextBlock() {
        }


        @Override
        public void renderImage(ImageRenderInfo renderInfo) {
        }


        public String getContent() {
            return contentBuilder.toString();
        }


        public List<CharPosition> getcharPositions() {
            return charPositions;
        }
    }

    public static String pdf2img(String location){
        log.debug("进入pdf2img方法");
        if (location.startsWith("http://www.ezhougas.com/")){
            location = location.replaceFirst("http://www.ezhougas.com/", "D:\\\\\\\\");
        }
        log.debug("pdf location: " + location);
        final File targetFile = new File(location);
        final String imageType = "png";
        List<String> imgList = new ArrayList<>();
        try (final PDDocument document = PDDocument.load(targetFile)) {
            final PDFRenderer renderer = new PDFRenderer(document);
            final int pageSize = document.getNumberOfPages();
            for (int i = 0; i < pageSize; i++) {
                final BufferedImage image = renderer.renderImage(i, 2);
                final File outputPath = getOutputPath(targetFile, imageType, i + 1);
                ImageIO.write(image, imageType, outputPath);
                imgList.add(outputPath.getAbsolutePath());
            }
        }catch (Exception e){
            e.printStackTrace();
        }
        String ret = imgList.stream().map(Object::toString).collect(Collectors.joining(","));
        log.debug("ret: " + ret);
        return ret;
    }

    private static File getOutputPath(File file, String imageType, int page) {
        final String fullName = file.getName();
        final String preName = new StringBuilder(fullName).substring(0, fullName.length() - 4);
        final String parentPath = file.getParent();

        return new File(String.format("%s%s%s_%d.%s", parentPath, File.separator, preName, page, imageType));
    }
    private static class CharPosition {
        private int pageNum = 0;
        private float x = 0;
        private float y = 0;
        // 单个文字的宽度
        private float charWidth = 0;


        public CharPosition(int pageNum, float x, float y, float charWidth) {
            this.pageNum = pageNum;
            this.x = x;
            this.y = y;
            this.charWidth = charWidth;
        }


        public int getPageNum() {
            return pageNum;
        }


        public float getX() {
            return x;
        }


        public float getY() {
            return y;
        }

        public float getCharWidth() {
            return charWidth;
        }
    }
}
