package com.aote.pay.yuantiao;

import com.aote.util.ResourceHelper;
import org.bouncycastle.jce.provider.BouncyCastleProvider;

import java.io.IOException;
import java.io.InputStream;
import java.security.*;
import java.util.Base64;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

public class SignUtil {

    /**
     * 基于Map存储多商户RSA私钥
     */
    private final static Map<String, KeyStore> certKeyStoreMap = new ConcurrentHashMap<String, KeyStore>();


    /**
     * 签名
     *
     * @param: pfxPath
     * @param: pfxPwd
     * @param: object
     * @return: String
     */
    public static String sign(String pfxPath, String pfxPwd, String object) throws Exception {
        /*=====1.根据原文获取摘要信息=====*/
        String digest = DigestUtil.getDigest(object);
        /*=====2.获取私钥信息=====*/
        PrivateKey privateKey = getPrivateKey(pfxPath, pfxPwd);
        String prikeyStr = Base64.getEncoder().encodeToString(privateKey.getEncoded());
        /*=====3.对摘要信息进行签名=====*/
        String signMsg = getSignMsg(privateKey, digest);
        return signMsg;
    }


    /**
     * 获取签名信息
     *
     * @param: privateKey
     * @param: digest
     * @return: String
     */
    public static String getSignMsg(PrivateKey privateKey, String digest)
            throws Exception {
        /*====1.根据对应算法获取签名对象实例====*/
        Signature signature = Signature.getInstance("SHA256withRSA");
        /*====2.根据私钥初始化签名对象=====*/
        signature.initSign(privateKey);
        /*====3.将摘要信息更新到签名对象====*/
        byte[] signDigest = digest.getBytes("UTF-8");
        signature.update(signDigest);
        /*====4.加签====*/
        byte[] result = signature.sign();
        /*====5.返回签名信息====*/
        String signMsg = Base64.getEncoder().encodeToString(result);
        return signMsg;
    }

    /**
     * 获取私钥信息
     *
     * @param: certPath
     * @param: certPwd
     * @return: PrivateKey
     */
    public static PrivateKey getPrivateKey(String certPath, String certPwd) {
        /*=====1.如果缓存中无私钥，重新加载私钥信息=====*/
        if (!certKeyStoreMap.containsKey(certPath)) {
            loadRsaCert(certPath, certPwd);
        }
        try {
            /*=====2.从缓存中获取私钥信息=====*/
            Enumeration<String> aliasenum = certKeyStoreMap.get(certPath).aliases();
            String keyAlias = null;
            if (aliasenum.hasMoreElements()) {
                keyAlias = aliasenum.nextElement();
            }
            PrivateKey privateKey = (PrivateKey) certKeyStoreMap.get(certPath)
                    .getKey(keyAlias, certPwd.toCharArray());
            return privateKey;
        } catch (KeyStoreException e) {
            e.printStackTrace();
            return null;
        } catch (UnrecoverableKeyException e) {
            e.printStackTrace();
            return null;
        } catch (NoSuchAlgorithmException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 加载RSA签名证书
     *
     * @param certFilePath：私钥文件路径
     * @param certPwd：私钥密码
     */
    public static void loadRsaCert(String certFilePath, String certPwd) {
        KeyStore keyStore = null;
        try {
            keyStore = getKeyInfo(certFilePath, certPwd, "PKCS12");
            certKeyStoreMap.put(certFilePath, keyStore);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 将证书文件读取为证书存储对象
     *
     * @param pfxkeyfile 证书文件名
     * @param keypwd     证书密码
     * @param type       证书类型
     * @return 证书对象
     * @throws IOException
     */
    public static KeyStore getKeyInfo(String pfxkeyfile, String keypwd,
                                      String type) throws IOException {
        InputStream inputStream = null;
        try {
            KeyStore ks = null;
            if ("JKS".equals(type)) {
                ks = KeyStore.getInstance(type);
            } else if ("PKCS12".equals(type)) {
                String jdkVendor = System.getProperty("java.vm.vendor");
                if (null != jdkVendor && jdkVendor.startsWith("IBM")) {
                    // 如果使用IBMJDK,则强制设置BouncyCastleProvider的指定位置,解决使用IBMJDK时兼容性问题
                    Security.insertProviderAt(
                            new BouncyCastleProvider(), 1);
                } else {
                    Security.addProvider(new BouncyCastleProvider());
                }
                ks = KeyStore.getInstance(type);
            }
            inputStream  = ResourceHelper.getStream(pfxkeyfile);
            char[] nPassword = null;
            nPassword = null == keypwd || "".equals(keypwd.trim()) ? null
                    : keypwd.toCharArray();
            if (null != ks) {
                ks.load(inputStream, nPassword);
            }
            return ks;
        } catch (Exception e) {
            e.printStackTrace();
            if (Security.getProvider("BC") == null) {
            }
            if ((e instanceof KeyStoreException) && "PKCS12".equals(type)) {
                Security.removeProvider("BC");
            }
            throw new RuntimeException("店铺私钥文件和密码不对应！");
            //return null;
        } finally {
            if (null != inputStream) {
                inputStream.close();
            }
        }
    }


}

