package com.aote.webmeter.module.apply.timer;

import com.aote.entity.EntityServer;
import com.aote.sql.SqlServer;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Profile;
import org.springframework.scheduling.annotation.Scheduled;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

// 每小时自动更新任务统计数据
@Component
@Profile("prod")
@Transactional
public class UpdateWorkflowTaskStatisticsSchedule {

    @Autowired
    SqlServer sqlServer;
    @Autowired
    EntityServer entityServer;

    private JSONArray applyList;
    private String currentTime;
    private Map<Integer, List<JSONObject>> applyStepMap;

    @Scheduled(cron = "0 0 * * * ?")
    public void doJob() throws Exception {
//        查询数据
        this.applyList = sqlServer.query("getApply", "", 0);
        JSONArray applyStepList = sqlServer.query("getApplyStep", "", 0);
//        stepList 转 Map
        this.applyStepMap = new HashMap<>(256);
        for (Object stepObject : applyStepList) {
            JSONObject step = (JSONObject) stepObject;
            int applyId = step.getInt("apply_id");
            List<JSONObject> list = this.applyStepMap.computeIfAbsent(applyId, k -> new ArrayList<>(8));
            list.add(step);
        }
//        当前时间
        this.currentTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));
//        更新统计数据
        this.updateEmployeeTaskStatistics();
        this.updateEntryStatistics("process");
        this.updateEntryStatistics("project");

    }

    /**
     * 更新条目统计数据
     *
     * @param entryName 条目名称
     * @throws Exception 数据更新异常
     */
    public void updateEntryStatistics(String entryName) throws Exception {
//        条目键名称
        String keyName = entryName + "_name";
//        结果数据集
        Map<String, JSONObject> resultMap = new HashMap<>(128);
        for (Object applyObject : this.applyList) {
            JSONObject apply = (JSONObject) applyObject;
//            当前条目统计数据
            JSONObject entryStatisticsData = resultMap.computeIfAbsent(apply.getString(keyName), k -> {
                JSONObject data = new JSONObject();
                data.put("f_" + keyName, apply.getString(keyName));
                data.put("f_progress_task_num", 0);
                data.put("f_overdue_num", 0);
                data.put("f_total_tasks", 0);
                data.put("f_completed_tasks_num", 0);
                data.put("f_update_time", currentTime);
                return data;
            });
//            总任务数
            this.jsonObjectKeyIncrement("f_total_tasks", entryStatisticsData);
//            已完成任务数量
            if (apply.getInt("state") == 1) {
                this.jsonObjectKeyIncrement("f_completed_tasks_num", entryStatisticsData);
            }
//            正在进行任务数量
            if (apply.getInt("state") != 1) {
                this.jsonObjectKeyIncrement("f_progress_task_num", entryStatisticsData);
//                正在进行中逾期任务数量
                List<JSONObject> stepList = this.applyStepMap.get(apply.getInt("id"));
                for (JSONObject step : stepList) {
                    if (step.getInt("step_id") == apply.getInt("step_id") && this.isTaskOverdue(step)) {
                        this.jsonObjectKeyIncrement("f_overdue_num", entryStatisticsData);
                        break;
                    }
                }
            }
        }
        for (Map.Entry<String, JSONObject> entry : resultMap.entrySet()) {
            entityServer.partialSave("t_apply_" + entryName + "_statistics", entry.getValue());
        }

    }

    /**
     * 更新员工任务统计数据
     *
     * @throws Exception 数据更新异常
     */
    public void updateEmployeeTaskStatistics() throws Exception {
//        查询数据
        JSONArray employeeList = sqlServer.query("getAllEmployee", "");
//        遍历员工，计算统计数据
        for (Object e : employeeList) {
            JSONObject employee = (JSONObject) e;
            String employeeName = employee.getString("f_employee_name");
            int todoNum = 0;
            int overdueNum = 0;
            int totalTasks = 0;
            int completedTasksNum = 0;
            for (Object applyObject : this.applyList) {
                JSONObject apply = (JSONObject) applyObject;
                List<JSONObject> stepList = this.applyStepMap.get(apply.getInt("id"));
//                跳过未初始化流程
                if (stepList == null) {
                    continue;
                }
//                是否完成
                boolean isCompleted = false;
//                当前员工是否参与
                boolean isJoinIn = false;
//                是否逾期
                boolean isTimeout = false;
//                步骤顺序排序，方便获取
                stepList.sort(Comparator.comparingInt(o -> o.getInt("step_id")));
                for (JSONObject step : stepList) {
                    if (!step.getString("handler").contains(employeeName)) {
                        continue;
                    }
//                    待办数量
                    if (apply.getInt("state") != 1 && apply.getInt("step_id") == step.getInt("step_id")) {
                        todoNum++;
                    }
//                    当前进行中任务是否逾期
                    if (apply.getInt("state") != 1 && apply.getInt("step_id") == step.getInt("step_id")) {
                        isTimeout = this.isTaskOverdue(step);
                    }
                    if (!isCompleted && apply.getInt("state") == 1) {
                        isCompleted = true;
                    }
                    if (!isJoinIn) {
                        isJoinIn = true;
                    }
                }
                if (isCompleted) {
                    completedTasksNum++;
                }
                if (isJoinIn) {
                    totalTasks++;
                }
                if (isTimeout) {
                    overdueNum++;
                }
            }
            employee.put("f_todo_num", todoNum);
            employee.put("f_overdue_num", overdueNum);
            employee.put("f_overdue_rate", Math.round((double) overdueNum / todoNum * 100));
            employee.put("f_total_tasks", totalTasks);
            employee.put("f_completed_tasks_num", completedTasksNum);
            employee.put("f_update_time", this.currentTime);
            entityServer.partialSave("t_apply_employee_task_statistics", employee);
        }
    }

    /**
     * JSONObject 字段增加 1
     *
     * @param key key
     * @param o   JSONObject
     */
    private void jsonObjectKeyIncrement(String key, JSONObject o) {
        o.put(key, o.optInt(key) + 1);
    }

    /**
     * 判断步骤是否逾期
     *
     * @param step step JSONObject
     * @return boolean
     */
    private boolean isTaskOverdue(JSONObject step) {
        String deadline = step.optString("deadline");
        String submitDate = step.optString("step_date");
        return (submitDate.isEmpty() ? currentTime : submitDate).compareTo(deadline) > 0;
    }

}
