package com.aote.webmeter.tools.iot.onenet;

import com.af.plugins.HttpConnectionPoolUtil;
import com.af.plugins.RestTools;
import com.aote.webmeter.enums.OnenetApiEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.http.client.methods.HttpPost;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import java.io.IOException;

/**
 * 移动Onenet-IOT平台 API调用工具
 * @author Mr.river
 */
public class OnenetRestTools {
    private static final Logger logger = Logger.getLogger(OnenetRestTools.class);

    /**
     * 创建设备
     * @param params 请求参数
     * @param imei imei码
     * @param imsi imsi码
     * @return 响应结果
     */
    public static JSONObject createDevice(JSONObject params, String imei, String imsi) throws IOException {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue();
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //设置imei和imsi
        JSONObject authInfo = new JSONObject();
        authInfo.put(imei, imsi);
        params.put("auth_info", authInfo);
        //发送请求
        return new JSONObject(HttpConnectionPoolUtil.request(path,
                params.toString(),
                headers.toString(),
                new HttpPost()
        ));
    }

    /**
     * 获取设备信息
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public static JSONObject getDevice(String deviceId) {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue() + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        return new JSONObject(RestTools.getJson(path, headers));
    }

    /**
     * 删除设备
     *
     * @param deviceId 设备ID
     * @return 响应结果
     */
    public static JSONObject deleteDevice(String deviceId) {
        //指定请求路径
        String path = OnenetApiEnum.DEVICE_URL.getValue() + "/" + deviceId;
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        return new JSONObject(RestTools.delete(path, null, headers));
    }

    /**
     * 读取设备资源
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @return 响应结果
     */
    public static JSONObject readDeviceRes(String imei, String timeout) {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString("onenetObjectID"));
        params.put("obj_inst_id", WebMeterInfo.getString("onenetObjectInstID"));
        params.put("res_id", WebMeterInfo.getString("onenetResID"));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.RESOURCE_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        return new JSONObject(RestTools.getJson(path, headers));
    }

    /**
     * 写入设备资源
     * @param imei 设备imei
     * @param mode 写入模式（1或2）
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public static JSONObject writeDeviceRes(String imei, String mode, String timeout, JSONObject content) throws IOException {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString("onenetObjectID"));
        params.put("obj_inst_id", WebMeterInfo.getString("onenetObjectInstID"));
        params.put("mode", mode == null ? 1 : mode);
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.RESOURCE_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        return new JSONObject(HttpConnectionPoolUtil.request(path,
                content.toString(),
                headers.toString(),
                new HttpPost()
        ));
    }

    /**
     * 命令下发
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public static JSONObject sendCommand(String imei, String timeout, JSONObject content) throws IOException {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString("onenetObjectID"));
        params.put("obj_inst_id", WebMeterInfo.getString("onenetObjectInstID"));
        params.put("res_id", WebMeterInfo.getString("onenetResID"));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.COMMAND_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        long begin = System.currentTimeMillis();
        String result = HttpConnectionPoolUtil.request(path,
                content.toString(),
                headers.toString(),
                new HttpPost());
        long end = System.currentTimeMillis();
        logger.warn("移动请求接口耗时："+(end-begin)+"ms");
        return new JSONObject(result);
    }

    /**
     * 订阅
     *
     * @param params 请求参数
     * @return 响应结果
     */
    public static JSONObject subscribe(JSONObject params) throws IOException {
        //指定请求路径
        String path = RestTools.formatURL(OnenetApiEnum.SUBSCRIPTIONS_URL.getValue(), params);
        //设置请求头
        JSONObject headers = new JSONObject();
        headers.put("api-key", WebMeterInfo.getString("onenetAPIKey"));
        //发送请求
        String result = HttpConnectionPoolUtil.request(path,
                null,
                headers.toString(),
                new HttpPost());
        return new JSONObject(result);
    }
}
