package com.aote.webmeter.tools;

import com.aote.module.ModuleMapper;
import com.aote.webmeter.enums.ModuleEnum;
import com.aote.webmeter.enums.SaleVersionEnum;
import com.aote.webmeter.enums.WebmeterExceptionCodeEnum;
import com.aote.webmeter.enums.WebmeterPropertiesEnum;
import com.aote.webmeter.exception.WebmeterException;
import org.apache.log4j.Logger;

import java.util.HashMap;
import java.util.Map;

/**
 * 物联网表模块参数工具类
 * moduleName用于获取参数时指定模块
 * @author Mr.river
 */
public class WebMeterInfo {
    private static final Logger LOGGER = Logger.getLogger(WebMeterInfo.class);
    /**
     * 物联网表模块参数文件
     */
    private static final Map<String, HashMap<String,String>> PROPERTIES_MAP;
    /**
     * 物联网表模块名
     */
    private static final ThreadLocal<String> MODULE_NAME;
    /**
     * 物联网表营收版本名
     */
    private static final ThreadLocal<SaleVersionEnum> SALE_NAME;

    static {
        MODULE_NAME = new ThreadLocal<>();
        SALE_NAME = new ThreadLocal<>();
        PROPERTIES_MAP = new HashMap<>(1);
        //获取物联网表配置文件并加载
        load();
    }

    /**
     * 获取物联网表配置文件并加载
     */
    public static void load() {
        // 获取所有引入的模块
        Map<String, Map<String, String>> map = ModuleMapper.getMap();
        for (String moduleName : map.keySet()) {
            //获取物联网表模块配置文件
            HashMap<String, String> config;
            String modulePath = map.get(moduleName).get("path");
            if (modulePath != null) {
                config = WebmeterPropertiesTools.readWebMeter(modulePath, moduleName);
            } else {
                config = WebmeterPropertiesTools.readWebMeter(moduleName);
            }
            if (config != null) {
                String saleNameParam = WebmeterPropertiesEnum.SALE_NAME.getValue();
                //如果配置了营收版本，但没有找到对应的类型，则抛出异常
                if (config.containsKey(saleNameParam)) {
                    if (!SaleVersionEnum.is(config.get(saleNameParam))) {
                        throw new WebmeterException(WebmeterExceptionCodeEnum.SALE_VERSION_NOT_FIND_FAIL, "模块：" + moduleName);
                    }
                } else {
                    LOGGER.warn("【" + moduleName + "】模块参数文件未配置营收版本名称(" + saleNameParam + ")，可能导致通用业务兼容性问题");
                }
                PROPERTIES_MAP.put(moduleName, config);
                LOGGER.debug("【" + moduleName + "】模块参数文件已加载");
            }
        }
    }

    /**
     * 设置物联网表模块
     * @param name 物联网表模块名称
     */
    public static void setModuleName(String name){
        if(PROPERTIES_MAP.containsKey(name)) {
            MODULE_NAME.set(name);
            //如果模块配置文件中含有营收版本名称，则设置营收版本名称值
            String saleNameParam = WebmeterPropertiesEnum.SALE_NAME.getValue();
            if(has(saleNameParam)){
                WebMeterInfo.setSaleName(SaleVersionEnum.toType(getString(saleNameParam)));
            } else if(has(ModuleEnum.PUBLIC_MODULE.getValue(),saleNameParam)){
                //如果公共模块配置文件中含有营收版本名称，则设置营收版本名称值
                WebMeterInfo.setSaleName(SaleVersionEnum.toType(getStringByModule(ModuleEnum.PUBLIC_MODULE.getValue(),saleNameParam)));
            }
        } else {
            throw new WebmeterException(WebmeterExceptionCodeEnum.MODULE_NOT_FIND_FAIL,"模块："+name);
        }
    }

    /**
     * 设置物联网表营收名称
     * @param saleVersionEnum 物联网表营收名称枚举
     */
    public static void setSaleName(SaleVersionEnum saleVersionEnum){
        SALE_NAME.set(saleVersionEnum);
    }

    /**
     * 获取值
     * @param key key
     * @return value
     */
    public static String getString(String key){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).get(key);
    }

    /**
     * 获取值（取不到时取默认值）
     * @param key key
     * @param defaultValue 默认值
     * @return value
     */
    public static String getString(String key, String defaultValue){
        String value = PROPERTIES_MAP.get(MODULE_NAME.get()).get(key);
        return value != null ? value : defaultValue;
    }

    /**
     * 获取当前模块的参数Map
     * @return value
     */
    public static Map<String, String> getMap(){
        return PROPERTIES_MAP.get(MODULE_NAME.get());
    }

    /**
     * 获取指定模块的值
     * @param key key
     * @return value
     */
    public static String getStringByModule(String moduleName, String key){
        return PROPERTIES_MAP.get(moduleName).get(key);
    }

    /**
     * 获取当前模块名
     * @return moduleName
     */
    public static String getModuleName() {
        return MODULE_NAME.get();
    }

    /**
     * 获取当前营收版本名称
     * @return saleName
     */
    public static String getSaleName() {
        return SALE_NAME.get().getValue();
    }

    /**
     * 获取当前营收版本名称枚举
     * @return saleNameEnum
     */
    public static SaleVersionEnum getSaleNameEnum() throws WebmeterException{
        SaleVersionEnum saleVersionEnum = SALE_NAME.get();
        if(saleVersionEnum == null){
            throw new WebmeterException(WebmeterExceptionCodeEnum.SALE_VERSION_NULL_FAIL,"模块："+MODULE_NAME.get());
        }
        return saleVersionEnum;
    }

    /**
     * 判断是否含有该key
     * @param key key
     * @return isHas
     */
    public static Boolean has(String key){
        return PROPERTIES_MAP.get(MODULE_NAME.get()).containsKey(key);
    }

    /**
     * 判断指定模块是否含有该key
     * @param key key
     * @return isHas
     */
    public static Boolean has(String module, String key){
        return PROPERTIES_MAP.get(module).containsKey(key);
    }

    /**
     * 清除设置的物联网表模块
     */
    public static void removeModuleName(){
        MODULE_NAME.remove();
    }

    /**
     * 清除设置的物联网营收名称
     */
    public static void removeSaleName(){
        SALE_NAME.remove();
    }

    /**
     * 判断是否含有该模块
     * @param key key
     * @return isHas
     */
    public static Boolean hasModule(String key){
        return PROPERTIES_MAP.containsKey(key);
    }
}
