package com.aote.webmeter.timer;

import com.aote.logic.LogicServer;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * 物联网表缓存推送队列Timer
 * *****
 * 重要：该定时器用于IOT平台高并发推送情景的业务处理，处理能力尚未经过完全测试，谨慎用于生产项目中
 * 目前已用于以下生产项目：
 * 1.榆林榆川金卡，千嘉，中奕物联网表兼容
 * *****
 * 实现原理：将IOT平台推送数据存入缓存队列中，由定时器定时启动线程，获取并消费数据
 * 优点：先响应请求，再处理数据，可以有效提高高并发情况下的响应速度，避免平台重复推送数据
 *
 * @author Mrriver
 * @date 2019/8/14
 */

@Component
public class WebMeterCallBackTimer {
    /**
     * 日志对象
     */
    private static final Logger log = Logger.getLogger(WebMeterCallBackTimer.class);
    /**
     * Logic调用支持
     */
    @Autowired
    private LogicServer logicServer;
    /**
     * 回调数据缓存队列（线程安全）
     */
    private static final Queue<JSONObject> callBackQueue = new ConcurrentLinkedQueue<>();

    /**
     * 线程池线程数量（默认：4）
     */
    private int threadCount = 4;

    public void setThreadCount(int threadCount) {
        this.threadCount = threadCount;
    }

    /**
     * 将数据存入缓存队列
     *
     * @param msg 数据JSON
     */
    public static void add(JSONObject msg) {
        callBackQueue.add(msg);
    }

    /**
     * 执行业务
     */
    public void doBusiness() {
        //启动线程池
        ExecutorService service = Executors.newFixedThreadPool(threadCount);
        //循环消费缓存队列中的数据
        for (int i = 0; i <= threadCount; i++) {
            service.execute(new Thread(() -> {
                JSONObject data = callBackQueue.poll();
                if (data != null) {
                    //从回调数据中获取logic名并调用
                    String logicName = data.getString("logicName");
                    runLogic(logicName, data);
                }
            }));
        }
        //关闭线程池，否则会导致内存溢出
        service.shutdown();
    }

    private void runLogic(String logicName, JSONObject data) {
        try {
            logicServer.run(logicName, data);
        } catch (Exception e) {
            log.error("【" + logicName + "】出现异常：", e);
        }
    }
}
