package com.aote.webmeter.tools.iot;

import com.af.plugins.DateTools;
import com.af.plugins.HttpConnectionPoolUtil;
import com.af.plugins.xq.JsonTools2;
import com.aote.webmeter.enums.RestCodeEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import com.aote.webmeter.enums.NotifyTypeEnum;
import com.aote.webmeter.enums.TelecomApiEnum;
import com.iotplatform.client.NorthApiClient;
import com.iotplatform.client.NorthApiException;
import com.iotplatform.client.dto.*;
import com.iotplatform.client.invokeapi.SignalDelivery;
import com.iotplatform.utils.JsonUtil;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

/**
 * 电信IOT平台指令发送工具
 * @author Mr.river
 */
public class SignalDeliveryTools {
	private static final Logger LOGGER = Logger.getLogger(SignalDeliveryTools.class);

	/**
	 * 错误标识
	 */
	private static final String ERROR_PARAMS_KEY = "error_code";

	/**
	 * 查询命令状态
	 * @param deviceId 设备号
	 * @return 请求结果
	 */
	public static JSONArray getCommandStatus(String deviceId){
		SignalDelivery sd = new SignalDelivery();
		sd.setNorthApiClient(NorthApiClientTools.getNorthApiClient());
		QueryDeviceCommandInDTO2 acid = new QueryDeviceCommandInDTO2();
		acid.setDeviceId(deviceId);
		acid.setPageNo(0);
		acid.setPageSize(150);
		acid.setStartTime(DateTools.getDiffDate("DATE","-2","yyyyMMdd'T'000000'Z'"));
		QueryDeviceCommandOutDTO2 quid;
		try {
			quid = sd.queryDeviceCommand(acid,AuthenticationTools.getAuthToken());
		} catch (NorthApiException e) {
			LOGGER.error("IOT接口调用错误！，信息：",e);
			return null;
		}
        JSONArray result = new JSONArray();
        for(DeviceCommandRespV4 command : quid.getData()){
            JSONObject item = new JSONObject();
            item.put("commandId",command.getCommandId());
            item.put("status",command.getStatus());
            result.put(item);
        }
        return result;
	}

	/**
	 * 发送命令请求
	 * @param serviceId 服务ID
	 * @param method 命令名称
	 * @param deviceId 设备ID
	 * @param jo 命令参数JSON
	 * @param expireTime 等待时间
	 * @return 请求结果JSON
	 */
	public static JSONObject postCommand(String serviceId, String method, String deviceId, JSONObject jo,Integer expireTime) {
		SignalDelivery sd = new SignalDelivery();
        sd.setNorthApiClient(NorthApiClientTools.getNorthApiClient());
		PostDeviceCommandInDTO2 paid = new PostDeviceCommandInDTO2();
        paid.setDeviceId(deviceId);
		CommandDTOV4 aldo = new CommandDTOV4();

		aldo.setServiceId(serviceId);
		aldo.setMethod(method);
		try {
			aldo.setParas(JsonUtil.convertObject2ObjectNode(jo.toString()));
		} catch (Exception e) {
			e.printStackTrace();
		}
		paid.setCommand(aldo);
		if(expireTime != null){
			paid.setExpireTime(expireTime);
		}
		paid.setCallbackUrl(WebMeterInfo.getString("telecomCallBackUrl"));
		PostDeviceCommandOutDTO2 pecos;
		long begin = System.currentTimeMillis();
		try {
			pecos = sd.postDeviceCommand(paid, WebMeterInfo.getString("telecomAPIKey"), AuthenticationTools.getAuthToken());
			long end = System.currentTimeMillis();
			LOGGER.warn("电信请求接口耗时："+(end-begin)+"ms");
		} catch (NorthApiException e) {
			LOGGER.error("IOT接口调用错误！，信息：",e);
			long end = System.currentTimeMillis();
			LOGGER.warn("电信请求接口耗时："+(end-begin)+"ms");
			return null;
		}
        JSONObject object = new JSONObject();
        object.put("commandId", pecos.getCommandId());
        object.put("appId", pecos.getAppId());
        object.put("deviceId", pecos.getDeviceId());
        object.put("callbackUrl", pecos.getCallbackUrl());
        object.put("status", pecos.getStatus());
        object.put("result", pecos.getResult());
        return object;
	}
	public static JSONObject postCommand(String serviceId, String method, String deviceId, JSONObject jo) {
		return postCommand(serviceId,method,deviceId,jo,0);
	}


	/**
	 * 发送命令请求（自主API）
	 * @param serviceId 服务ID
	 * @param method 命令名称
	 * @param deviceId 设备ID
	 * @param jo 命令参数JSON
	 * @return 请求结果JSON
	 */
	public static JSONObject myPostCommand(String serviceId, String method, String deviceId, JSONObject jo,Integer expireTime) throws IOException {
		String accessToken = AuthenticationTools.getAuthToken();

		String appId = WebMeterInfo.getString("telecomAPIKey");
		String callbackUrl = WebMeterInfo.getString("telecomCallBackUrl");

		Integer maxRetransmit = 3;

		Map<String, Object> paramCommand = new HashMap<>(3);
		paramCommand.put("serviceId", serviceId);
		paramCommand.put("method", method);
		paramCommand.put("paras", jo);

		if(expireTime == null){
			expireTime = 0;
		}

		JSONObject paramCreateDeviceCommand = new JSONObject();
		paramCreateDeviceCommand.put("deviceId", deviceId);
		paramCreateDeviceCommand.put("command", paramCommand);
		paramCreateDeviceCommand.put("callbackUrl", callbackUrl);
		paramCreateDeviceCommand.put("expireTime", expireTime);
		paramCreateDeviceCommand.put("maxRetransmit", maxRetransmit);

		JSONObject header = new JSONObject();
		header.put("app_key", appId);
		header.put("Authorization", "Bearer" + " " + accessToken);
		header.put("Content-Type","application/json");

		String responseBody = HttpConnectionPoolUtil.request(
				TelecomApiEnum.SEND_INSTRUCT_URL.getValue(),
				paramCreateDeviceCommand.toString(),
				header.toString(),
				new HttpPost(),
				AuthenticationTools.getHttpClient()
		);
		Map<String,Object> result = JsonTools2.jsonToMap(new JSONObject(responseBody));
		if(result.get(ERROR_PARAMS_KEY) != null){
			throw new RuntimeException(responseBody);
		} else {
			return new JSONObject().put("commandId",result.get("commandId"));
		}
	}
	public static JSONObject myPostCommand(String serviceId, String method, String deviceId, JSONObject jo) throws IOException {
		return myPostCommand(serviceId,method,deviceId,jo,0);
	}

	/**
	 * 发送透传命令请求（即时）
	 * @param method 命令名称
	 * @param deviceId 设备ID
	 * @param base64String base64编码字符串
	 * @return 请求结果JSON
	 */
	public static JSONObject postCommand(String method,String deviceId,String base64String) {
		return postCommand(method,deviceId,base64String,0);
	}

	public static JSONObject myPostCommand(String method,String deviceId,String base64String) throws IOException {
		return myPostCommand(method,deviceId,base64String,0);
	}

	/**
	 * 发送透传命令请求
	 * @param method 命令名称
	 * @param deviceId 设备ID
	 * @param base64String base64编码字符串
	 * @param expireTime 等待时间
	 * @return 请求结果JSON
	 */
	public static JSONObject postCommand(String method,String deviceId,String base64String, Integer expireTime) {
		JSONObject object = new JSONObject();
		object.put("rawData", base64String);
		return postCommand("RawData", method, deviceId, object,expireTime);
	}

	public static JSONObject myPostCommand(String method,String deviceId,String base64String, Integer expireTime) throws IOException {
		JSONObject object = new JSONObject();
		object.put("rawData", base64String);
		return myPostCommand("RawData", method, deviceId, object,expireTime);
	}

	/**
	 * 发送订阅请求
	 * @param notifyTypeEnum  订阅类型
	 * @param callBackUrl 回调地址
	 * @return 请求结果JSON
	 */
	public static JSONObject subscribe(NotifyTypeEnum notifyTypeEnum, String callBackUrl) {
		String accessToken = AuthenticationTools.getAuthToken();
		NorthApiClient northApiClient = NorthApiClientTools.getNorthApiClient();

		Map<String, String> header = new HashMap<>(2);
		header.put("app_key", WebMeterInfo.getString("telecomAPIKey"));
		header.put("Authorization", "Bearer " + accessToken);
		JSONObject bodyParams = new JSONObject();
		bodyParams.put("notifyType", notifyTypeEnum.getValue());
		bodyParams.put("callbackUrl", callBackUrl);
		try {
			return northApiClient.invokeAPI(
					TelecomApiEnum.SUBSCRIPTIONS_URL.getValue(),
					RestCodeEnum.POST.getValue(),
					null,
					bodyParams.toString(),
					header, null,
					null,
					ContentType.APPLICATION_JSON,
					null,
					JSONObject.class
			);
		} catch (NorthApiException e) {
			LOGGER.error("IOT接口调用错误！，信息：",e);
			return null;
		}
	}

	/**
	 * 删除所有订阅
	 * @return 请求结果JSON
	 */
	public static JSONObject deleteAllSubscribe() {
		String accessToken = AuthenticationTools.getAuthToken();
		NorthApiClient northApiClient = NorthApiClientTools.getNorthApiClient();

		Map<String, String> header = new HashMap<>(16);
		header.put("app_key", WebMeterInfo.getString("telecomAPIKey"));
		header.put("Authorization", "Bearer " + accessToken);
		try {
			return northApiClient.invokeAPI(
					TelecomApiEnum.SUBSCRIPTIONS_URL.getValue(),
					RestCodeEnum.DELETE.getValue(),
					null,
					null,
					header,
					null,
					null,
					ContentType.APPLICATION_JSON,
					null,
					JSONObject.class
			);
		} catch (NorthApiException e) {
			LOGGER.error("IOT接口调用错误！，信息：",e);
			return null;
		}
	}
}
