package com.aote.webmeter.server.tcp;

import com.af.plugins.ConvertTools;
import com.aote.logic.LogicServer;
import com.aote.webmeter.enums.MsgTypeEnum;
import org.apache.log4j.Logger;
import org.json.JSONObject;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.ServerSocketChannel;
import java.nio.channels.SocketChannel;
import java.util.Iterator;

/**
 * TCP服务
 * @author Mrriver
 */
public class TcpServer extends Thread {
    private Selector selector;
    /**
     * TCP端口
     */
    private int port;
    /**
     * logic对象
     */
    private LogicServer service;
    /**
     * logic名称
     */
    private String logicName;
    /**
     * TCP接收消息类型
     */
    private MsgTypeEnum msgReceiveType;
    /**
     * TCP发送消息类型
     */
    private MsgTypeEnum msgSendType;
    private static Logger log = Logger.getLogger(TcpServer.class);

    public TcpServer(String logicName, int port, LogicServer service, MsgTypeEnum msgReceiveType, MsgTypeEnum msgSendType) {
        this.logicName = logicName;
        this.port = port;
        this.service = service;
        this.msgReceiveType = msgReceiveType;
        this.msgSendType = msgSendType;
    }

    /**
     * 获得一个ServerSocket通道，并对该通道做一些初始化的工作  
     */
    private void initServer() {
        try {
            // 获得一个通道管理器
            this.selector = Selector.open();
            // 打开通道
            ServerSocketChannel serverChannel = ServerSocketChannel.open();
            // 设置通道为非阻塞
            serverChannel.configureBlocking(false);
            // 将该通道对应的ServerSocket绑定ip和端口
            InetSocketAddress address = new InetSocketAddress(port);
            serverChannel.bind(address);
            // 注册事件
            serverChannel.register(selector, SelectionKey.OP_ACCEPT);
            // 监听事件
            listen();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    private void listen() throws Exception {
        System.out.println("服务端【" + port + "】启动成功！");
        // 轮询访问selector  
        while (selector.select() > 0) {
            //当注册的事件到达时，方法返回；否则,该方法会一直阻塞  
            selector.select();
            //获得selector中选中的项的迭代器，选中的项为注册的事件  
            Iterator<SelectionKey> ite = this.selector.selectedKeys().iterator();
            while (ite.hasNext()) {
                SelectionKey key = ite.next();
                // 删除已选的key,以防重复处理
                ite.remove();
                if (key.isAcceptable()) {
                    ServerSocketChannel server = (ServerSocketChannel) key.channel();
                    SocketChannel sc = server.accept();
                    if (sc == null) {
                        continue;
                    }
                    sc.configureBlocking(false);
                    sc.register(selector, SelectionKey.OP_READ);
                    System.out.println(sc.getRemoteAddress() + " 发起连接");
                } else if (key.isReadable()) {
                    // 获得了可读的事件
                    read(key);
                }
            }
        }
    }

    private void read(SelectionKey key) throws IOException {
        // 服务器可读取消息:得到事件发生的Socket通道  
        SocketChannel channel = (SocketChannel) key.channel();
        // 设置成非阻塞
        channel.configureBlocking(false);
        // 创建读取的缓冲区  
        ByteBuffer buffer = ByteBuffer.allocate(1024);
        int recvCount;
        try {
            recvCount = channel.read(buffer);
        } catch (IOException e) {
            log.debug(channel.getRemoteAddress() + "断开连接...");
            channel.close();
            log.debug("清除缓存...");
            buffer.clear();
            return;
        }
        if (recvCount > 0) {
            byte[] data = buffer.array();
            String msg;
            switch (msgReceiveType) {
                case NORMAL:
                default:
                    msg = ConvertTools.bytesToStr(data);
                    break;
                case HEX:
                    msg = ConvertTools.byteToHexStr(data);
                    break;
            }
            //去除空格，尾部0，转大写
            msg = msg.trim().replaceAll("0*$","").toUpperCase();
            log.debug("服务端收到信息："+msg);
            JSONObject params = new JSONObject();
            params.put("data", new JSONObject().put("value", msg));
            try {
                String result = String.valueOf(service.run(logicName, params));
                log.debug("服务端返回信息：" + result);
                byte[] sendMsgBytes;
                ByteBuffer outBuffer;
                switch (msgSendType) {
                    case NORMAL:
                    default:
                        sendMsgBytes = result.getBytes();
                        break;
                    case BASE64:
                        sendMsgBytes = ConvertTools.base64Decode(result.getBytes());
                        break;
                    case HEX:
                        sendMsgBytes = ConvertTools.hexStrToByte(result);
                        break;
                    case HEX_BASE64:
                        sendMsgBytes = ConvertTools.base64Decode(ConvertTools.hexStrToByte(result));
                        break;
                }
                outBuffer = ByteBuffer.wrap(sendMsgBytes);
                channel.write(outBuffer);
            } catch (Exception e) {
                log.error("出现异常：", e);
                log.debug("服务端主动断开连接...");
                channel.close();
            }
        } else {
            log.debug("客户端请求断开连接...");
            channel.close();
        }
        log.debug("清除缓存...");
        buffer.clear();
    }

    @Override
    public void run() {
        initServer();
    }
}
