package com.aote.webmeter.tools.iot.utils;

import com.aote.redis.RedisService;
import com.aote.redis.RedisUtil;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.util.concurrent.atomic.AtomicReference;

/**
 * 通用鉴权实现类
 * <p>
 * 带分布式锁实现，避免重复获取token
 *
 * @param <T> 鉴权结果类型
 */
public abstract class AuthTools<T> {

    private static final Logger LOGGER = LoggerFactory.getLogger(AuthTools.class);
    private static final String PUBLIC_ORG_NAME = "Public";

    /**
     * 鉴权入口
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @return 鉴权结果
     */
    public final T authorization(String orgId) throws InterruptedException {
        RedisService redisService = RedisUtil.getInstance();
        if (orgId == null) {
            orgId = PUBLIC_ORG_NAME;
        }
        String key = getTokenCacheKey(orgId);
        T tokenObj = (T) redisService.get(key);
        if (tokenObj != null) {
            LOGGER.info("{}:获取现有token[{}]的结果:{}", getName(), key, tokenObj);
            if (tokenObj instanceof String) {
                return (T)(new JSONObject(tokenObj.toString()));
            }
            return tokenObj;
        } else {
            String finalOrgId = orgId;
            AtomicReference<T> response = new AtomicReference<>();
            redisService.lock(key, () -> {
                // 再次尝试从缓存获取，避免集群多实例重复请求获取auth
                T newTokenObj = (T) redisService.get(key);
                if (newTokenObj != null) {
                    LOGGER.info("{}:获取现有token[{}]的结果:{}", getName(), key, newTokenObj);
                    if (newTokenObj instanceof String) {
                        response.set((T)(new JSONObject(newTokenObj.toString())));
                    } else {
                        response.set(newTokenObj);
                    }
                    return;
                }
                LOGGER.info("{}平台：{}:进行鉴权操作", getName(), key);
                AuthResult result;
                try {
                    result = getToken(finalOrgId);
                } catch (UnsupportedEncodingException e) {
                    throw new RuntimeException(e);
                }
                LOGGER.info(result.value.getClass().getName());
                // 缓存到redis中
                redisService.set(key, result.value, result.expiresIn);
                response.set(result.value);
            });
            return response.get();
        }
    }

    public final T authorization() throws InterruptedException {
        return authorization(null);
    }

    /**
     * 强制刷新token
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     */
    public final synchronized void forceRefreshToken(String orgId) throws InterruptedException {
        RedisService redisService = RedisUtil.getInstance();
        if (orgId == null) {
            orgId = PUBLIC_ORG_NAME;
        }
        String key = getTokenCacheKey(orgId);
        redisService.lock(key, () -> {
            redisService.delete(key);
        });
    }

    /**
     * 鉴权实现
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @return 鉴权结果
     */
    protected abstract AuthResult getToken(String orgId) throws UnsupportedEncodingException;

    /**
     * 鉴权缓存的redisKey
     *
     * @param orgId 组织机构ID，为null时代表公共组织
     * @return key值
     */
    private String getTokenCacheKey(String orgId) {
        return getName() + "Token@" + orgId;
    }

    /**
     * 区分鉴权模块的唯一标识名
     *
     * @return 标识名
     */
    protected abstract String getName();

    /**
     * 鉴权结果对象
     */
    protected final class AuthResult {
        /**
         * 鉴权结果
         */
        private final T value;
        /**
         * 过期时间
         */
        private final Integer expiresIn;

        public AuthResult(T value, Integer expiresIn) {
            this.value = value;
            this.expiresIn = expiresIn;
        }
    }
}
