package com.aote.webmeter.tools.iot.onenet;

import com.af.plugins.RestAsyncTools;
import com.af.plugins.RestTools;
import com.aote.webmeter.enums.WebmeterPropertiesIOTEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 移动Onenet-IOT平台 API调用工具
 *
 * @see <a href="https://open.iot.10086.cn/doc/v5/fuse">中国移动物联网开放平台</a>
 * @see <a href="https://open.iot.10086.cn/doc/v5/fuse/detail/1465">新版接口</>
 * @see <a href="https://open.iot.10086.cn/doc/v5/fuse/detail/1485">OneNET平台API升级情况说明</>
 * @author Mr.river
 */
@Component
public class OnenetApiService2 {
    private static final Logger LOGGER = LogManager.getLogger(OnenetApiService2.class);

    @Autowired
    private OnenetApiAuthTools onenetApiAuthTools;

    /**
     * 创建设备
     * @param imei imei码
     * @param deviceNo 设备号
     * @return 响应结果
     */
    public JSONObject createDevice(String imei, String deviceNo) throws InterruptedException {
        String path = OnenetApi2.DEVICE_URL + "/create";
        String productId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_PRODUCT_ID);
        JSONObject headers = onenetApiAuthTools.authorization(productId);
        // 设置请求参数
        JSONObject params = new JSONObject();
        params.put("product_id", productId);
        params.put("device_name", deviceNo);
        params.put("imei", imei);
        params.put("imsi", "000000000000000");
        //发送请求
        JSONObject result = new JSONObject(RestTools.post(path, params.toString(), headers.toString()));
        LOGGER.info("api: {}，header: {}，body: {}，response: {}，type: sync", path, headers, params, result);
        int errno = result.getInt("code");
        String msg;
        if(errno == 0){
            msg = result.getJSONObject("data").get("did").toString();
        } else {
            msg = "注册IOT设备失败(" + errno + ")："+result.getString("msg");
            errno = -1;
        }
        return new JSONObject().put("code", errno).put("msg", msg);
    }

    /**
     * 删除设备
     *
     * @param imei 设备IMEI
     * @return 响应结果
     */
    public JSONObject deleteDevice(String imei) throws InterruptedException {
        String path = OnenetApi2.DEVICE_URL + "/delete";
        String productId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_PRODUCT_ID);
        JSONObject headers = onenetApiAuthTools.authorization(productId);
        // 设置请求参数
        JSONObject params = new JSONObject();
        params.put("product_id", productId);
        params.put("imei", imei);
        //发送请求
        JSONObject result = new JSONObject(RestTools.post(path, params.toString(), headers.toString()));
        LOGGER.info("api: {}，header: {}，body: {}，response: {}，type: sync", path, headers, params, result);
        int errno = result.getInt("code");
        String msg;
        if(errno == 0){
            msg = result.getJSONObject("data").get("did").toString();
        } else {
            if(errno == 10410){
                errno = 0;
                msg = "尝试删除时已不存在";
            } else {
                errno = -1;
                msg = "删除IOT设备失败(" + errno + ")："+result.getString("msg");
            }
        }
        return new JSONObject().put("code", errno).put("msg", msg);
    }

    /**
     * 写入设备资源
     * @param imei 设备imei
     * @param mode 写入模式（1或2）
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public JSONObject writeDeviceRes(String imei, String mode, String timeout, String content) throws InterruptedException {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("mode", mode == null ? 1 : mode);
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi2.RESOURCE_URL, params);
        String productId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_PRODUCT_ID);
        //设置请求头
        JSONObject headers = onenetApiAuthTools.authorization(productId);
        //发送请求
        JSONObject body = new JSONObject();
        JSONArray data = new JSONArray();
        JSONObject dataItem = new JSONObject();
        dataItem.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        dataItem.put("type", 1);
        dataItem.put("val", content);
        data.put(dataItem);
        body.put("data",data);
        RestAsyncTools.post(path, body.toString(), headers.toString());
        LOGGER.info("api: {}，header: {}，body: {}，type: async", path, headers, body);
        return new JSONObject().put("code",0);
    }

    /**
     * 命令下发
     * @param imei 设备imei
     * @param timeout 超时时间，默认25秒
     * @param content 请求内容(实际请求参数)
     * @return 响应结果
     */
    public JSONObject sendCommand(String imei, String timeout, String content) throws InterruptedException {
        JSONObject params = new JSONObject();
        params.put("imei", imei);
        params.put("obj_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_ID));
        params.put("obj_inst_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_OBJECT_INST_ID));
        params.put("res_id", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_RES_ID));
        params.put("timeout", timeout == null ? "25" : timeout);
        //指定请求路径
        String path = RestTools.formatURL(OnenetApi2.COMMAND_URL, params);
        String productId = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.ONE_NET_PRODUCT_ID);
        //设置请求头
        JSONObject headers = onenetApiAuthTools.authorization(productId);
        //发送请求
        JSONObject body = new JSONObject();
        body.put("args", content);
        RestAsyncTools.post(path, body.toString(), headers.toString());
        LOGGER.info("api: {}，header: {}，body: {}", path, headers, content);
        return new JSONObject().put("code", 0);
    }
}
