package com.aote.sensitive;

import com.aote.config.SystemConfig;
import com.aote.util.SubstitutionCipherUtil;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.util.HashMap;
import java.util.Set;

/**
 * 敏感数据拦截器
 * 作为EntityServer与敏感数据处理系统的唯一接口
 * <p>
 * 设计原则：
 * - 最小侵入：EntityServer只需调用一个方法
 * - 可插拔：可以通过配置完全禁用敏感数据处理
 * - 职责单一：只负责拦截和就地加密处理
 */
@Component
public class SensitiveDataInterceptor {

    static Logger log = Logger.getLogger(SensitiveDataInterceptor.class);
    private static final String PROTECTED_ROW_LOG = "[PROTECTED]";

    /**
     * 在数据保存前处理所有敏感字段：统一在原字段就地写入替代密码密文
     *
     * @param entityName 实体名称（业务表名）
     * @param row        数据行对象（字段会在此就地修改）
     * @param metaData   实体元数据（包含 sensitiveFields 配置）
     */
    public void beforeSave(String entityName, JSONObject row, HashMap<String, Object> metaData) {

        if (!SystemConfig.DataSecurity.getMaskSensitiveData()) {
            return;
        }

        if (row == null || row.length() == 0 || metaData == null) {
            return;
        }

        // 获取敏感字段配置
        @SuppressWarnings("unchecked")
        HashMap<String, SensitiveFieldConfigLoader.SensitiveFieldConfig> sensitiveConfigs =
                (HashMap<String, SensitiveFieldConfigLoader.SensitiveFieldConfig>) metaData.get("sensitiveFields");

        if (sensitiveConfigs == null || sensitiveConfigs.isEmpty()) {
            return;
        }

        try {
            // 就地处理：检测 -> 加密回写/跳过
            applySensitiveFieldEncryption(entityName, row, sensitiveConfigs.keySet());
        } catch (Exception e) {
            handleProcessingError(entityName, row, e);
        }
    }

    /**
     * 统一错误处理方法
     */
    private void handleProcessingError(String entityName, JSONObject row, Exception e) {
        String rowInfo = (row != null) ? PROTECTED_ROW_LOG : "null";
        log.error("敏感数据处理失败，将跳过处理: " + entityName + ", row=" + rowInfo, e);
    }

    /**
     * 处理单个字段错误
     */
    private void handleFieldError(String fieldName, Exception e) {
        log.warn("敏感字段加密异常: field=" + fieldName + ", error=" + e.getMessage());
        if (log.isDebugEnabled()) {
            log.debug("敏感字段加密详细错误:", e);
        }
    }

    /**
     * 在原字段就地处理敏感数据：已脱敏则跳过，已是密文则跳过，否则加密写回
     */
    private void applySensitiveFieldEncryption(String entityName, JSONObject row, Set<String> sensitiveFields) {
        log.debug("处理敏感字段(就地加密): " + entityName + ", 字段数=" + sensitiveFields.size());

        for (String fieldName : sensitiveFields) {
            if (row.opt(fieldName) == null) continue;
            processSingleField(fieldName, row);
        }
    }

    /**
     * 处理单个敏感字段
     */
    private void processSingleField(String fieldName, JSONObject row) {
        Object val = row.opt(fieldName);
        if (!(val instanceof String)) {
            return;
        }

        String strVal = (String) val;
        if (strVal.isEmpty()) {
            return;
        }

        // 1) 已脱敏输入（包含 *）=> 跳过该字段的更新，避免覆盖真实密文
        if (isMaskedValue(strVal)) {
            row.remove(fieldName);
            log.debug("检测到已脱敏输入，跳过字段更新: " + fieldName);
            return;
        }

        // 2) 已是替代密码密文（严格判定）=> 跳过加密，保留原值
        if (SubstitutionCipherUtil.isStrictCipher(strVal)) {
            log.debug("检测到已是替代密码密文，跳过加密: " + fieldName);
            return;
        }

        // 3) 明文 => 加密后回写到原字段
        try {
            String cipher = SubstitutionCipherUtil.encrypt(strVal);
            if (cipher != null) {
                row.put(fieldName, cipher);
                log.debug("敏感字段已加密写回: " + fieldName + 
                         " (原文长度: " + strVal.length() + ", 密文长度: " + cipher.length() + ")");
            }
        } catch (Exception ex) {
            handleFieldError(fieldName, ex);
        }
    }

    /**
     * 判定是否为已脱敏输入（包含 *）
     */
    private boolean isMaskedValue(String value) {
        return value != null && value.contains("*");
    }

}