package com.aote.sensitive;

import org.apache.log4j.Logger;
import org.hibernate.cfg.Configuration;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.MetaAttribute;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;

import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 敏感字段配置加载器（简化版）
 * 仅读取两个 Meta：sensitive 与 mask-strategy
 * - 返回按实体维度的配置映射（字段名 -> 配置对象，仅包含字段名与策略）
 * - 同时维护全局 Map（字段别名/列名小写 -> 脱敏策略）供查询出库使用
 */
public class SensitiveFieldConfigLoader {
    
    static Logger log = Logger.getLogger(SensitiveFieldConfigLoader.class);
    
    private static final ConcurrentHashMap<String, String> SENSITIVE_MASK_MAP = new ConcurrentHashMap<>();

    /**
     * 内部简化配置类：仅存字段名与策略
     */
    public static class SensitiveFieldConfig {
        private String fieldName;
        private String maskStrategy;

        public SensitiveFieldConfig() {}

        public String getFieldName() {
            return fieldName;
        }

        public void setFieldName(String fieldName) {
            this.fieldName = fieldName;
        }

        public String getMaskStrategy() {
            return maskStrategy;
        }

        public void setMaskStrategy(String maskStrategy) {
            this.maskStrategy = maskStrategy;
        }

        public String getMaskStrategyOrDefault() {
            return maskStrategy != null ? maskStrategy : "default";
        }
    }

    /**
     * 获取全局敏感字段 -> 脱敏策略 的映射
     */
    public static Map<String, String> getSensitiveMaskMap() {
        return SENSITIVE_MASK_MAP;
    }

    /**
     * 通过 Hibernate Configuration 获取指定实体的敏感字段配置（简化）
     * @param configuration Hibernate配置对象
     * @param entityName 实体名称
     * @return 敏感字段配置映射 (字段名 -> 配置对象，仅 fieldName 与 maskStrategy)
     */
    public static HashMap<String, SensitiveFieldConfig> loadSensitiveFieldConfigs(
            Configuration configuration, String entityName) {

        HashMap<String, SensitiveFieldConfig> configs = new HashMap<>();
        try {
            PersistentClass pc = configuration.getClassMapping(entityName);
            if (pc == null) {
                log.debug("未找到实体映射: " + entityName);
                return configs;
            }
            
            @SuppressWarnings("unchecked")
            Iterator<Property> props = pc.getPropertyIterator();
            while (props.hasNext()) {
                Property prop = props.next();
                if (prop == null || prop.getMetaAttributes() == null) continue;

                if (!prop.getMetaAttributes().containsKey("sensitive")) continue;
                Map<String, String> meta = extractMetaAttributes(prop);
                if (!"true".equalsIgnoreCase(meta.get("sensitive"))) continue;

                String fieldName = prop.getName();
                String strategy = meta.get("mask-strategy");
                if (strategy == null || strategy.trim().isEmpty()) {
                    // 使用 SensitiveDataEnhancer 的推断策略，确保与出库脱敏逻辑一致
                    strategy = SensitiveDataEnhancer.inferStrategy(fieldName);
                }

                SensitiveFieldConfig cfg = new SensitiveFieldConfig();
                cfg.setFieldName(fieldName);
                cfg.setMaskStrategy(strategy);
                configs.put(fieldName, cfg);

                // 同步写入全局 Map：属性名与首列列名均加入，键统一小写
                addToGlobalMaskMap(fieldName, strategy, prop);
            }
            
            if (!configs.isEmpty()) {
                log.info("敏感字段配置加载完成: 实体=" + entityName + ", 数量=" + configs.size());
            }
        } catch (Exception e) {
            log.error("加载敏感字段配置失败: " + entityName, e);
        }
        
        return configs;
    }

    /**
     * 提取元属性（只获取 sensitive 和 mask-strategy）
     */
    private static Map<String, String> extractMetaAttributes(Property prop) {
        Map<String, String> metaAttributes = new HashMap<>();
        try {
            for (Object o : prop.getMetaAttributes().entrySet()) {
                @SuppressWarnings("unchecked")
                Map.Entry<String, MetaAttribute> entry = (Map.Entry<String, MetaAttribute>) o;
                String key = entry.getKey();
                MetaAttribute metaAttr = entry.getValue();
                if (metaAttr.getValues() != null && !metaAttr.getValues().isEmpty()) {
                    String value = metaAttr.getValues().get(0).toString();
                    metaAttributes.put(key, value);
                }
            }
        } catch (Exception e) {
            log.error("提取meta属性失败: " + prop.getName(), e);
        }
        return metaAttributes;
    }

    /**
     * 将字段配置添加到全局掩码映射
     * 包括字段名和列名（都转换为小写以支持不同的查询别名形式）
     */
    private static void addToGlobalMaskMap(String fieldName, String strategy, Property prop) {
        SENSITIVE_MASK_MAP.put(fieldName.toLowerCase(), strategy);
        
        try {
            Iterator<?> colIt = prop.getColumnIterator();
            if (colIt != null && colIt.hasNext()) {
                Object sel = colIt.next();
                if (sel instanceof Column) {
                    String colName = ((Column) sel).getName();
                    if (colName != null) {
                        SENSITIVE_MASK_MAP.put(colName.toLowerCase(), strategy);
                    }
                }
            }
        } catch (Exception e) {
            log.debug("添加列名到全局掩码映射失败: " + fieldName, e);
        }
    }
}