package com.aote.sql;

import com.aote.ThreadResource;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * 权限工具类：检查用户是否有查看敏感字段明文的权限
 * <p>权限匹配规则（优先级从高到低）：</p>
 * <ol>
 *   <li>*:*:* - 全局所有权限（通过所有检查）</li>
 *   <li>sensitive:view:all - 查看所有敏感字段明文</li>
 *   <li>sensitive:view:* - 通配符匹配所有敏感字段</li>
 *   <li>sensitive:view:{fieldName} - 查看特定字段明文（如 sensitive:view:phone）</li>
 * </ol>
 * <p>权限格式为 ":"分隔的字符串，存储在用户对象的 permissions JSONArray 中</p>
 */
public final class AuthUtil {
    private static final Logger log = LoggerFactory.getLogger(AuthUtil.class);
    
    // 权限前缀常量
    private static final String PERMISSION_WILDCARD_ALL = "*:*:*";
    private static final String PERMISSION_SENSITIVE_VIEW_ALL = "sensitive:view:all";
    private static final String PERMISSION_SENSITIVE_VIEW_WILDCARD = "sensitive:view:*";
    private static final String PERMISSION_SENSITIVE_PREFIX = "sensitive:view";

    private AuthUtil() {
        throw new UnsupportedOperationException("Utility class");
    }

    /**
     * 判断是否有查看指定敏感字段明文的权限
     * 
     * <p>权限检查流程（按优先级）：</p>
     * <ol>
     *   <li>检查全局权限：*:*:*</li>
     *   <li>检查所有敏感字段权限：sensitive:view:all</li>
     *   <li>检查通配符权限：sensitive:view:*</li>
     *   <li>检查特定字段权限：sensitive:view:{fieldName}</li>
     * </ol>
     * 
     * @param fieldName 字段名（如：phone、idcard、email）
     * @return true = 有权限查看明文，false = 无权限（返回脱敏值）
     */
    public static boolean hasSensitiveViewPermission(String fieldName) {
        try {
            // 1. 检查全局权限：*:*:*
            if (hasPermission(PERMISSION_WILDCARD_ALL)) {
                log.debug("用户具有全局权限 {}", PERMISSION_WILDCARD_ALL);
                return true;
            }

            // 2. 检查所有敏感字段权限：sensitive:view:all
            if (hasPermission(PERMISSION_SENSITIVE_VIEW_ALL)) {
                log.debug("用户具有查看所有敏感字段权限 {}", PERMISSION_SENSITIVE_VIEW_ALL);
                return true;
            }

            // 3. 检查通配符权限：sensitive:view:*
            if (hasPermission(PERMISSION_SENSITIVE_VIEW_WILDCARD)) {
                log.debug("用户具有通配符权限 {}，可查看字段 {}", PERMISSION_SENSITIVE_VIEW_WILDCARD, fieldName);
                return true;
            }

            // 4. 检查特定字段权限：sensitive:view:{fieldName}
            String specificPermission = PERMISSION_SENSITIVE_PREFIX + ":" + fieldName;
            if (hasPermission(specificPermission)) {
                log.debug("用户具有特定字段权限 {}", specificPermission);
                return true;
            }

            // 无任何匹配的权限
            log.debug("用户无权限查看字段 {}", fieldName);
            return false;

        } catch (Exception e) {
            // 权限检查异常时，默认返回 false（安全优先）
            log.error("权限检查异常: fieldName={}", fieldName, e);
            return false;
        }
    }

    /**
     * 检查用户是否拥有指定权限
     * <p>内部处理：</p>
     * <ol>
     *   <li>从 ThreadLocal 获取当前登陆用户</li>
     *   <li>从用户对象的 permissions JSONArray 中获取权限列表</li>
     *   <li>在权限列表中查找匹配的权限</li>
     * </ol>
     * 
     * @param permission 要检查的权限字符串
     * @return true 如果用户拥有该权限
     */
    private static boolean hasPermission(String permission) {
        // 第一步：获取当前用户信息
        JSONObject loginUser = ThreadResource.LoginUser.get();
        if (loginUser == null) {
            log.debug("当前线程无登陆用户信息");
            return false;
        }

        // 第二步：获取用户的权限列表
        JSONArray permissions = loginUser.optJSONArray("permissions");
        if (permissions == null || permissions.length() == 0) {
            log.debug("用户无权限信息");
            return false;
        }

        // 第三步：在权限列表中查找该权限
        for (int i = 0; i < permissions.length(); i++) {
            String userPermission = permissions.optString(i, "");
            if (permission.equals(userPermission)) {
                return true;
            }
        }

        return false;
    }
}


