package com.aote.filter;
import com.aote.config.Config;
import com.aote.util.JsonTransfer;
import com.aote.v4.compatibility.liuli.config.service.LiuLiConfigService;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.ContextLoader;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
/**
 * @Author: JoeCdy
 * @Description:
 * @Date: Created in 10:35 2023/4/17
 */
public class DesensitizationTool {
    public static final String KEY = "7xjgtQc4M8FOXikU7JkwcUI0wKhYkREt";
    static Logger log = Logger.getLogger(DesensitizationTool.class);

    public static void main(String[] args) throws Exception {
//        String phone = "13976247686";
//        String phoneEncrypt = phoneEncrypt(phone);
//        String idNumber = "32062319690415861X";
//        String idNumberEncrypt = idNumberEncrypt(idNumber);
//        System.out.println("原文" + phone + ",密文:" + phoneEncrypt);
//        System.out.println("原文" + idNumber + ",密文:" + idNumberEncrypt);
//        System.out.println(RSAUtil.aesDecrypt("lgbnCwSb5Nu6P2AGfW8WDg==", KEY));
//        System.out.println(checkID("370102197406202117"));
//        System.out.println(checkPhone("18292552417"));
        System.out.println(
                maskUserName("杨东生")
        );
    }

    /**
     * 对查询结果进行脱敏处理
     * 处理 |-...-| 格式的加密数据，根据用户权限进行解密或脱敏
     * @param list 查询结果集
     * @param token 用户token
     */
    public static JSONArray processEncryption(String token, List<Map<String, Object>> list) {
        JSONArray array = new JSONArray();
        // 检查是否需要脱敏处理
        if (!Config.isDesensitization()) {
            for (Map<String, Object> map : list) {
                array.put(JsonTransfer.MapToJson(map));
            }
            return array;
        }
        log.debug("权限判断"+ token);
        // 检查授权
        boolean isAuthorized = Config.isAuthorization(token);
        LiuLiConfigService liuLiConfigService = getSpringBean();
        // 从琉璃拿配置获取要脱敏的列
        JSONObject desensitizationConfig = liuLiConfigService.get("desensitizationConfig");
        log.debug("获取到的配置" + desensitizationConfig);
        JSONArray field = desensitizationConfig.optJSONArray("field", new JSONArray());
        Set<String> fieldSet = new HashSet<>();
        for (int i = 0; i < field.length(); i++) {
            fieldSet.add(field.getString(i));
        }
        log.debug("权限判断"+isAuthorized);
        for (Map<String, Object> map : list) {
            JSONObject json = JsonTransfer.MapToJson(map);
            // 遍历JSON对象的所有字段
            for (String key : json.keySet()) {
                Object value = json.get(key);
                if (value instanceof String) {
                    String strValue = (String) value;
                    // 检查是否包含 |-...-| 格式的加密数据
                    if (strValue.contains("|-") && strValue.contains("-|")) {
                        String processedValue = processEncryptedField(strValue, isAuthorized);
                        if (!isAuthorized) {
                            if (fieldSet.contains(key)) {
                                json.put(key, "f_user_phone".equals(key) ? maskPhoneNumber(processedValue) : maskUserName(processedValue));
                            } else {
                                json.put(key, "f_user_name".equals(key) ? maskUserName(processedValue) : processedValue);
                            }
                        } else {
                            json.put(key, processedValue);
                        }
                    } else {
                        if (!isAuthorized) {
                            if (fieldSet.contains(key)) {
                                json.put(key, "f_user_phone".equals(key) ? maskPhoneNumber(strValue) : maskUserName(strValue));
                            } else {
                                json.put(key, "f_user_name".equals(key) ? maskUserName(strValue) : strValue);
                            }
                        } else {
                            json.put(key, strValue);
                        }
                    }
                }
            }
            array.put(json);
        }
        return array;
    }

    /**
     * 处理单个加密字段
     * @param value 字段值
     * @param isAuthorized 是否有权限
     * @return 处理后的值
     */
    private static String processEncryptedField(String value, boolean isAuthorized) {
        // 使用正则表达式匹配 |-...-| 格式
        Pattern pattern = Pattern.compile("\\|-.*?-\\|");
        Matcher matcher = pattern.matcher(value);
        StringBuffer sb = new StringBuffer(value.length());

        while (matcher.find()) {
            String match = matcher.group(0);
            String replacement = match; // 默认不替换

            if (isAuthorized) {
                // 有权限：解密
                try {
                    replacement = RSAUtil.aesDecrypt(match.replace("|-", "").replace("-|", ""), KEY);
                } catch (Exception e) {
                    log.debug("解密异常", e);
                    // 解密失败时保持原值
                    replacement = match;
                }
            } else {
                // 无权限：脱敏处理，提取 |- 之前和 -| 之后的部分，中间用****替换
                String beforeEncrypt = match.substring(0, match.indexOf("|-"));
                String afterEncrypt = match.substring(match.indexOf("-|") + 2);
                replacement = beforeEncrypt + "****" + afterEncrypt;
            }
            matcher.appendReplacement(sb, replacement);
        }
        matcher.appendTail(sb);
        return sb.toString();
    }

    /**
     * 对用户姓名进行脱敏处理
     * @param name 原始姓名
     * @return 脱敏后的姓名
     */
    private static String maskUserName(String name) {
        if (name == null || name.isEmpty()) {
            return name;
        }

        int length = name.length();
        StringBuilder masked = new StringBuilder();

        switch (length) {
            case 1:
                // 单字符直接返回
                return name;
            case 2:
                // 两位：把姓改成*
                masked.append("*").append(name.charAt(1));
                break;
            case 3:
                // 三位：把中间改成*
                masked.append(name.charAt(0)).append("*").append(name.charAt(2));
                break;
            case 4:
                // 四位：把中间两位改成*
                masked.append(name.charAt(0)).append("**").append(name.charAt(3));
                break;
            case 5:
                // 五位：把中间三位改成*
                masked.append(name.charAt(0)).append("***").append(name.charAt(4));
                break;
            case 6:
                // 六位：把中间四位改成*
                masked.append(name.charAt(0)).append("****").append(name.charAt(5));
                break;
            case 7:
                // 七位：把中间五位改成*
                masked.append(name.charAt(0)).append("*****").append(name.charAt(6));
                break;
            case 8:
                // 八位：把中间六位改成*
                masked.append(name.charAt(0)).append("******").append(name.charAt(7));
                break;
            default:
                // 超过8位的情况，保留首尾，中间用*代替
                masked.append(name.charAt(0));
                for (int i = 1; i < length - 1; i++) {
                    masked.append("*");
                }
                masked.append(name.charAt(length - 1));
                break;
        }

        return masked.toString();
    }
    /**
     * 电话号码脱敏处理
     * 规则：
     * - 手机号：保留前3位和后4位，中间用****替换（如138****5678）
     * - 座机号：保留完整区号和分隔符，本地号隐藏中间部分，保留后4位（长度不变）
     * @param phone 原始电话号码
     * @return 脱敏后的电话号码，若输入无效则返回原字符串
     */
    private static String maskPhoneNumber(String phone) {
        if (phone == null || phone.trim().isEmpty() || !validatePhoneNumber(phone)) {
            return phone; // 无效号码不脱敏
        }

        String pureNum = phone.replaceAll("[^0-9]", "");

        // 手机号脱敏：前3位 + **** + 后4位
        if (pureNum.matches("^1[3-9]\\d{9}$")) {
            return pureNum.substring(0, 3) + "****" + pureNum.substring(7);
        }

        // 座机号脱敏：保留完整区号 + 分隔符 + 隐藏中间部分 + 后4位（保持原长度）
        Pattern landlinePattern = Pattern.compile("(0\\d{2,3})([\\-\\s]?)(\\d{7,8})");
        Matcher matcher = landlinePattern.matcher(phone);
        if (matcher.matches()) {
            String areaCode = matcher.group(1);
            String separator = matcher.group(2);
            String localNum = matcher.group(3);

            // 计算本地号需要隐藏的位数（保留后4位）
            int hideLength = Math.max(localNum.length() - 4, 0);
            // 生成对应数量的*（兼容低版本Java）
            StringBuilder hideStrBuilder = new StringBuilder();
            for (int i = 0; i < hideLength; i++) {
                hideStrBuilder.append("*");
            }
            String hideStr = hideStrBuilder.toString();
            // 拼接脱敏后的本地号（保持原长度）
            String maskedLocal = hideStr + localNum.substring(Math.max(localNum.length() - 4, 0));
            return areaCode + separator + maskedLocal;
        }

        // 处理纯数字座机号（无分隔符）
        if (pureNum.startsWith("0")) {
            // 提取区号（3-4位）：10位为3位区号+7位本地号；11位为4位区号+7位本地号
            int areaCodeLen = (pureNum.length() == 10) ? 3 : 4;
            String areaCode = pureNum.substring(0, areaCodeLen);
            String localNum = pureNum.substring(areaCodeLen);

            // 计算本地号需要隐藏的位数（保留后4位）
            int hideLength = Math.max(localNum.length() - 4, 0);
            // 生成对应数量的*（兼容低版本Java）
            StringBuilder hideStrBuilder = new StringBuilder();
            for (int i = 0; i < hideLength; i++) {
                hideStrBuilder.append("*");
            }
            String hideStr = hideStrBuilder.toString();
            // 拼接脱敏后的本地号（保持原长度）
            String maskedLocal = hideStr + localNum.substring(Math.max(localNum.length() - 4, 0));
            return areaCode + maskedLocal;
        }

        return phone;
    }

    private static LiuLiConfigService getSpringBean() {
        ApplicationContext wac =  ContextLoader.getCurrentWebApplicationContext();
        return (LiuLiConfigService) wac.getBean("liuLiConfigService");
    }

    /**
     * 验证电话号码是否符合中国手机号规则
     * @param phone 待验证的电话号码（支持带国家码、分隔符等格式）
     * @return 符合规则返回 true，否则返回 false
     */
    private static boolean validatePhoneNumber(String phone) {
        if (phone == null || phone.trim().isEmpty()) {
            return false;
        }

        // 移除所有非数字字符（用于快速判断手机号）
        String pureNum = phone.replaceAll("[^0-9]", "");

        // 验证手机号：11位数字，1开头，第二位3-9
        if (pureNum.matches("^1[3-9]\\d{9}$")) {
            return true;
        }

        // 验证座机号（支持带分隔符和纯数字）
        // 格式：0 + 3-4位区号 + (可选分隔符) + 7-8位本地号
        Pattern landlinePattern = Pattern.compile("^0\\d{2,3}([\\-\\s]?)\\d{7,8}$");
        Matcher landlineMatcher = landlinePattern.matcher(phone);
        if (landlineMatcher.matches()) {
            return true;
        }

        // 特殊情况：纯数字座机号（无分隔符）
        if (pureNum.length() >= 10 && pureNum.length() <= 11
                && pureNum.startsWith("0")
                && pureNum.substring(1).matches("\\d{9,10}")) {
            return true;
        }

        return false;
    }
}