package com.aote.sensitive;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.crypto.Cipher;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;
import java.util.regex.Pattern;

/**
 * 敏感数据增强处理工具类 包含所有敏感数据处理功能：
 * 1. 哈希优化：通过 SHA256 哈希对比避免重复处理
 * 2. 智能分类：自动识别数据状态，跳过已处理数据
 * 3. 同表模式：支持在主表中直接存储哈希和加密字段
 * 4. 完整脱敏：提供各种脱敏策略（姓名、手机、身份证、地址等）
 * 5. AES加密：独立的加密解密功能
 * 6. 性能优化：大幅提升敏感数据处理性能
 */
public class SensitiveDataEnhancer {

    private static final Logger log = LoggerFactory.getLogger(SensitiveDataEnhancer.class);

    // AES 加密密钥
    public static final String AES_KEY = "7xjgtQc4M8FOXikU7JkwcUI0wKhYkREt";

    // 加密算法
    private static final String AES_ALGORITHM = "AES/ECB/PKCS5Padding";

    // 预编译正则表达式，提升性能
    private static final Pattern DIGITS_ONLY_PATTERN = Pattern.compile("\\D");
    private static final Pattern PHONE_FORMAT_PATTERN = Pattern.compile("\\d+");
    private static final Pattern ID_NUMBER_PATTERN = Pattern.compile("\\d{6}\\d{8}[\\dXx]");

    // 策略匹配关键词集合（预编译，避免重复创建）
    private static final Set<String> NAME_STRATEGIES = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            "name", "姓名", "用户名", "username", "realname", "nickname"
    )));
    private static final Set<String> PHONE_STRATEGIES = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            "phone", "mobile", "手机", "电话", "tel", "telephone"
    )));
    private static final Set<String> ADDRESS_STRATEGIES = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            "address", "addr", "地址", "住址", "location"
    )));
    private static final Set<String> ID_STRATEGIES = Collections.unmodifiableSet(new HashSet<>(Arrays.asList(
            "idnumber", "idcard", "身份证", "证件号", "cardnumber"
    )));

    /**
     * 重复字符串方法，替代 String.repeat()
     */
    private static String repeatString(String str, int count) {
        if (count <= 0) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < count; i++) {
            sb.append(str);
        }
        return sb.toString();
    }

    /**
     * 应用脱敏策略
     * 使用传统 switch 语句和自动策略推断
     */
    public static String applyMaskingStrategy(String fieldName, String originalValue, String strategy) {
        if (originalValue == null || originalValue.isEmpty()) {
            return originalValue;
        }

        // 策略推断：如果未指定策略，根据字段名自动推断
        String actualStrategy = strategy != null ? strategy : inferStrategy(fieldName);

        try {
            switch (actualStrategy.toLowerCase()) {
                case "name":
                    return maskUserName(originalValue);
                case "phone":
                case "mobile":
                    return maskPhone(originalValue);
                case "address":
                case "addr":
                    return maskAddress(originalValue);
                case "idnumber":
                case "idcard":
                    return maskIdNumber(originalValue);
                default:
                    return maskDefault(originalValue);
            }
        } catch (Exception e) {
            log.error("脱敏处理失败: {}, 策略: {}", fieldName, actualStrategy, e);
            return "**MASKED**";
        }
    }

    /**
     * 根据字段名推断脱敏策略
     * 使用预编译的策略集合，避免重复创建对象
     */
    public static String inferStrategy(String fieldName) {
        String lower = fieldName.toLowerCase();

        if (NAME_STRATEGIES.stream().anyMatch(lower::contains)) {
            return "name";
        }

        if (PHONE_STRATEGIES.stream().anyMatch(lower::contains)) {
            return "phone";
        }

        if (ADDRESS_STRATEGIES.stream().anyMatch(lower::contains)) {
            return "address";
        }

        if (ID_STRATEGIES.stream().anyMatch(lower::contains)) {
            return "idnumber";
        }

        return "default";
    }

    /**
     * 姓名脱敏
     * 使用传统 switch 语句
     */
    private static String maskUserName(String name) {
        if (name == null || name.isEmpty()) {
            return name;
        }

        int length = name.length();
        switch (length) {
            case 1:
                return name;
            case 2:
                return "*" + name.charAt(1);
            default:
                return name.charAt(0) + repeatString("*", length - 2) + name.charAt(length - 1);
        }
    }

    /**
     * 手机号脱敏
     * 支持多种格式：手机号、座机号(带/不带区号)，使用预编译正则表达式和传统 switch 语句
     */
    private static String maskPhone(String phone) {
        if (phone == null || phone.isEmpty()) {
            return phone;
        }

        // 使用预编译正则表达式提取数字，性能更优
        String digitsOnly = DIGITS_ONLY_PATTERN.matcher(phone).replaceAll("");
        int length = digitsOnly.length();

        if (length < 4) {
            return phone; // 太短，不脱敏
        }

        // 根据数字长度使用不同的脱敏策略，使用预编译正则表达式替换
        switch (length) {
            case 4:
            case 5:
            case 6:
            case 7: {
                // 4-7位：保留前2位和后1位
                String prefix = digitsOnly.substring(0, 2);
                String suffix = digitsOnly.substring(length - 1);
                return PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + repeatString("*", length - 3) + suffix);
            }
            case 8:
            case 9:
            case 10:
            case 11: {
                // 8-11位：保留前3位和后2位
                String prefix = digitsOnly.substring(0, 3);
                String suffix = digitsOnly.substring(length - 2);
                return PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + "****" + suffix);
            }
            default: {
                // 超过11位：保留前3位和后4位
                String prefix = digitsOnly.substring(0, 3);
                String suffix = digitsOnly.substring(length - 4);
                return PHONE_FORMAT_PATTERN.matcher(phone).replaceAll(prefix + "****" + suffix);
            }
        }
    }

    /**
     * 身份证脱敏
     * 使用传统 switch 语句、条件表达式和预编译正则表达式验证
     */
    private static String maskIdNumber(String idNumber) {
        if (idNumber == null || idNumber.isEmpty()) {
            return idNumber;
        }

        // 基本长度验证 + 可选的格式验证
        switch (idNumber.length()) {
            case 15:
                // 15位老身份证
                return idNumber.substring(0, 6) + repeatString("*", 6) + idNumber.substring(12);
            case 18: {
                // 18位新身份证
                // 可选：使用预编译正则表达式进行更严格的格式验证
                if (ID_NUMBER_PATTERN.matcher(idNumber).matches()) {
                    return idNumber.substring(0, 6) + repeatString("*", 8) + idNumber.substring(14);
                } else {
                    // 格式不匹配，采用通用脱敏策略
                    return idNumber.substring(0, 6) + repeatString("*", 8) + idNumber.substring(14);
                }
            }
            default:
                return idNumber.length() >= 10
                        ? idNumber.substring(0, 6) + repeatString("*", Math.min(8, idNumber.length() - 10)) +
                        idNumber.substring(Math.max(6, idNumber.length() - 4))
                        : idNumber; // 太短，不脱敏
        }
    }

    /**
     * 地址脱敏
     * 使用传统 switch 语句按长度分类处理
     */
    private static String maskAddress(String address) {
        if (address == null || address.isEmpty()) {
            return address;
        }

        int length = address.length();
        switch (length) {
            case 1:
            case 2:
            case 3:
                return address; // 太短，不脱敏
            case 4:
            case 5:
            case 6:
                return address.substring(0, 2) + "***";
            case 7:
            case 8:
            case 9:
            case 10:
                return address.substring(0, 3) + "***" + address.substring(length - 2);
            default:
                return address.substring(0, 4) + "***" + address.substring(length - 3);
        }
    }

    /**
     * 默认脱敏策略
     * 使用传统 switch 语句
     */
    private static String maskDefault(String value) {
        if (value == null || value.isEmpty()) {
            return value;
        }

        int length = value.length();
        switch (length) {
            case 1:
            case 2:
                return repeatString("*", length); // 短字符串全部脱敏
            case 3:
            case 4:
            case 5:
            case 6:
                return value.charAt(0) + repeatString("*", length - 2) + value.charAt(length - 1);
            default:
                return value.substring(0, 2) + repeatString("*", length - 4) + value.substring(length - 2);
        }
    }

    /**
     * 检查字符串是否为null或空白
     *
     * @param input 输入字符串
     * @return true 如果为null或空白，否则 false
     */
    private static boolean isNullOrBlank(String input) {
        return input == null || input.trim().isEmpty();
    }

    /**
     * AES 加密
     * 移除不必要的KeyGenerator，直接使用SecretKeySpec
     */
    public static String aesEncrypt(String input) throws Exception {
        if (isNullOrBlank(input)) {
            return null;
        }

        // 直接创建密钥规范，无需KeyGenerator（性能优化）
        SecretKeySpec keySpec = new SecretKeySpec(AES_KEY.getBytes(StandardCharsets.UTF_8), "AES");
        Cipher cipher = Cipher.getInstance(AES_ALGORITHM);
        cipher.init(Cipher.ENCRYPT_MODE, keySpec);

        byte[] encryptedBytes = cipher.doFinal(input.getBytes(StandardCharsets.UTF_8));
        return java.util.Base64.getEncoder().encodeToString(encryptedBytes);
    }

    /**
     * AES 解密
     * 移除不必要的KeyGenerator，直接使用SecretKeySpec
     */
    public static String aesDecrypt(String encrypted) throws Exception {
        if (isNullOrBlank(encrypted)) {
            return null;
        }

        try {
            // 直接创建密钥规范，无需KeyGenerator（性能优化）
            SecretKeySpec keySpec = new SecretKeySpec(AES_KEY.getBytes(StandardCharsets.UTF_8), "AES");
            Cipher cipher = Cipher.getInstance(AES_ALGORITHM);
            cipher.init(Cipher.DECRYPT_MODE, keySpec);

            byte[] encryptedBytes = java.util.Base64.getDecoder().decode(encrypted);
            byte[] decryptedBytes = cipher.doFinal(encryptedBytes);
            return new String(decryptedBytes, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.error("AES解密失败，可能是数据格式错误或密钥不匹配", e);
            throw e; // 重新抛出异常，让调用方处理
        }
    }

}
