package com.aote.redis;

import com.aote.config.SystemConfig;
import com.aote.rs.mapper.WebException;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.redisson.Redisson;
import org.redisson.api.*;
import org.redisson.client.codec.Codec;
import org.redisson.client.codec.StringCodec;
import org.redisson.codec.FstCodec;
import org.redisson.codec.JsonJacksonCodec;
import org.redisson.codec.KryoCodec;
import org.redisson.codec.SerializationCodec;
import org.redisson.config.Config;
import org.redisson.config.SingleServerConfig;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Redis操作工具
 *
 * @author Mr.river
 */
public class RedisUtil {
    private static volatile RedisUtil redisUtil;
    /**
     * redis操作类
     */
    private static final RedissonClient REDISSON_CLIENT;
    private static final Logger log = Logger.getLogger(RedisUtil.class);

    private static final Codec codec;

    public static RedisUtil getInstance(){
        if(redisUtil == null) {
            synchronized (RedisUtil.class) {
                if (redisUtil == null && REDISSON_CLIENT != null) {
                    redisUtil = new RedisUtil();
                }
            }
        }
        return redisUtil;
    }

    private RedisUtil(){

    }

    static {
        if(SystemConfig.Redis.hasHostName()) {
            Config config = new Config();
            if(SystemConfig.Redis.hasCodecType()){
                switch (SystemConfig.Redis.getCodecType()) {
                    case "Jackson":
                        codec = new JsonJacksonCodec();
                        break;
                    case "Serialization":
                        codec = new SerializationCodec();
                        break;
                    case "String":
                        codec = new StringCodec();
                        break;
                    case "Fst":
                        codec = new FstCodec();
                        break;
                    case "Kryo":
                        codec = new KryoCodec();
                        break;
                    default:
                        throw new RuntimeException("不支持的序列化方式：" + SystemConfig.Redis.getCodecType());
                }
            } else {
                codec = new JsonJacksonCodec();
            }
            config.setCodec(codec);
            SingleServerConfig singleServerConfig = config.useSingleServer();
            singleServerConfig.setAddress("redis://"+ SystemConfig.Redis.getHostName());
            if (SystemConfig.Redis.hasPassword()) {
                singleServerConfig.setPassword(SystemConfig.Redis.getPassword());
            }
            REDISSON_CLIENT = Redisson.create(config);
            log.info("***redis实例已加载完成***");
        } else {
            codec = null;
            REDISSON_CLIENT = null;
            log.warn("***redis实例未配置***");
        }
    }

    public <T> void set(String key, T value) {
        set(key,value,-1);
    }

    public <T> void set(String key, T value, long timeoutSec) {
        RBucket<T> rBucket = REDISSON_CLIENT.getBucket(key, codec);
        if(timeoutSec > 0){
            rBucket.set(value,timeoutSec, TimeUnit.SECONDS);
        } else {
            rBucket.set(value);
        }
    }

    public <T> void setList(String key, List<T> value) {
        setList(key, value, -1);
    }

    public <T> void setList(String key, List<T> value, long timeoutSec) {
        RList<T> rList = REDISSON_CLIENT.getList(key, codec);
        rList.addAll(value);
        if(timeoutSec > 0){
            rList.expire(timeoutSec, TimeUnit.SECONDS);
        }
    }

    public <T> void setListKey(String key, T value) {
        List<T> list = REDISSON_CLIENT.getList(key, codec);
        list.add(value);
    }

    public void deleteList(String key) {
        REDISSON_CLIENT.getList(key, codec).delete();
    }

    public void clearList(String key) {
        REDISSON_CLIENT.getList(key, codec).clear();
    }

    public void deleteListKeyByIndex(String key, Integer index) {
        REDISSON_CLIENT.getList(key, codec).remove(index);
    }

    public void deleteListKeyByObject(String key, Object object) {
        REDISSON_CLIENT.getList(key, codec).remove(object);
    }



    public void setHash(String hashKey, JSONObject value) {
        setHash(hashKey, value.toMap());
    }

    public <T> void setHash(String hashKey, Map<String, T> value) {
        setHash(hashKey,value,-1);
    }

    public <T> void setHash(String hashKey, Map<String, T> value, long timeoutSec) {
        RMap<String, T> rMap = REDISSON_CLIENT.getMap(hashKey, codec);
        rMap.putAll(value);
        if(timeoutSec > 0){
            rMap.expire(timeoutSec, TimeUnit.SECONDS);
        }
    }

    public Map<Object, Object> getHash(String hashKey) {
        return REDISSON_CLIENT.getMap(hashKey, codec).readAllMap();
    }

    public <T> T getHash(String hashKey, String key) {
        Map<String, T> map = REDISSON_CLIENT.getMap(hashKey, codec);
        return map.get(key);
    }

    public <T> void setHashKey(String hashKey, String key, T value) {
        Map<String, T> map = REDISSON_CLIENT.getMap(hashKey, codec);
        map.put(key, value);
    }

    public void deleteHash(String hashKey) {
        REDISSON_CLIENT.getMap(hashKey, codec).delete();
    }

    public void clearHash(String hashKey) {
        REDISSON_CLIENT.getMap(hashKey, codec).clear();
    }

    public void deleteHashKey(String hashKey, String key) {
        REDISSON_CLIENT.getMap(hashKey, codec).remove(key);
    }

    public Boolean hasHashKey(String hashKey, String key) {
        return REDISSON_CLIENT.getMap(hashKey, codec).containsKey(key);
    }

    public void lock(String key, Runnable runnable) {
        RLock lock = REDISSON_CLIENT.getLock("LOCK-"+key);
        boolean isLock;
        try{
            isLock = lock.tryLock(10,20,TimeUnit.SECONDS);
            if(isLock){
                runnable.success();
            } else {
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            log.error("分布式锁"+lock.getName()+"获取失败");
            e.printStackTrace();
        } finally {
            if(lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }


    public <T> T get(String key) {
        RBucket<T> bucket = REDISSON_CLIENT.getBucket(key, codec);
        return bucket.get();
    }

    public void delete(String key) {
        REDISSON_CLIENT.getBucket(key, codec).delete();
    }

    public void deleteList(Iterable<Object> keys) {
        keys.forEach((item) -> delete(String.valueOf(item)));
    }

    public void deleteList(JSONObject keys) {
        deleteList(new ArrayList<>(keys.keySet()));
    }

    @FunctionalInterface
    public interface Runnable {
        void success();
    }
}
