package com.aote.config;

import com.aote.exception.FileNotFoundException;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Optional;

/**
 * 系统配置
 *
 * @author Mr.river
 */
public class SystemConfig {
    private static final Logger LOGGER = Logger.getLogger(SystemConfig.class);

    public static final String CONFIG_PATH = "systemConfig.json";

    public static final String OLD_CONFIG_PATH = "config.json";

    private static volatile boolean isLoaded = false;

    /**
     * 获取配置文件并加载
     */
    private static void load() {
        //获取配置文件
        if(!isLoaded){
            isLoaded = true;
            JSONObject params = new JSONObject();
            if (SystemConfig.class.getClassLoader().getResource(OLD_CONFIG_PATH) != null) {
                JSONObject oldParams = JsonTools.readJsonFile(OLD_CONFIG_PATH);
                JSONObject system = new JSONObject();
                if (oldParams.has("proxy")) {
                    system.put("proxyUrl", oldParams.getString("proxy"));
                }
                if (oldParams.has("ldap")) {
                    system.put("ldapUrl", oldParams.getString("ldap"));
                }
                params.put("system", system);
                JSONObject redis = new JSONObject();
                if (oldParams.has("logicRegister")) {
                    JSONObject logicRegister = oldParams.getJSONObject("logicRegister");
                    redis.put("registerUrl", logicRegister.getString("url"));
                    JSONObject redisObj = logicRegister.getJSONObject("redis");
                    redis.put("hostName", redisObj.getString("host") + ":" + redisObj.getString("port"));
                    if (redisObj.has("redispwd")) {
                        redis.put("password", redisObj.getString("redispwd"));
                    }
                }
                params.put("redis", redis);
                params.put("persistenceLog", new JSONObject());
                params.put("resources", new JSONObject());
            }
            // 合并配置, 新配置会覆盖旧配置
            if (SystemConfig.class.getClassLoader().getResource(CONFIG_PATH) != null) {
                params = JsonTools.addJSON(params, JsonTools.readJsonFile(CONFIG_PATH));
            }
            if (params.isEmpty()) {
                throw new FileNotFoundException("系统配置文件" + CONFIG_PATH + "不存在", CONFIG_PATH);
            }
            /* 系统配置 **/
            JSONObject system = params.getJSONObject("system");
            System.webAppName = system.optString("webAppName");
            if(system.has("proxyUrl")){
                System.proxyUrl = system.getString("proxyUrl");
            }
            if(system.has("ldapUrl")){
                System.ldapUrl = system.getString("ldapUrl");
            }
            if(system.has("check")){
                System.check = system.getString("check");
            }
            /* 日志持久化配置 **/
            JSONObject persistenceLog = params.optJSONObject("persistenceLog");
            PersistenceLog.isEnabled = persistenceLog.optBoolean("isEnabled", false);
            if (PersistenceLog.isEnabled) {
                LOGGER.info("异常日志持久化记录已开启，如需关闭，请将" + CONFIG_PATH + "中[persistenceLog:isEnabled]配置为false.");
            } else {
                LOGGER.warn("异常日志持久化记录已关闭，如需开启，请将" + CONFIG_PATH + "中[persistenceLog:isEnabled]配置为true.");
            }
            /* Redis配置 **/
            JSONObject redis = params.getJSONObject("redis");
            if (!redis.has("hostName")) {
                LOGGER.warn("Redis未配置，部分功能将无法正常使用");
            } else {
                Redis.hostName = redis.getString("hostName");
                if(redis.has("password")){
                    Redis.password = redis.getString("password");
                }
            }
            if (!redis.has("registerUrl")) {
                LOGGER.warn("业务redis注册地址未配置，部分功能将无法正常使用");
            } else {
                Redis.registerUrl = redis.getString("registerUrl");
            }
            /* 资源类配置 **/
            JSONObject resources = params.getJSONObject("resources");
            if (!resources.has("fileRootPath")) {
                LOGGER.warn(CONFIG_PATH + "中资源根目录参数[resources:fileRootPath]不存在，以按照缺省值[D:/" + System.webAppName + "Res]配置");
            }
            Resources.fileRootPath = resources.optString("fileRootPath", "D:/" + System.webAppName + "Res");
            //  如果有扩展目录配置，则追加
            if (resources.has("extraPaths")) {
                Resources.extraPaths = resources.getJSONObject("extraPaths");
            }
            /* 实体映射配置 **/
            SubClass.subClassArray = params.optJSONArray("subclass");
            /* 数据安全配置 **/
            JSONObject dataSecurity = params.optJSONObject("dataSecurity");
            if (Optional.ofNullable(dataSecurity).isPresent()) {
                DataSecurity.enableDataEncryption = dataSecurity.optBoolean("enableDataEncryption", false);
                DataSecurity.aesKey = dataSecurity.optString("aesKey");
                DataSecurity.injectionCheck = dataSecurity.optBoolean("injectionCheck", false);
            } else {
                DataSecurity.enableDataEncryption = false;
                DataSecurity.injectionCheck = false;
            }
        }
    }

    /**
     * 系统
     */
    public static class System {

        private static String webAppName;

        private static String proxyUrl;

        private static String ldapUrl;
        private static String check;

        static {
            SystemConfig.load();
        }

        public static String getWebAppName() {
            return webAppName;
        }

        public static String getProxyUrl() {
            return proxyUrl;
        }

        public static boolean hasProxyUrl() {
            return proxyUrl != null;
        }
        public static String getCheck() {
            return check;
        }

        public static boolean hasCheck() {
            return check != null;
        }
        public static String getLdapUrl() {
            return ldapUrl;
        }

        public static boolean hasLdapUrl() {
            return ldapUrl == null;
        }
    }

    /**
     * 日志持久化
     */
    public static class PersistenceLog {
        private static Boolean isEnabled;

        static {
            SystemConfig.load();
        }

        public static Boolean getEnabled() {
            return isEnabled;
        }
    }

    /**
     * Redis
     */
    public static class Redis {
        private static String hostName;
        private static String password;

        private static String registerUrl;

        static {
            SystemConfig.load();
        }

        public static String getHostName() {
            return hostName;
        }
        public static boolean hasHostName() {
            return hostName != null;
        }
        public static String getPassword() {
            return password;
        }
        public static boolean hasPassword() {
            return password != null;
        }
        public static String getRegisterUrl() {
            return registerUrl;
        }
        public static boolean hasRegisterUrl() {
            return registerUrl != null;
        }
    }

    /**
     * 资源
     */
    public static class Resources {
        private static String fileRootPath;
        private static JSONObject extraPaths;

        static {
            SystemConfig.load();
        }

        public static String getFileRootPath() {
            return fileRootPath;
        }

        public static String getExtraPath(String key) {
            return extraPaths.getString(key);
        }
    }

    /**
     * 实体映射
     */
    public static class SubClass {
        private static JSONArray subClassArray;

        static {
            SystemConfig.load();
        }

        public static JSONArray getSubClassArray() {
            return subClassArray;
        }
        public static boolean hasSubClassArray() {
            return subClassArray != null;
        }
    }

    /**
     * 数据安全
     */
    public static class DataSecurity {
        /**
         * 启用数据加密
         */
        private static Boolean enableDataEncryption;
        /**
         * AES密钥
         */
        private static String aesKey;
        /**
         * SQL注入检查
         */
        private static Boolean injectionCheck;

        static {
            SystemConfig.load();
        }
        public static boolean getEnableDataEncryption() { return enableDataEncryption; }
        public static String getAesKey() { return aesKey; }
        public static boolean hasAesKey() { return aesKey != null; }
        public static boolean getInjectionCheck() { return injectionCheck; }
    }
}
