package com.aote.rs;

import com.aote.logic.LogicServer;

import com.aote.transaction.SessionPool;
import com.aote.util.*;
import com.aote.util.other.AESUtil;
import javassist.NotFoundException;
import net.sf.json.JSON;
import net.sf.json.xml.XMLSerializer;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import javax.inject.Singleton;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;

@Path("logic")
@Singleton
@Component
public class LogicService {
	static Logger log = Logger.getLogger(LogicService.class);
	static Logger reviewLog = Logger.getLogger("reviewLog");

	@Autowired
	private SessionPool sessionPool;

	@Autowired
	private LogicServer logicServer;

	public static final ThreadLocal<String> reviewKey = new ThreadLocal<>();
	public static final Map<String, JSONArray> reviewCache = new HashMap<>();

	/**
	 * 清理Logic源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearLogicCache(){
		ResourceHelper.clearLogicCache();
		return "ok";
	}

	/**
	 * 清理所有缓存
	 * @return ok
	 */
	@GET
	@Path("/clearall")
	public String clearAllCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		return "ok";
	}

	/**
	 * 执行业务逻辑(GET方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) throws Exception {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		String values = params.toString();
		return (Response) runMain(logicName, values);
	}

	/**
	 * 执行业务逻辑(POST方式)
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("{logic}")
	public Response xtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) throws Exception {
		values = getValues(values,request);
		return (Response) runMain(logicName, values);
	}

	/**
	 * 异步执行业务逻辑(GET方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param request   HttpServletRequest对象
	 * @return 执行结果
	 */
	@GET
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, @Context HttpServletRequest request) {
		Map paramMap = request.getParameterMap();
		JSONObject params = new JSONObject(paramMap);
		try {
			runAsync(logicName, params.toString());
		} catch (Exception ignore){}
		//移动平台推送特殊处理
		if(params.has("msg")){
			return Response.ok(params.getJSONArray("msg").get(0)).build();
		}
		return Response.ok().build();
	}

	/**
	 * 异步执行业务逻辑(POST方式)
	 * https://stackoverflow.com/questions/31401640/spring-xml-equivalent-of-enableasync
	 *
	 * @param logicName 业务逻辑名称
	 * @param values JSON资源字符串
	 * @return 执行结果
	 */
	@POST
	@Path("async/{logic}")
	public Response asyncXtSave(@PathParam("logic") String logicName, String values, @Context HttpServletRequest request) {
		try {
			runAsync(logicName, values);
		} catch (Exception ignore){}
		return Response.ok().build();
	}

	private String getValues(String values,HttpServletRequest request){
		if (!values.startsWith("{")){
			JSONObject paramJson = new JSONObject();
			paramJson.put("str",values);
			//加入请求头数据
			Enumeration headerNames = request.getHeaderNames();
			if(headerNames != null){
				JSONObject header = new JSONObject();
				while (headerNames.hasMoreElements()){
					String temp = headerNames.nextElement().toString();
					header.put(temp, request.getHeader(temp));
				}
				if(header.has("request_id")){
					reviewKey.set(String.valueOf(header.get("request_id")));
				}
				paramJson.put("header",header);
			}
			values = paramJson.toString();
		}
		return values;
	}

	public String xtSave(String logicName, String values) throws Exception {
		return xtSave(logicName,values,null).getEntity().toString();
	}

	private Object runMain(String logicName, String values) throws Exception {
		//如果有action，执行action处理过程，返回sessionid
		JSONObject jsonObject = new JSONObject(values);
		if (jsonObject.has("action")){
			int id = judgeAction(jsonObject);
			return id + "";
		}
		String id = reviewKey.get();
		long begin = System.currentTimeMillis();
		JSONObject reviewObj = new JSONObject();
		if(id != null){
			reviewCache.put(id, new JSONArray());
		}
		try {
			String result = AESUtil.encryptData(jsonObject,
					(data) -> logicServer.runMain(logicName, data));
			Response.ResponseBuilder builder = Response.ok();
			if (result.startsWith("{")) {
				JSONObject paramJson = new JSONObject(result);
				//如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
				if (paramJson.has("header") && paramJson.has("content")) {
					Map<String, Object> paramHeader = paramJson.getJSONObject("header").toMap();

					for (Map.Entry<String, Object> stringObjectEntry : paramHeader.entrySet()) {
						builder.header(stringObjectEntry.getKey(), String.valueOf(stringObjectEntry.getValue()));
					}
					return builder.type("text/plain").entity(String.valueOf(paramJson.get("content"))).build();
				}
			}
			reviewObj.put("status", 0);
			return builder.entity(result).build();
		} catch (Exception e){
			reviewObj.put("status",-1);
			throw e;
		} finally {
			if(id != null){
				long end = System.currentTimeMillis();
				reviewObj.put("requestId", id);
				reviewObj.put("bTime", begin);
				reviewObj.put("eTime", end);
				reviewObj.put("constTime", end - begin);
				reviewObj.put("type", "logic");
				reviewObj.put("items", reviewCache.get(id));
				reviewLog.debug(reviewObj);
				reviewKey.remove();
				reviewCache.remove(id);
			}
		}
	}

	@POST
	@Path("xml/{logic}")
	@Consumes({"application/xml"})
	public String xmlXtSave(@PathParam("logic") String logicName, @Context  HttpServletRequest request) throws Exception {
		ServletInputStream inputStream = request.getInputStream();
		StringBuilder content = new StringBuilder();
		byte[] b = new byte[1024];
		int lens;
		while ((lens = inputStream.read(b)) > 0) {
			content.append(new String(b, 0, lens));
		}
		// 内容
		String strcont = content.toString();
		JSON json;
		try {
			XMLSerializer xmlSerializer = new XMLSerializer();
			json = xmlSerializer.read(strcont);
		}catch (Exception e){
			throw new RuntimeException("传入的数据不是XML格式");
		}
		String jsonString = new JSONObject().put("data",json).toString();
		log.debug("调用XML接口，Logic："+logicName+"  参数："+json);
		return logicServer.runMain(logicName,jsonString);
	}

	//服务器启动时加载模块至redis
	@PostConstruct
	public void redisCacheLogic() {
		logicServer.redisCacheLogic();
	}

	/**
	 * 执行action中的动作
	 * @param jsonObject {action: 'actionname', id: 'sessionid'}
	 * @return 启动事务的id
	 * @throws NotFoundException
	 */
	private int judgeAction(JSONObject jsonObject)throws NotFoundException {
		int sessionid;
		//不为空则，获取action，并判断action请求为何种类型
		String actObj = (String)jsonObject.get("action");
		if("begin".equals(actObj)) {
			//如果为begin类型，openSession，并获取sessionid；
			sessionid = sessionPool.createSession();
			return sessionid;
		} else {
			//否则直接获取id
			sessionid = jsonObject.getInt("id");
		}
		return sessionid;
	}

	private void runAsync(String logicName, String values) throws Exception {
		logicServer.runAsync(logicName, values);
	}
}
