package com.aote.util.other;

import org.json.JSONArray;
import org.json.JSONObject;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Set;

/**
 * json工具类
 *
 * @author zt
 *
 */
public class JsonTools {

	/**
	 * 对参数进行验证
	 * @param obj  传入Json对象
	 * @param fields 传入一个对象
	 * @throws Exception 异常
	 */
	public static void validateField(JSONObject obj, Object fields) throws Exception {
		String errors = null;

		for (Object s : fields instanceof JSONObject ? new JSONObject(fields.toString()).keySet().toArray() : ((String)fields).split(",")) {
			if ("".equals(obj.get((String)s)) || (obj.get((String)s)) == null) {
				errors = (errors == null ? s : errors + s) + " ";
			}
		}
		if (errors != null) {
			throw new Exception(errors + "字段值不能为空！！！");
		}
	}

	/**
	 * 从jo中移除colsToBeRemoved中的列
	 * @param colsToBeRemoved json对象
	 * @param jo json对象
	 * @return 移除完返回jo对象
	 */
	public static JSONObject remove(JSONObject colsToBeRemoved, JSONObject jo) {
		Set<String> keySet1 = colsToBeRemoved.keySet();
		for (String s : keySet1) {
			jo.remove(s);
		}
		return jo;
	}

	/**
	 * 给一个JSONObject中添加对象
	 * @param obj  json对象
	 * @param addLabel 键
	 * @param addVale 值
	 * @return 添加完成后返回obj
	 */
	public static JSONObject add(JSONObject obj, String addLabel, Object addVale) {
		obj.put(addLabel, addVale);
		return obj;
	}

	/**
	 * 给一个JSONObject追加一个JSONObject中的所有值
	 *
	 * @param obj    被追加的JSONObject
	 * @param addObj 追加的JSONObject
	 * @return 追加完的JSONObject
	 */
	public static JSONObject addJSON(JSONObject obj, JSONObject addObj) {
		if (addObj == null) {
			throw new NullPointerException("追加的JSONObject不存在！");
		}
		Set<String> keys = addObj.keySet();
		for (String key : keys) {
			Object item = addObj.get(key);
			if(item instanceof JSONObject){
				JSONObject cloneItem = new JSONObject(((JSONObject) item).toMap());
				obj.put(key, cloneItem);
			} else if(item instanceof JSONArray){
				JSONArray cloneItem = new JSONArray(((JSONArray) item).toList());
				obj.put(key, cloneItem);
			} else {
				obj.put(key, item);
			}
		}
		return obj;
	}

	/**
	 * 判断JSONObject对象中是否包含key
	 * @param obj  json对象
	 * @param key  是否包含的key键
	 * @return boolean值的返回值
	 */
	public static boolean isContains(JSONObject obj, String key) {
		return obj.has(key);
	}

	/**
	 * JSONObject串删除id
	 * @param obj  json对象
	 * @param key obj中需要移除的键
	 * @return 返回obj
	 */
	public static JSONObject removeKey(JSONObject obj, String key) {
		obj.remove(key);
		return obj;
	}

	/**
	 * 获取json对象中的key
	 *
	 * @param obj
	 *
	 * @return key的数组
	 */
	public static JSONArray getJsonKeys(JSONObject obj) {
		JSONArray array = new JSONArray();
		if (obj != null) {
			for(String str: obj.keySet()) {
				array.put(str);
			}
		}
		return array;
	}

	/**
	 * JSONObject中key是否为空
	 * @param obj json对象
	 * @param key 需要查看的key
	 * @return 返回的boolean值
	 */
	public static boolean isNull(JSONObject obj, String key) {
		return obj.isNull(key);
	}

	/**
	 * JSONObject是否为空
	 * @param obj  查看的json对象
	 * @return 返回boolean
	 */
	public static boolean isNull(JSONObject obj) {
		return obj.isEmpty();
	}

	/**
	 * 将字符串转化为json
	 *
	 * @param str 要转为json的字符串
	 * @return 原生json
	 */
	public static JSONObject convertToJson(String str) {
		if (str == null || "".equals(str)) {
			return new JSONObject();
		}
		return new JSONObject(str);
	}

	/**
	 * 把字符串转换成JSON数组
	 * @param str 需要转换的String
	 * @return 转换完成后的json数组
	 */
	public static JSONArray parseArray(String str) {
		return new JSONArray(str);
	}

	/**
	 * 返回一个JSON数组实例
	 *
	 * @return JSON数组实例
	 */
	public static JSONArray getArray() {
		return new JSONArray();
	}

	/**
	 * 返回一个制定长度的JSON数组实例
	 *
	 * @param val 长度
	 * @return
	 */
	public static JSONArray getArray(Object val) {
		int len = 0;
		try {
			len = new BigDecimal(val.toString()).intValue();
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
		return new JSONArray(new String[len]);
	}

	/**
	 * 返回一个JSON实例
	 *
	 * @return JSON数组实例
	 */
	public static JSONObject getJson() {return new JSONObject();}

	/**
	 * 这个方法是将json转换成HashMap
	 * @param object json对象
	 * @return 转换后的map
	 */
	public static HashMap<String, Object> toHashMap(JSONObject object) {
		HashMap<String, Object> data = new HashMap<>(16);
		// 将json字符串转换成jsonObject
		Iterator<String> it = object.keys();
		// 遍历jsonObject数据，添加到Map对象
		while (it.hasNext()) {
			String key = String.valueOf(it.next());
			if (object.isNull(key)) {
                data.put(key, null);
            } else {
				Object value = object.get(key);
				data.put(key, value);
			}
		}
		return data;
	}

	/**
	 * 比较两个JSONObject,  返回的JSONArray格式为[{field:xxx, oldVal:xxx, newVal: xxx}]
	 *
	 * @param jo1 新值
	 * @param jo2 旧值
	 * @return JSONArray
	 */
	public static JSONArray findDiff(JSONObject jo1, JSONObject jo2) {
		JSONArray jsonArray;
		Set<String> keySet1 = jo1.keySet();
		jsonArray = new JSONArray();
		for (String s : keySet1) {
			if (isValueEquals(jo1, jo2, s)) {
				JSONObject jo = new JSONObject();
				jo.put("field", s);
				jo.put("newVal", jo1.get(s) == null ? null : jo1.get(s)
						.toString());
				jo.put("oldVal", jo2.get(s) == null ? null : jo2.get(s)
						.toString());
				jsonArray.put(jo);
			}
		}
		return jsonArray;
	}

	/**
	 * 判断两个json中值是否相同
	 *
	 * @param jo1 新值
	 * @param jo2 旧值
	 * @return 判断结果
	 */
	private static boolean isValueEquals(JSONObject jo1, JSONObject jo2, String s) {
		// 针对级联json数据
		if ((jo1.isNull(s) && !jo2.isNull(s))
				|| (!jo1.isNull(s) && jo2.isNull(s))) {
			return false;
		}
		if (jo1.isNull(s) && (jo2.isNull(s) || "null".equals(jo2.get(s)))) {
			return false;
		}
		if (jo1.get(s) instanceof Integer) {
			if (jo2.get(s) instanceof BigDecimal) {
				return new BigDecimal(jo1.get(s).toString())
						.compareTo((BigDecimal) jo2.get(s)) != 0;
			} else {
				try {
					return new BigDecimal(jo1.get(s).toString())
							.compareTo(new BigDecimal(jo2
									.get(s).toString())) != 0;
				} catch (Exception e) {
					return false;
				}
			}
		}
		if (jo1.get(s) instanceof String) {
			if (jo2.get(s) instanceof String) {
				return !jo1.get(s).equals(jo2.get(s));
			} else {
				try {
					// 界面传过来的是字符串，库里拿到的是BigDecimal类型
					return new BigDecimal(jo1.get(s).toString())
							.compareTo((BigDecimal)jo2.get(s)) != 0;
				} catch (Exception e) {
					return false;
				}
			}
		}
		if (jo1.get(s) instanceof Double) {
			if (jo2.get(s) instanceof Integer) {
				return new BigDecimal(jo1.get(s).toString())
						.compareTo(new BigDecimal(jo2.get(s).toString())) != 0;
			} else {
				try {
					return new BigDecimal(jo1.get(s).toString())
							.compareTo(new BigDecimal(jo2.get(s).toString())) != 0;
				} catch (Exception e) {
					return false;
				}
			}
		}
		return !(jo2.get(s).equals(jo1.get(s)));
	}

	/**
	 * 应用场景：对于售气收费，为匹配气价的计算，将json字符串进行相应的包装
	 * 未包装前:{"data":{"price":"1","amount":"12"}}
	 * 包装后:{"data":[{"price":"1","amount":"12"}]}
	 *
	 * @param json json对象
	 * @param pricetype 气价类型
	 * @return 包装后的json数组
	 */
	public static JSONArray packJson(JSONObject json, String pricetype) {
		JSONArray result = new JSONArray();
		if ("fixed".equals(pricetype)) {
			JSONObject value = new JSONObject();
			value.put("f_unitprice", new BigDecimal(String.valueOf(json.get("f_unitprice"))));
			result.put(value);
		} else if ("mixed".equals(pricetype)) {
			JSONArray array = (JSONArray) json.get("detailprice");
			JSONObject obj = new JSONObject();
			obj.put("f_hybridprice1",
					new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_price"))));
			obj.put("f_hybridprice2",
					new BigDecimal(String.valueOf(array.getJSONObject(1).get("f_price"))));
			obj.put("f_hybridprice1rate",
					new BigDecimal(String.valueOf(array.getJSONObject(0).get("f_ratio"))));
			result.put(obj);
		} else if ("staired".equals(pricetype)) {
			JSONArray array = (JSONArray) json.get("detailprice");
			for (Object object : array) {
				JSONObject obj = new JSONObject();
				obj.put("price", new BigDecimal(String.valueOf(((JSONObject) object).get("f_price"))));
				obj.put("amount",new BigDecimal(String.valueOf(((JSONObject) object).get("f_amount"))));
				result.put(obj);
			}
			JSONObject obj = new JSONObject();
			obj.put("sumamount", json.get("sumamount"));
			result.put(obj);
		}
		return result;
	}

	/**
	 * 把对象数组里的某个字段内容，转换成'name1', 'name2'的形式
	 * @param array 对象数组
	 * @param field String
	 * @return 转换后的String
	 */
	public static String toString(JSONArray array, String field) {
		StringBuilder result = new StringBuilder();
		for (Object object : array) {
			JSONObject obj = (JSONObject)object;
			String value = obj.optString(field);
			if (!"".equals(result.toString())) {
				result.append(",");
			}
			result.append("'").append(value).append("'");
		}
		return result.toString();
	}

	/**
	 * 读取JSON文件，返回JSON对象
	 * @param fileName 文件路径
	 * @return JSONObject
	 */
	public static JSONObject readJsonFile(String fileName) {
		BufferedReader reader = null;
		StringBuilder laststr = new StringBuilder();
		try {
			InputStream input = JsonTools.class
					.getResourceAsStream("/" + fileName);
			InputStreamReader inputStreamReader = new InputStreamReader(input,
					StandardCharsets.UTF_8);
			reader = new BufferedReader(inputStreamReader);
			String tempString;
			while ((tempString = reader.readLine()) != null) {
				laststr.append(tempString);
			}
			reader.close();
			return new JSONObject(laststr.toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * 读取JSON文件，返回JSON数组
	 * @param fileName 文件路径
	 * @return JSONArray
	 */
	public static JSONArray readJsonArrayFile(String fileName) {
		BufferedReader reader = null;
		StringBuilder laststr = new StringBuilder();
		try {
			InputStream input = JsonTools.class
					.getResourceAsStream("/"+fileName);
			InputStreamReader inputStreamReader = new InputStreamReader(input,
					StandardCharsets.UTF_8);
			reader = new BufferedReader(inputStreamReader);
			String tempString;
			while ((tempString = reader.readLine()) != null) {
				laststr.append(tempString);
			}
			reader.close();
			return new JSONArray(laststr.toString());
		} catch (IOException e) {
			throw new RuntimeException(e);
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}

	/**
	 * JSON键值对替换（保留原JSON中的键值对）
	 * @param oldObj 被替换的JSON
	 * @param newObj 替换的JSON
	 * @return 处理后的JSON
	 */
	public static JSONObject replaceJSON(JSONObject oldObj,JSONObject newObj){
		for(String key : newObj.keySet()){
			//如果新JSON中有原JSON的键值对
			oldObj.put(key,newObj.get(key));
		}
		return oldObj;
	}
}
