package com.aote.entity;

import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.StrUtil;
import com.af.plugins.io.IOTools;
import com.af.util.Pair;
import com.aote.config.SystemConfig;
import com.aote.module.ModuleMapper;
import com.aote.rs.mapper.WebException;
import com.aote.sql.SqlMapper;
import com.aote.sql.SqlServer;
import com.aote.transaction.SessionPool;
import com.aote.util.JsonHelper;
import com.aote.util.SqlHelper;
import org.apache.log4j.Logger;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.hibernate.SQLQuery;
import org.hibernate.SessionFactory;
import org.hibernate.engine.spi.SessionFactoryImplementor;
import org.hibernate.exception.SQLGrammarException;
import org.hibernate.id.*;
import org.hibernate.internal.SessionImpl;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.persister.entity.AbstractEntityPersister;
import org.hibernate.type.CollectionType;
import org.hibernate.type.ManyToOneType;
import org.hibernate.type.OneToOneType;
import org.hibernate.type.Type;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.PostConstruct;
import java.io.*;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.Statement;
import java.text.SimpleDateFormat;
import java.util.*;

@Component
@Transactional
public class EntityServer {
    public static final String DIALECT_SQLSERVER = "SqlServer";
    public static final String DIALECT_ORACLE = "Oracle";
    public static final String DIALECT_MYSQL = "MySQL";
    public static final String DIALECT_DmDialect = "DmDialect";

    private static final String COL_STRING = "STRING";
    private static final String COL_DATE = "DATE";
    private static final String COL_TIME = "TIME";
    private static final String COL_BOOLEAN = "BOOLEAN";
    private static final String COL_NUMBER = "NUMBER";
    private static final String COL_CLOB = "CLOB";
    private static final String COL_NOTSUPPORTED = "NOT_SUPPORTED";


    private static final String ID_GUID = "ID_GUID";
    private static final String ID_SEQ = "ID_SEQ";
    private static final String ID_AUTO = "ID_AUTO";
    private static final String ID_ASSIGNED = "ID_ASSIGNED";
    private static final String ID_FOREIGNER = "ID_FOREIGNER";

    public static HashMap<String, HashMap<String, Object>> metaMap = new HashMap<>();

    public static HashMap<String, String> entityLiftMap = new HashMap<>();

    static Logger log = Logger.getLogger(EntityServer.class);

    private static String dialectType = "";

    @Autowired
    public SessionFactory sessionFactory;

    @Autowired
    private SessionPool sessionPool;

    @PostConstruct
    public void init() throws Exception {
        SessionFactoryImplementor sf = (SessionFactoryImplementor) sessionFactory;
        String dialect = sf.getDialect().toString().toLowerCase();
        if (dialect.contains("sqlserver")) {
            dialectType = EntityServer.DIALECT_SQLSERVER;
        } else if (dialect.contains("oracle")) {
            dialectType = EntityServer.DIALECT_ORACLE;
        } else if (dialect.contains("mysql")) {
            dialectType = EntityServer.DIALECT_MYSQL;
        } else if(dialect.contains("dm")){
            dialectType = EntityServer.DIALECT_DmDialect;
        } else {
            throw new Exception("数据库方言配置异常。");
        }
    }

    // 删除实体
    public String delete(String entityName, String id) {
        HashMap<String, Object> map = metaMap.get(entityName);
        String hql = "delete from " + getRealTableName(map) + " where " + map.get("idName") + "=" + normalizeValue(findDialect(), id, (String) map.get("idType"));
        SqlHelper.bulkSQLUpdate(sessionPool.getSession(), hql);
        return "ok";
    }

    /**
     * find out the JDBC dialect
     */
    public static String findDialect() {
        return dialectType;
    }


    /**
     * normalize type
     */
    private String normalizeType(String type) {
        type = type.toLowerCase();
        switch (type) {
            case "string":
                return EntityServer.COL_STRING;
            case "date":
                return EntityServer.COL_DATE;
            case "time":
            case "timestamp":
                return EntityServer.COL_TIME;
            case "boolean":
            case "yes_no":
                return EntityServer.COL_BOOLEAN;
            case "integer":
            case "double":
            case "big_decimal":
            case "big_integer":
            case "long":
                return EntityServer.COL_NUMBER;
            case "clob":
                return EntityServer.COL_CLOB;
            default:
                return EntityServer.COL_NOTSUPPORTED;
        }
    }


    /**
     * get entity meta data and cache it
     */
    @SuppressWarnings("unchecked")
    private HashMap<String, Object> getMetaData(String entityName) throws Exception {
        // 如果元数据已经获取，直接返回
        if (metaMap.containsKey(entityName)) {
            return metaMap.get(entityName);
        }

        HashMap<String, Object> map = new HashMap<>();

        //实体名称
        map.put("entityName", entityName);

        //获取表名
        ClassMetadata cmd = sessionFactory.getClassMetadata(entityName);
        AbstractEntityPersister aep = (AbstractEntityPersister) cmd;
        map.put("tableName", aep.getTableName());

        //获取主键属性名称
        String attr = cmd.getIdentifierPropertyName();
        if (attr == null) {
            log.warn("停止处理【" + entityName + "】的映射，可能含有不支持的映射结构");
            return null;
        }
        map.put("idName", attr);
        map.put("idColName", aep.getPropertyColumnNames(attr)[0]);
        //获取主键类型
        String idType = normalizeType(cmd.getIdentifierType().getName());
        map.put("idType", idType);

        //仅支持guid、sequence（oracle）、auto-increment、assigned、outside（id来自主实体的一对一）
        IdentifierGenerator idg = aep.getIdentifierGenerator();
        if (idg instanceof GUIDGenerator || idg instanceof UUIDGenerator
                || idg instanceof UUIDHexGenerator) {
            map.put("idGenerator", EntityServer.ID_GUID);
        } else if (idg instanceof SequenceGenerator) {
            String name = ((SequenceGenerator) idg).getSequenceName();
            // 创建sequence，hibernate没有自动创建 (被MyOracle10gDialect代替)
            // this.createSequence(name);
            map.put("sequence", name);
            map.put("idGenerator", EntityServer.ID_SEQ);
        } else if (idg instanceof IdentityGenerator) {
            map.put("idGenerator", EntityServer.ID_AUTO);
        } else if (idg instanceof Assigned) {
            map.put("idGenerator", EntityServer.ID_ASSIGNED);
        } else if (idg instanceof ForeignGenerator) {
            map.put("idGenerator", EntityServer.ID_FOREIGNER);
        } else {
            throw new Exception("Unsupported id generator strategy:" + idg.getClass().getName());
        }

        //判断该实体是否有乐观锁（即版本号）控制
        if (cmd.isVersioned()) {
            attr = cmd.getPropertyNames()[cmd.getVersionProperty()];
            map.put("verName", attr);
            map.put("verColName", aep.getPropertyColumnNames(attr)[0]);
            map.put("verType", normalizeType(cmd.getPropertyType(attr).getName()));
        }

        //普通列
        HashMap<String, Pair> columns = new HashMap<>();
        map.put("columns", columns);

        //一对一，一对多关系映射
        HashMap<String, Pair> links = new HashMap<>();
        map.put("links", links);

        //一对一
        HashMap<String, Pair> onetoone = new HashMap<>();
        map.put("onetoone", onetoone);

        //一对多
        HashMap<String, Pair> onetomany = new HashMap<>();
        map.put("onetomany", onetomany);

        //遍历实体属性集合
        for (String property : cmd.getPropertyNames()) {
            //获取类型
            Type type = cmd.getPropertyType(property);
            if (type instanceof CollectionType || type instanceof OneToOneType) {
                org.hibernate.persister.entity.Joinable ja;
                String linkEntity;
                SessionFactoryImplementor sf = (SessionFactoryImplementor) sessionFactory;
                if (type instanceof CollectionType) {
                    CollectionType st = (CollectionType) type;
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                } else {
                    OneToOneType st = (OneToOneType) type;
                    ja = st.getAssociatedJoinable(sf);
                    linkEntity = st.getAssociatedEntityName(sf);
                }
                String foreignKey = ja.getKeyColumnNames()[0];
                String link;
                if (type instanceof OneToOneType) {
                    link = property;
                } else {
                    link = ja.getName().substring(entityName.length() + 1);

                    HashMap<String, Object> linkedMap;
                    //prevent infinite recursion
                    if (linkEntity.equals(entityName)) {
                        linkedMap = map;
                    } else {
                        linkedMap = getMetaData(linkEntity);
                    }
                    // associations原先想要处理反向的属性，已经无用
                    HashMap<String, Object> pkfk;
                    assert linkedMap != null;
                    if (linkedMap.containsKey("associations")) {
                        pkfk = (HashMap<String, Object>) linkedMap.get("associations");
                        pkfk.put(entityName, new Pair(foreignKey, idType));
                    } else {
                        pkfk = new HashMap<>();
                        pkfk.put(entityName, new Pair(foreignKey, idType));
                        linkedMap.put("associations", pkfk);
                    }
                }
                links.put(link, new Pair(linkEntity, foreignKey));
                // 把一对多和一对一分开
                if (type instanceof CollectionType) {
                    onetomany.put(link, new Pair(linkEntity, foreignKey));
                } else {
                    onetoone.put(link, new Pair(linkEntity, foreignKey));
                }
            } else if (type instanceof ManyToOneType) {
                continue;
            } else {
                String columnName = ((AbstractEntityPersister) cmd).getPropertyColumnNames(property)[0];
                columns.put(property, new Pair(columnName, normalizeType(type.getName())));
            }
        }

        metaMap.put(entityName, map);
        return map;
    }

    /**
     * de clone
     */
    @SuppressWarnings("unchecked")
    public <T extends Serializable> T deepClone(T o) throws Exception {
        ByteArrayOutputStream byteOut = new ByteArrayOutputStream();
        ObjectOutputStream out = new ObjectOutputStream(byteOut);
        out.writeObject(o);
        out.flush();
        ObjectInputStream in = new ObjectInputStream(new ByteArrayInputStream(byteOut.toByteArray()));
        return (T) o.getClass().cast(in.readObject());
    }

    /**
     * enchanceMetaData
     */
    @SuppressWarnings("unchecked")
    private void enchanceMetaData(HashMap<String, Object> pmd, String entityName) throws Exception {
        HashMap<String, Object> md = metaMap.get(entityName);

        HashMap<String, Pair> links = (HashMap<String, Pair>) md.get("links");
        HashMap<String, Pair> plinks = (HashMap<String, Pair>) pmd.get("links");
        for (String key : links.keySet()) {
            plinks.put(key, deepClone(links.get(key)));
        }

        Map<String, Pair> sub;
        if (md.containsKey("subclasses")) {
            sub = (LinkedHashMap<String, Pair>) md.get("subclasses");
        } else {
            sub = new LinkedHashMap<>();
        }
        Map<String, Pair> psub;
        if (pmd.containsKey("subclasses")) {
            psub = (LinkedHashMap<String, Pair>) pmd.get("subclasses");
        } else {
            psub = new LinkedHashMap<>();
        }
        for (String key : sub.keySet()) {
            psub.put(key, deepClone(sub.get(key)));
        }

        HashMap<String, String> inv = (HashMap<String, String>) md.get("inverses");
        HashMap<String, String> pinv = (HashMap<String, String>) pmd.get("inverses");
        for (String key : inv.keySet()) {
            pinv.put(key, inv.get(key));
        }

        HashMap<String, Pair> invid;
        if (md.containsKey("inverseid")) {
            invid = (HashMap<String, Pair>) md.get("inverseid");
        } else {
            invid = new HashMap<>();
        }
        HashMap<String, Pair> pinvid;
        if (pmd.containsKey("inverseid")) {
            pinvid = (HashMap<String, Pair>) pmd.get("inverseid");
        } else {
            pinvid = new HashMap<>();
        }

        for (String key : invid.keySet()) {
            pinvid.put(key, deepClone(invid.get(key)));
        }
    }

    public String partialSaveByEntity(String entityName, Object entity) {
        try {
            JSONObject object = JsonHelper.toJSON(entity);
            return partialSave(entityName, object);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }


    /**
     * 部分保存对象
     */
    public String partialSave(String entityName, JSONObject row) throws Exception {
        return partialSave(entityName, row, false);
    }

    public String partialSave(String entityName, JSONObject row, boolean isNoLog) throws Exception {

        HashMap<String, Object> md;

        // lift possible child entity
        if (entityLiftMap.containsKey(entityName)) {
            String pEntityName = entityLiftMap.get(entityName);
            md = deepClone(metaMap.get(pEntityName));
            enchanceMetaData(md, entityName);
        } else {
            md = metaMap.get(entityName);
        }


        String dialect = findDialect();

        Serializable aid = partialSave(row, md, null, null, dialect, isNoLog);
        JSONObject result = new JSONObject();
        result.put((String) md.get("idName"), aid.toString());
        return result.toString();
    }

    /**
     * 部分保存对象
     */
    public String partialSave(String entityName, String values) throws Exception {
        JSONObject row = new JSONObject(values);
        return partialSave(entityName, row);
    }

    private Serializable partialSave(JSONObject row, HashMap<String, Object> md, Object pIdValue, String pEntityName, String dialect) throws Exception {
        return partialSave(row, md, pIdValue, pEntityName, dialect, false);
    }

    private Serializable partialSave(JSONObject row, HashMap<String, Object> md, Object pIdValue, String pEntityName, String dialect, boolean isNoLog) throws Exception {

        Object idValue = null;
        String idName = (String) md.get("idName");
        String idType = (String) md.get("idType");
        if (row.has(idName)) {
            idValue = row.get(idName);
        }

        @SuppressWarnings("unchecked")
        HashMap<String, Pair> columns = (HashMap<String, Pair>) md.get("columns");
        @SuppressWarnings("unchecked")
        HashMap<String, Pair> links = (HashMap<String, Pair>) md.get("links");

        boolean isInsert = false;
        int affectedRows;

        if (idValue == null || idValue == JSONObject.NULL) {
            affectedRows = doInsert(row, md, pIdValue, pEntityName, dialect, idName, idType, columns, isNoLog);
        } else {

            //if not assigned key or not the ancestor object, and the idvalue is not null, simply update it
            if (!md.get("idGenerator").equals(EntityServer.ID_ASSIGNED)) {
                affectedRows = doUpdate(row, md, pIdValue, pEntityName, dialect, idValue, idType, columns);
            } else {
                if (hasKeyRow(dialect, getRealTableName(md), (String) md.get("idColName"), idValue, idType)) {
                    affectedRows = doUpdate(row, md, pIdValue, pEntityName, dialect, idValue, idType, columns);
                } else {
                    isInsert = true;
                    affectedRows = doInsert(row, md, pIdValue, pEntityName, dialect, idName, idType, columns, isNoLog);
                }
            }
        }

        //stale object
        if (affectedRows == 0) {
            JSONObject errorMsg = new JSONObject();
            errorMsg.put("status", -1);
            errorMsg.put("erroMsg", "修改数据失败，返回的影响行数为:0，可能是以下原因之一：\n" +
                    "1.有其他业务同时修改该条数据导致触发hibernate乐观锁，请检索：hibernate乐观锁\n" +
                    "2.update的条件表达式有误");
            String msg = errorMsg.toString();
            log.error(msg);
            throw new WebException(505, msg);
        }

        //possible modified idValue
        idValue = row.get(idName);


        //remove idName in row
        if (isInsert) {
            row.remove(idName);
        }

        //save or update one-to-many
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            if (links.containsKey(attr) && row.has(attr)) {
                Pair pair = links.get(attr);
                HashMap<String, Object> cmd = metaMap.get(pair.col);
                String cEntityName = (String) md.get("entityName");
                //如果为空，如果是OneToOne，删除子
                if (row.isNull(attr)) {
                    if (pair.col.equals(attr))
                    //如果是一对一，子没传值，则删除子。
                    {
                        deleteOneToOne(dialect, getRealTableName(cmd), (String) cmd.get("idColName"), row.get(idName), idType);
                    }
                } else {
                    Object obj = row.get(attr);
                    if (obj instanceof JSONArray) {
                        JSONArray rows = row.getJSONArray(attr);
                        for (int i = 0; i < rows.length(); i++) {
                            partialSave(rows.getJSONObject(i), cmd, idValue, cEntityName, dialect);
                        }
                    } else {
                        partialSave((JSONObject) obj, cmd, idValue, cEntityName, dialect);
                    }
                }
            }
        }
        return (Serializable) idValue;
    }

    private void deleteOneToOne(String dialect, String tableName, String idCol, Object idValue, String idType) {
        String sql = "delete from " + tableName + " where " + idCol + "=" + normalizeValue(dialect, idValue, idType);
        log.info("一对一删除sql:" + sql);
        SQLUpdate(sql);
    }

    /**
     * hasKeyRow
     */
    private boolean hasKeyRow(String dialect, String tableName, String idColName, Object idValue, String idType) {
        return sessionPool.getSession().
                createSQLQuery("select 1 from " + tableName + " where " + idColName + "=" + normalizeValue(dialect, idValue, idType))
                .list().size() > 0;
    }

    private int doUpdate(JSONObject row, HashMap<String, Object> md,
                         Object pIdValue, String pEntityName, String dialect,
                         Object idValue, String idType, HashMap<String, Pair> columns) throws Exception {
        // 如果只有一个id号，不用更新，返回-1，避免认为更新失败
        if (row.keySet().size() == 1) {
            return -1;
        }
        Object verValue = null;
        String verName = (String) md.get("verName");
        String verType = (String) md.get("verType");
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        StringBuilder sb = new StringBuilder();

        String foreignkey = null;

        //update possible foreign key column value
        if (pEntityName != null) {
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> associations = (HashMap<String, Pair>) md.get("associations");
            if (associations != null && associations.containsKey(pEntityName)) {
                Pair pair = associations.get(pEntityName);
                foreignkey = pair.col;
                sb.append(" ").append(foreignkey).append("=").append(normalizeValue(dialect, pIdValue, pair.type)).append(", ");
            }
        }

        if (verName != null) {
            sb.append(" ").append(verName).append("=").append(normalizeVer(dialect, verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(foreignkey) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sb.append(" ").append(pair.col).append("=").append(row.isNull(attr) ? "null, " : normalizeValue(dialect, row.get(attr), pair.type) + ", ");
            }
        }

        if (sb.length() == 0) {
            log.error("提交的数据错误：" + row);
            throw new Exception("提交的数据错误：" + row);
        }

        sb.delete(sb.length() - 2, sb.length());
        sb.insert(0, "update " + getRealTableName(md) + " set");
        sb.append(" where ").append(md.get("idColName")).append(" = ").append(normalizeValue(dialect, idValue, idType));

        if (md.get("verName") != null) {
            sb.append(" and ").append(md.get("verColName")).append("=").append(normalizeValue(dialect, verValue, verType));
        }


        log.info("生成的更新sql: " + sb);
        int affectedRow = SQLUpdate(sb.toString());

        //update possible sub-entity
        if (md.containsKey("subclasses")) {
            @SuppressWarnings("unchecked")
            Map<String, Pair> subclasses = (LinkedHashMap<String, Pair>) md.get("subclasses");
            for (String sentity : subclasses.keySet()) {
                Pair p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col)) {
                    break;
                }
                //if discriminator matches
                String dv = row.getString(p.col);
                if (p.type.equals(dv) || p.type.contains(dv + ",") || p.type.contains("," + dv)) {
                    StringBuilder buf = new StringBuilder();
                    //update row
                    HashMap<String, Object> smd = metaMap.get(sentity);
                    String keyAttr = (String) smd.get("idName");
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> attrs = (HashMap<String, Pair>) smd.get("columns");
                    for (String attr : attrs.keySet()) {
                        //skip key attr
                        if (attr.equals(keyAttr)) {
                            continue;
                        }
                        if (row.has(attr)) {
                            p = attrs.get(attr);
                            buf.append(p.col).append("=").append(normalizeValue(dialect, row.isNull(attr) ? null : row.get(attr), p.type)).append(", ");
                        }
                    }

                    buf.delete(buf.length() - 2, buf.length());
                    buf.insert(0, "update " + getRealTableName(smd) + " set");
                    buf.append(" where ").append(smd.get("idColName")).append(" = ").append(normalizeValue(dialect, idValue, idType));

                    log.info("生成的子更新sql: " + buf);
                    SQLUpdate(buf.toString());

                    //继续检查子类的关联，否则就断了链了，但如果父有关联了，儿子就偷懒了
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> links = (HashMap<String, Pair>) smd.get("links");
                    HashMap<String, Pair> pLinks = (HashMap<String, Pair>) md.get("links");
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            Pair pair = links.get(attr);
                            HashMap<String, Object> cmd = metaMap.get(pair.col);
                            String cEntityName = (String) smd.get("entityName");
                            //如果为空，如果是OneToOne，删除子
                            if (row.isNull(attr)) {
                                if (pair.col.equals(attr))
                                //如果是一对一，子没传值，则删除子。
                                {
                                    deleteOneToOne(dialect, getRealTableName(cmd), (String) cmd.get("idColName"), idValue, idType);
                                }
                            } else {
                                Object obj = row.get(attr);
                                if (obj instanceof JSONArray) {
                                    JSONArray rows = row.getJSONArray(attr);
                                    for (int i = 0; i < rows.length(); i++) {
                                        partialSave(rows.getJSONObject(i), cmd, idValue, cEntityName, dialect);
                                    }
                                } else {
                                    partialSave((JSONObject) obj, cmd, idValue, cEntityName, dialect);
                                }
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md, dialect);

        return affectedRow;
    }

    private void handleInverses(JSONObject row, HashMap<String, Object> md,
                                String dialect) throws Exception {
        //inverse relation
        @SuppressWarnings("unchecked")
        HashMap<String, String> inverses = (HashMap<String, String>) md.get("inverses");
        for (String inverse : inverses.keySet()) {
            if (row.isNull(inverse)) {
                continue;
            }
            Object arow = row.get(inverse);
            if (arow instanceof JSONObject) {
                HashMap<String, Object> amd = metaMap.get(inverses.get(inverse));
                partialSave((JSONObject) arow, amd, null, null, dialect);
            }
        }
    }

    private int doInsert(JSONObject row, HashMap<String, Object> md,
                         Object pIdValue, String pEntityName, String dialect,
                         String idName, String idType, HashMap<String, Pair> columns) throws Exception {
        return doInsert(row, md, pIdValue, pEntityName, dialect, idName, idType, columns, false);
    }

    private int doInsert(JSONObject row, HashMap<String, Object> md,
                         Object pIdValue, String pEntityName, String dialect,
                         String idName, String idType, HashMap<String, Pair> columns, Boolean isNoLog) throws Exception {

        Object verValue = null;
        String verName = (String) md.get("verName");
        String verType = (String) md.get("verType");
        if (verName != null && row.has(verName)) {
            verValue = row.get(verName);
        }

        String idColName = (String) md.get("idColName");

        StringBuilder sbCols = new StringBuilder();
        StringBuilder sbValues = new StringBuilder();

        String foreignkey = null;
        //update possible foreign key column value
        if (pEntityName != null) {
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> associations = (HashMap<String, Pair>) md.get("associations");
            if (associations != null) {
                //处理一对多的关系
                for (String ass : associations.keySet()) {
                    if (ass.equals(pEntityName) || hasParent(ass, pEntityName)) {
                        Pair pair = associations.get(ass);
                        foreignkey = pair.col;
                        if (sbCols.indexOf(foreignkey + ",") == -1) {
                            sbCols.append(foreignkey).append(", ");
                            sbValues.append(normalizeValue(dialect, pIdValue, pair.type)).append(", ");
                        }
                    }
                }
            }
        }

        if (verName != null) {
            sbCols.append(verName).append(", ");
            sbValues.append(normalizeVer(dialect, verValue, verType)).append(", ");
        }

        //enumerate every property in the row
        Iterator<String> itr = row.keys();
        while (itr.hasNext()) {
            String attr = itr.next();
            //average column, maybe foreign key column, but foreign key column is ignored since it is handled already
            if (columns.containsKey(attr) && !attr.equals(foreignkey) && !attr.equals(verName)) {
                Pair pair = columns.get(attr);
                sbCols.append(pair.col).append(", ");
                sbValues.append(row.isNull(attr) ? "null, " : normalizeValue(dialect, row.get(attr), pair.type) + ", ");
            }
        }


        //handle id problems
        String idStrategy = (String) md.get("idGenerator");
        switch (idStrategy) {
            case EntityServer.ID_GUID:
                sbCols.append(idColName).append(", ");
                String guid = UUID.randomUUID().toString().replace("-", "");
                sbValues.append("'").append(guid).append("', ");
                row.put(idName, guid);
                break;
            case EntityServer.ID_SEQ: {
                sbCols.append(idColName).append(", ");
                String aid = getLastSeqId((String) md.get("sequence"));
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
                break;
            }
            case EntityServer.ID_ASSIGNED:
                sbCols.append(idColName).append(", ");
                sbValues.append(normalizeValue(dialect, row.get(idName), idType)).append(", ");
                row.put(idName, row.get(idName));
                break;
            case EntityServer.ID_FOREIGNER: {
                sbCols.append(idColName).append(", ");
                String aid = normalizeValue(dialect, pIdValue, idType);
                sbValues.append(aid).append(", ");
                row.put(idName, aid);
                break;
            }
            default:
                break;
        }

        if (md.containsKey("inverses")) {
            //handle inverse problems
            @SuppressWarnings("unchecked")
            HashMap<String, String> inverses = (HashMap<String, String>) md.get("inverses");
            @SuppressWarnings("unchecked")
            HashMap<String, Pair> inversesid = (HashMap<String, Pair>) md.get("inverseid");
            for (String inverse : inverses.keySet()) {
                if (row.isNull(inverse)) {
                    continue;
                }
                Object arow = row.get(inverse);
                //log.debug("发现多对一属性赋值:");
                if (!(arow instanceof JSONObject)) {
                    Pair pair = inversesid.get(inverse);
                    sbCols.append(pair.col).append(", ");
                    sbValues.append(normalizeValue(dialect, arow, pair.type)).append(", ");
                    //log.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                } else {
                    // 多对一对象关系，属性名与列名必须一致
                    sbCols.append(inverse).append(", ");
                    // 值从给定的对象中取
                    JSONObject json = (JSONObject) arow;
                    Pair pair = inversesid.get(inverse);
                    arow = json.get(pair.col);
                    sbValues.append(normalizeValue(dialect, arow, pair.type)).append(", ");
                    //log.debug("发现多对一属性赋值:属性:" + pair.col + "，值：" + arow);
                }
            }
        }

        sbCols.delete(sbCols.length() - 2, sbCols.length());
        sbCols.insert(0, "insert into " + getRealTableName(md) + " (");
        sbValues.delete(sbValues.length() - 2, sbValues.length());
        sbCols.append(") values (");
        sbCols.append(sbValues);
        sbCols.append(")");

        if (!isNoLog) {
            log.info("生成的插入sql；" + sbCols);
        }

        int affectedRows;

        if (idStrategy.equals(EntityServer.ID_AUTO)) {
            Pair pair = rawJdbcUpdate(sbCols.toString());
            row.put(idName, pair.type);
            affectedRows = Integer.parseInt(pair.col);
        } else {
            affectedRows = SQLUpdate(sbCols.toString());
        }

        //insert possible sub-entity
        if (md.containsKey("subclasses")) {
            @SuppressWarnings("unchecked")
            Map<String, Pair> subclasses = (LinkedHashMap<String, Pair>) md.get("subclasses");
            for (String sentity : subclasses.keySet()) {
                Pair p = subclasses.get(sentity);
                //if discriminator is not found
                if (row.isNull(p.col)) {
                    break;
                }
                String dv = row.getString(p.col);
                if (p.type.equals(dv) || p.type.contains(dv + ",") || p.type.contains("," + dv)) {
                    StringBuilder fields = new StringBuilder();
                    StringBuilder vals = new StringBuilder();

                    //update row
                    HashMap<String, Object> smd = metaMap.get(sentity);
                    String keyCol = (String) smd.get("idColName");

                    //handle it
                    fields.append(keyCol).append(", ");
                    vals.append(normalizeValue(dialect, row.get(idName), idType)).append(", ");

                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> attrs = (HashMap<String, Pair>) smd.get("columns");
                    for (String attr : attrs.keySet()) {
                        if (row.has(attr)) {
                            p = attrs.get(attr);
                            // 外键不当属性加入
                            if (!(p.col.equals(foreignkey))) {
                                fields.append(p.col).append(", ");
                                vals.append(normalizeValue(dialect, row.isNull(attr) ? null : row.get(attr), p.type)).append(", ");
                            }
                        }
                    }

                    //处理反向查id的情况
                    if (smd.containsKey("inverseid")) {
                        @SuppressWarnings("unchecked")
                        HashMap<String, Pair> lookup = (HashMap<String, Pair>) smd.get("inverseid");
                        for (String property : lookup.keySet()) {
                            if (row.has(property) && !row.isNull(property)) {
                                Object obj = row.get(property);
                                Pair pair = lookup.get(property);
                                if (obj instanceof JSONObject) {
                                    JSONObject record = (JSONObject) obj;
                                    if (record.has(pair.col) && !record.isNull(pair.col)) {
                                        fields.append(pair.col).append(", ");
                                        vals.append(normalizeValue(dialect, record.get(pair.col), pair.type)).append(", ");
                                    }
                                } else {
                                    fields.append(pair.col).append(", ");
                                    vals.append(normalizeValue(dialect, row.get(property), pair.type)).append(", ");
                                }
                            }
                        }
                    }

                    //处理子类的外键
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> associations = (HashMap<String, Pair>) md.get("associations");
                    if (associations != null) {
                        //处理一对多的关系
                        for (String ass : associations.keySet()) {
                            //如果子类的定义包括此列
                            Pair pair = associations.get(ass);
                            foreignkey = pair.col;
                            if (attrs.containsKey(foreignkey) && (pIdValue != null) && (fields.indexOf(foreignkey + ", ") == -1)) {
                                fields.append(foreignkey).append(", ");
                                vals.append(normalizeValue(dialect, pIdValue, pair.type)).append(", ");
                            }
                        }
                    }

                    fields.delete(fields.length() - 2, fields.length());
                    vals.delete(vals.length() - 2, vals.length());

                    fields.insert(0, "insert into " + getRealTableName(smd) + " (");
                    fields.append(") values (");
                    fields.append(vals);
                    fields.append(")");

                    log.info("生成的子插入sql: " + fields);
                    SQLUpdate(fields.toString());

                    //继续检查子类的关联，否则就断了链了
                    @SuppressWarnings("unchecked")
                    HashMap<String, Pair> links = (HashMap<String, Pair>) smd.get("links");
                    HashMap<String, Pair> pLinks = (HashMap<String, Pair>) md.get("links");
                    Iterator<String> itrLink = row.keys();
                    while (itrLink.hasNext()) {
                        String attr = itrLink.next();
                        if (!pLinks.containsKey(attr) && links.containsKey(attr) && row.has(attr)) {
                            Pair pair = links.get(attr);
                            HashMap<String, Object> cmd = metaMap.get(pair.col);
                            String cEntityName = (String) smd.get("entityName");
                            //如果为空，如果是OneToOne，删除子
                            if (row.isNull(attr)) {
                                if (pair.col.equals(attr))
                                //如果是一对一，子没传值，则删除子。
                                {
                                    deleteOneToOne(dialect, getRealTableName(cmd), (String) cmd.get("idColName"), row.get(idName), idType);
                                }
                            } else {
                                Object obj = row.get(attr);
                                if (obj instanceof JSONArray) {
                                    JSONArray rows = row.getJSONArray(attr);
                                    for (int i = 0; i < rows.length(); i++) {
                                        partialSave(rows.getJSONObject(i), cmd, row.get(idName), cEntityName, dialect);
                                    }
                                } else {
                                    partialSave((JSONObject) obj, cmd, row.get(idName), cEntityName, dialect);
                                }
                            }
                        }
                    }
                    break;
                }
            }
        }

        handleInverses(row, md, dialect);

        return affectedRows;
    }

    public String getRealTableName(Map<String, Object> maps){
        String tableName = maps.get("tableName").toString();
        if(!SystemConfig.Dataflow.getEnabled()){
            return tableName;
        }
        JSONArray partitionTables = SystemConfig.Dataflow.getPartitionTables();
        for(Object item : partitionTables){
            if(item.toString().equalsIgnoreCase(tableName)){
                String year = DateTime.now().toString("yyyy");
                return tableName + "_" + year;
            }
        }
        return tableName;
    }

    /**
     * 得到父类表的子表名字集合
     */
    public List<String> getChildName(String tableName) {
        List<String> childTable = new ArrayList<>();
        //输入字段为父类表名
        if (metaMap.containsKey(tableName)) {
            //log.debug("Parents table is exist");
            HashMap<String, Object> map = metaMap.get(tableName);
            Map<String, Object> childmap = (LinkedHashMap<String, Object>) map.get("subclasses");
            if (childmap != null) {
                //log.debug("Get all child tables");
                // 得到全部的也就是子表名字
                Set<String> keys = childmap.keySet();
                childTable.addAll(keys);
            }
        }
        return childTable;
    }

    /**
     * 检查继承关系
     */
    private boolean hasParent(String entity, String pEntityName) {
        if (entityLiftMap.containsKey(entity)) {
            return true;
        }
        while (entityLiftMap.containsKey(entity)) {
            String pEntity = entityLiftMap.get(entity);
            if (pEntity.equals(pEntityName)) {
                return true;
            }
            entity = pEntity;
        }
        return false;
    }

    private Pair rawJdbcUpdate(String sql) throws Exception {
        Pair pair = new Pair("", "");
        Connection con = ((SessionImpl) sessionPool.getSession()).connection();
        PreparedStatement statement = con.prepareStatement(sql, Statement.RETURN_GENERATED_KEYS);
        int affectedRows = statement.executeUpdate();
        pair.col = affectedRows + "";
        if (affectedRows == 0) {
            return pair;
        } else {
            ResultSet rs = statement.getGeneratedKeys();
            if (rs.next()) {
                pair.type = rs.getLong(1) + "";
            } else {
                throw new Exception("得到插入记录id失败。");
            }
            rs.close();
        }
        statement.close();
        return pair;
    }

    private int SQLUpdate(String sql) {
        SQLQuery query = sessionPool.getSession().createSQLQuery(sql);
        return query.executeUpdate();
    }

    private String query(String sql) {
        SQLQuery query = sessionPool.getSession().createSQLQuery(sql);
        return query.list().get(0) + "";
    }

    private String getLastSeqId(String seq) {
        return query("select " + seq + ".nextval newid from dual");
    }


    private String normalizeVer(String dialect, Object verValue, String verType) {
        if (verType.equals(EntityServer.COL_TIME)) {
            if (dialect.equals(DIALECT_ORACLE)) {
                return "current_timestamp";
            } else {
                SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
                return "'" + sdf.format(new Date()) + "'";
            }
        } else if (verValue == null) {
            return "1";
        } else {
            return (Integer.parseInt(verValue + "") + 1) + "";
        }
    }

    /**
     * normalizeValue
     */
    private String normalizeValue(String dialect, Object value, String valType) {
        if (value == null) {
            return "null";
        }
        switch (valType) {
            case EntityServer.COL_STRING:
                return "'" + value + "'";
            case EntityServer.COL_NUMBER:
                if ("".equals(value)) {
                    return "null";
                }
                return String.valueOf(value);
            case EntityServer.COL_BOOLEAN:
                if (dialect.equals(EntityServer.DIALECT_ORACLE)) {
                    return "'" + ((Boolean) value ? "Y" : "N") + "'";
                } else {
                    //该if是为了解决在SQLServer下保存数据时可能会导致Byte转Boolean时失败
                    if (value instanceof Byte) {
                        return value.toString();
                    } else if (value instanceof Integer) {
                        return value.toString();
                    } else {
                        return (Boolean) value ? "1" : "0";
                    }
                }
            case EntityServer.COL_DATE:
                if (dialect.equals(EntityServer.DIALECT_ORACLE)) {
                    return "TO_DATE(SUBSTR('" + value + "', 1, 10), 'YYYY-MM-DD')";
                } else {
                    return "'" + value + "'";
                }
            case EntityServer.COL_TIME:
                if (dialect.equals(EntityServer.DIALECT_ORACLE)) {
                    return "TO_TIMESTAMP('" + value + "', 'YYYY-MM-DD HH24:MI:SS')";
                } else {
                    return "'" + value + "'";
                }
            case EntityServer.COL_CLOB:
                if (dialect.equals(EntityServer.DIALECT_ORACLE)) {
                    StringBuilder sql = new StringBuilder();
                    String[] parts = StrUtil.split(value.toString(), 2000);
                    for(String part : parts){
                        sql.append("TO_CLOB('").append(part).append("')||");
                    }
                    int length = sql.length();
                    sql.delete(length - 2, length);
                    return sql.toString();
                } else {
                    return "'" + value + "'";
                }
            default:
                throw new WebException(504, "传递数据类型出错");
        }
    }


    /**
     * preload all meta data
     */
    public void loadMetaData() {
        //获取所有实体元数据
        Map<String, ClassMetadata> map = sessionFactory.getAllClassMetadata();
        //获取每个实体的元数据
        for (String entityName : map.keySet()) {
            try {
                this.getMetaData(entityName);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        //跟踪实体的继承关系
        SAXReader reader = new SAXReader();
        // 加载根目录下的subclass
        if (SqlMapper.class.getClassLoader().getResourceAsStream("subclass.xml") != null) {
            InputStream input = SqlMapper.class.getClassLoader().getResourceAsStream("subclass.xml");
            loadSubclass(reader, input);
        }

        // 加载模块下的subclass
        // 解析subclass.xml文件 获取模块
        Map<String, Map<String, String>> maps = ModuleMapper.getSubClassMap();
        for (String moduleName : maps.keySet()) {
            // module要添加路径分隔符
            if (!"".equals(moduleName)) {
                moduleName += "/";
            }
            if (SqlMapper.class.getResourceAsStream("/" + moduleName + "subclass.xml") == null) {
                log.debug(moduleName + "模块下无subclass.xml文件");
                continue;
            }
            InputStream input = SqlMapper.class.getClassLoader().getResourceAsStream(moduleName + "subclass.xml");
            loadSubclass(reader, input);
        }

        //上面处理的都是正向关系，下面处理反向多对一关系
        //find out inverse many to one relation
        for (String entityName : map.keySet()) {
            try {
                HashMap<String, Object> hash = metaMap.get(entityName);
                if (hash != null) {
                    ClassMetadata cmd = sessionFactory.getClassMetadata(entityName);
                    HashMap<String, String> inverses = new HashMap<>(16);
                    hash.put("inverses", inverses);
                    HashMap<String, Pair> inverseMap = new HashMap<>(16);
                    hash.put("inverseid", inverseMap);
                    for (String property : cmd.getPropertyNames()) {
                        Type type = cmd.getPropertyType(property);
                        //handle inverse relationship
                        if (type instanceof ManyToOneType) {
                            ManyToOneType mto = (ManyToOneType) type;
                            String linkedEntity = mto.getAssociatedEntityName();
                            String colName = mto.getAssociatedJoinable((SessionFactoryImplementor) sessionFactory).getKeyColumnNames()[0];
                            inverses.put(property, linkedEntity);
                            inverseMap.put(property, new Pair(colName, (String) metaMap.get(linkedEntity).get("idType")));
                        }
                    }
                } else {
                    log.warn("未处理【" + entityName + "】的反向多对一关系，可能含有不支持的映射结构");
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        //log.debug("meta data all loaded.");
    }

    private void loadSubclass(SAXReader reader, InputStream input) {
        try {
            Document document = reader.read(input);
            Element root = document.getRootElement();
            for (Iterator it = root.elementIterator("entity"); it.hasNext(); ) {
                Element elm = (Element) it.next();
                String entityName = elm.attribute("name").getValue();
                String parentEntity = elm.attribute("parentEntity").getValue();
                String discriminator = elm.attribute("discriminator").getValue();
                String discProperty = elm.attribute("discProperty").getValue();
                if (metaMap.containsKey(parentEntity)) {
                    HashMap<String, Object> em = metaMap.get(parentEntity);
                    LinkedHashMap<String, Pair> subclasses;
                    if (em.containsKey("subclasses")) {
                        subclasses = (LinkedHashMap<String, Pair>) em.get("subclasses");
                    } else {
                        subclasses = new LinkedHashMap<>();
                        em.put("subclasses", subclasses);
                    }
                    subclasses.put(entityName, new Pair(discProperty, discriminator));
                    entityLiftMap.put(entityName, parentEntity);
                }
            }
        } catch (DocumentException ex) {
            throw new RuntimeException(ex);
        }
    }
}
