package com.aote.queryparams;

import com.aote.entity.EntityServer;
import com.aote.exception.FileNotFoundException;
import com.aote.rs.mapper.WebException;
import com.aote.util.ResourceHelper;
import com.aote.util.ResourceType;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Iterator;

/**
 * 前后端数据交互服务
 *
 * 通过这个服务，可以实现以下功能
 *
 * 如果你要完成一个带表单和表格的页面，你不需要再前端写表单和表格，后端不需要再写SQL了
 * 只需要通过一个JSON配置文件（后期会改为可视化生成配置），即可完成以下的工作
 * > 自动生成查询，带条件查询的SQL语句以及分页用的count语句
 * > 自动生成前端动态表单配置（结合前端的X-Form组件使用）
 * @author Mr.river
 */
@Component
public class QueryParamsServer {

    private static final Logger LOGGER = Logger.getLogger(QueryParamsServer.class);

    /**
     * 根据配置文件生成查询用的SQL片段以及前端动态生成表单和表格的数据集合
     * @param source 查询配置源
     * @return tableAliasName 主表别名
     *         countSql countSQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     *         querySql querySQL片段，不能直接用于查询，需要结合getQuerySQL方法组织出完整SQL
     *         joinTableNameObject 配置文件预设的关联表集合
     *         canJoinTableNameObject 查询需要被关联的表集合
     *         selectColumn 动态生成表单查询条件的数据列集合
     *         orderBy      查询的排序字段
     *         condition    固定的查询表达式
     *         columnJson   生成的表格展示列集合
     *         formJson     生成的表单项集合
     */
    public static JSONObject getQueryParamsInfo(String source) {
        JSONObject realQueryParamsMap = new JSONObject(source);
        // 组织querySQL
        StringBuilder sql = new StringBuilder("SELECT ");
        // 组织countSQL
        StringBuilder countSql = new StringBuilder("SELECT COUNT(0) n");
        // 查询的主表名称
        String tableName = realQueryParamsMap.getString("tableName");
        // 查询的主表别名
        String tableAliasName = tableName.substring(tableName.indexOf(' ') + 1);
        // 配置文件预设的关联表集合
        JSONObject joinTableNameObject = realQueryParamsMap.getJSONObject("joinArray");
        // 配置文件的数据列集合
        JSONArray queryColumnArray = realQueryParamsMap.getJSONArray("column");
        // 组织查询需要被关联的表集合
        JSONObject canJoinTableNameObject = new JSONObject();
        // 组织动态生成表单查询条件的数据列集合
        JSONObject selectColumnArray = new JSONObject();
        // 组织生成的表格展示列集合
        JSONArray showColumnArray = new JSONArray(queryColumnArray.length());
        // 组织生成的表单项集合
        JSONArray formItemArray = new JSONArray(queryColumnArray.length());
        // 遍历数据列集合
        for (Object columnItem : queryColumnArray) {
            JSONObject column = (JSONObject) columnItem;
            // 获取该列的数据字段（表别名.字段名）
            String key = column.getString("key");
            // 获取该列的数据字段（字段名）
            int index = key.indexOf('.');
            if(index == -1){
                index = 0;
            } else {
                index = index + 1;
            }
            String realKey = key.substring(index);
            // 获取该列的数据模式
            DataModeEnum dataModeEnum;
            if(column.has("dataMode")){
                String dataMode = column.getString("dataMode");
                dataModeEnum = DataModeEnum.toType(dataMode);
                if(dataModeEnum == null){
                    throw new WebException(500, "列["+key+"]无法被处理，因为这个数据模式不存在："+ dataMode);
                }
            } else {
                dataModeEnum = DataModeEnum.ALL;
            }
            // 生成SQL查询项集合
            if (dataModeEnum == DataModeEnum.ALL || dataModeEnum == DataModeEnum.TABLE ||
                    dataModeEnum == DataModeEnum.ONLY_QUERY || dataModeEnum == DataModeEnum.TABLE_FORM) {
                // 获取这个数据字段对应的表别名
                String joinTableAlias = key.substring(0, key.indexOf('.'));
                String columnName;
                if(column.has("default")){
                    String dialect = EntityServer.findDialect();
                    String defaultValue = String.valueOf(column.get("default"));
                    switch (dialect) {
                        case EntityServer.DIALECT_SQLSERVER:
                        default:
                            columnName = "ISNULL(" + key + ",'" + defaultValue + "') " + joinTableAlias + "_" + realKey;
                            break;
                        case EntityServer.DIALECT_MYSQL:
                            columnName = "IFNULL(" + key + ",'" + defaultValue + "') " + joinTableAlias + "_" + realKey;
                            break;
                        case EntityServer.DIALECT_ORACLE:
                            columnName = "NVL(" + key + ",'" + defaultValue + "') " + joinTableAlias + "_" + realKey;
                            break;
                    }
                } else {
                    columnName = key + " " + joinTableAlias + "_" + realKey;
                }
                // 追加数据列到生成的查询SQL片段中
                sql.append("\n\t").append(columnName).append(",");
                // 根据数据列情况追加JOIN表达式
                // useJoinByCount:false 用于查询的子表数据列在生成count语句时不需要join
                putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, false);
            }
            // 生成表单查询条件集合
            if (dataModeEnum == DataModeEnum.ALL || dataModeEnum == DataModeEnum.FORM ||
                    dataModeEnum == DataModeEnum.TABLE_FORM || dataModeEnum == DataModeEnum.ONLY_CONDITION){
                // 获取该列的查询类型
                QueryTypeEnum queryTypeEnum;
                if(column.has("queryType")){
                    String queryType = column.getString("queryType");
                    queryTypeEnum = QueryTypeEnum.toType(queryType);
                    if(queryTypeEnum == null){
                        throw new WebException(500, "列["+key+"]无法生成表单项，因为这个表单查询类型不存在："+ queryType);
                    }
                } else {
                    queryTypeEnum = QueryTypeEnum.EQUALS;
                }
                // 组织动态生成表单查询条件的数据列集合
                JSONObject selectColumnItem = new JSONObject();
                selectColumnItem.put("queryType", queryTypeEnum.getValue());
                selectColumnItem.put("key", key);
                selectColumnArray.put(key, selectColumnItem);
            }
            // 生成表格列集合
            if (dataModeEnum == DataModeEnum.ALL || dataModeEnum == DataModeEnum.TABLE ||
                    dataModeEnum == DataModeEnum.ONLY_TABLE || dataModeEnum == DataModeEnum.TABLE_FORM){
                // 组织表格展示列
                JSONObject item = new JSONObject();
                item.put("title", column.getString("title"));
                item.put("dataIndex", realKey);
                // 指定宽度
                int width = 0;
                JSONObject slot;
                if (!column.has("slot")){
                    // 如果没有插槽，设置一个默认的文本溢出省略(length:16)
                    slot = new JSONObject();
                    slot.put("type","ellipsis");
                    slot.put("value",16);
                } else {
                    slot = column.getJSONObject("slot");
                }
                JSONObject scopedSlots = new JSONObject();
                scopedSlots.put("customRender",realKey);
                item.put("scopedSlots", scopedSlots);
                item.put("slots", scopedSlots);
                if(slot.has("type")){
                    // 插槽类型
                    String slotType = slot.getString("type");
                    SlotTypeEnum slotTypeEnum = SlotTypeEnum.toType(slotType);
                    if(slotTypeEnum == null){
                        throw new WebException(500, "列["+key+"]无法适用插槽，因为这个插槽类型不存在："+ slotType);
                    }
                    item.put("slotType",slotType);
                    switch (slotTypeEnum){
                        case DATE_TIME:
                            width = 160;
                            break;
                        case ELLIPSIS:
                            int ellipsisValue = slot.getInt("value");
                            BigDecimal b = new BigDecimal(String.valueOf(ellipsisValue * 7 + 20 + 32));
                            width = b.setScale(0, BigDecimal.ROUND_UP).intValue();
                            item.put("slotValue",ellipsisValue);
                            break;
                        case BADGE:
                            item.put("slotKeyMap",String.valueOf(slot.get("keyMap")));
                            break;
                    }
                } else {
                    throw new WebException(500, "列["+key+"]无法适用插槽，因为没有指定插槽类型参数");
                }
                if (column.has("width")){
                    width = column.getInt("width");
                }
                if(width > 0){
                    item.put("width",width);
                }
                // 如果有固定列
                if (column.has("fixed")) {
                    String fixedType = column.getString("fixed");
                    ColumnFixedTypeEnum columnFixedTypeEnum = ColumnFixedTypeEnum.toType(fixedType);
                    if(columnFixedTypeEnum == null){
                        throw new WebException(500, "列["+key+"]无法适用固定列，因为这个固定列类型不存在："+ fixedType);
                    }
                    item.put("fixed",fixedType);
                }
                showColumnArray.put(item);
            }
            // 生成表单项集合
            if (dataModeEnum == DataModeEnum.ALL || dataModeEnum == DataModeEnum.FORM ||
                    dataModeEnum == DataModeEnum.ONLY_FORM){
                // 获取该列的表单类型
                FormTypeEnum formTypeEnum;
                if(column.has("formType")){
                    String formType = column.getString("formType");
                    formTypeEnum = FormTypeEnum.toType(formType);
                    if(formTypeEnum == null){
                        throw new WebException(500, "列["+key+"]无法生成表单项，因为这个表单类型不存在："+ formType);
                    }
                } else {
                    formTypeEnum = FormTypeEnum.INPUT;
                }
                // 组织表单项
                JSONObject item = new JSONObject();
                item.put("model",key);
                item.put("type",formTypeEnum.getValue());
                item.put("name",column.getString("title"));
                if(column.has("selectKeyName")){
                    // 获取这个数据字段对应的表别名
                    String selectKeyName = column.getString("selectKeyName");
                    String joinTableAlias = selectKeyName.substring(0, selectKeyName.indexOf('.'));
                    // 获取该列的数据字段（字段名）
                    index = key.indexOf('.');
                    if(index == -1){
                        index = 0;
                    } else {
                        index = index + 1;
                    }
                    String realSelectKey = key.substring(index);
                    if(!selectKeyName.equals(key)){
                        // 追加数据列到生成的查询SQL片段中
                        sql.append("\n\t").append(key).append(" ").append(joinTableAlias).append("_").append(realSelectKey).append(",");
                        // 组织动态生成表单查询条件的数据列集合
                        JSONObject selectColumnItem = new JSONObject();
                        selectColumnItem.put("queryType", QueryTypeEnum.EQUALS.getValue());
                        selectColumnItem.put("key", selectKeyName);
                        selectColumnArray.put(selectKeyName, selectColumnItem);
                        // 修改表单项的model
                        item.put("model",selectKeyName);
                    }
                }
                if(column.has("selectKey")){
                    Object selectKey = column.get("selectKey");
                    if(selectKey instanceof JSONArray){
                        item.put("keys", selectKey);
                    } else {
                        item.put("keyName", selectKey);
                    }
                }
                if(column.has("rule")) {
                    JSONObject ruleObject = column.getJSONObject("rule");
                    // 获取该列的表单校验类型
                    RuleTypeEnum ruleTypeEnum;
                    if(ruleObject.has("type")){
                        String ruleType = ruleObject.getString("type");
                        ruleTypeEnum = RuleTypeEnum.toType(ruleType);
                        if(ruleTypeEnum == null){
                            throw new WebException(500, "列["+key+"]无法生成表单项，因为这个表单校验类型不存在："+ ruleType);
                        }
                    } else {
                        ruleObject.put("type",RuleTypeEnum.STRING.getValue());
                    }
                    item.put("rule", ruleObject);
                }
                if(column.has("placeholder")){
                    item.put("placeholder",column.getString("placeholder"));
                }
                formItemArray.put(item);
            }
        }
        // 取得OrderBy配置字段的表别名，判断是否需要Join
        String orderBy = realQueryParamsMap.getString("orderBy");
        String joinTableAlias = orderBy.substring(0, orderBy.indexOf('.'));
        // 根据OrderBy的列情况追加JOIN表达式，用于OrderBy的数据列在生成count语句时不需要join
        putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, false);
        // 如果使用了固定查询表达式，则追加该表达式
        String conditionStr;
        if(realQueryParamsMap.has("condition")){
            JSONObject condition = realQueryParamsMap.getJSONObject("condition");
            conditionStr = condition.getString("value");
            if(condition.has("join")){
                JSONArray joinArray = condition.getJSONArray("join");
                for(Object item : joinArray){
                    String joinItem = item.toString();
                    // 追加JOIN表达式
                    // 在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                    putJoinCondition(joinItem, tableAliasName, joinTableNameObject, canJoinTableNameObject, true);
                }
            }
        } else {
            conditionStr = "1=1";
        }
        // 移除查询列中最后多余的逗号
        sql.deleteCharAt(sql.length() - 1);
        // 追加SQL片段到querySQL中
        sql.append("\n").append("FROM ").append(tableName);
        // 追加SQL片段到countSQL中
        countSql.append("\n").append("FROM ").append(tableName);
        JSONObject result = new JSONObject();
        result.put("tableAliasName", tableAliasName);
        result.put("countSql",countSql.toString());
        result.put("querySql",sql.toString());
        result.put("joinTableNameObject",joinTableNameObject);
        result.put("canJoinTableNameObject",canJoinTableNameObject);
        result.put("selectColumn",selectColumnArray);
        result.put("orderBy",orderBy);
        result.put("condition",conditionStr);
        result.put("columnJson",showColumnArray);
        result.put("formJson",formItemArray);
        return result;
    }

    /**
     * 追加JOIN表达式
     * @param joinTableAlias 关联的表别名
     * @param tableAliasName 主表的别名
     * @param joinTableNameObject 配置文件预设的关联表集合
     * @param canJoinTableNameObject 查询需要被关联的表集合
     * @param useJoinByCount 生成count语句时是否需要join
     */
    public static void putJoinCondition(String joinTableAlias, String tableAliasName, JSONObject joinTableNameObject,
                                        JSONObject canJoinTableNameObject, Boolean useJoinByCount){
        // 如果数据字段是主表的，直接跳过
        if (joinTableAlias.equals(tableAliasName)) {
            return;
        }
        // 如果数据字段对应的表别名在配置文件预设的关联表集合中
        if (joinTableNameObject.has(joinTableAlias)) {
            JSONObject params;
            // 如果该表别名不在查询需要被关联的表集合中
            if (!canJoinTableNameObject.has(joinTableAlias)) {
                // 取到JOIN表达式
                String joinCondition = joinTableNameObject.getString(joinTableAlias).toLowerCase();
                // 判断该JOIN表达式是否需要处理多表关联的情况
                String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
                // 如果返回了需要额外关联的表别名
                if (shouldJoinTableAlias != null) {
                    // 递归处理JOIN表达式
                    putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, useJoinByCount);
                }
                params = new JSONObject();
                params.put("useJoinByCount",useJoinByCount);
                // 默认使用LEFT JOIN，加入到需要被关联的表集合中
                params.put("value", "LEFT JOIN " + joinCondition);
                canJoinTableNameObject.put(joinTableAlias,params);
            } else {
                // 更新这个JOIN表达式在生成count语句时的JOIN策略
                params = canJoinTableNameObject.getJSONObject(joinTableAlias);
                params.put("useJoinByCount",useJoinByCount);
                // 如果这个JOIN表达式在生成count语句时需要JOIN，则需要判断是否有多表关联的情况
                if(useJoinByCount){
                    // 取到JOIN表达式的内容
                    String joinCondition = params.getString("value").toLowerCase();
                    // 判断该JOIN表达式是否需要处理多表关联的情况
                    String shouldJoinTableAlias = getShouldJoinTableAlias(joinTableAlias, tableAliasName, joinCondition);
                    // 如果返回了需要额外关联的表别名
                    if(shouldJoinTableAlias != null){
                        // 则更新 这个表别名对应的JOIN表达式在生成count语句时的JOIN策略
                        params = canJoinTableNameObject.getJSONObject(shouldJoinTableAlias);
                        params.put("useJoinByCount", true);
                        // 递归处理JOIN表达式
                        putJoinCondition(shouldJoinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, true);
                    }
                }
            }
        } else {
            throw new WebException(500,"在组织JOIN表达式时，没有找到关于表别名[" + joinTableAlias + "]的JOIN表达式配置信息");
        }
    }

    /**
     * 判断该JOIN表达式是否需要处理多表关联的情况
     *
     *  因为不是所有业务都是主表关联子表，还有子表A关联子表B的情况，此时子表B可能还没有加入到需要被关联的表集合中
     *  所以此处需要进行判断，看是否需要处理多表关联的情况
     *  依次拿到JOIN表达式的第一个表别名和第二个表别名，看是不是自身表的别名或者是主表别名，如果不是，则代表有多表关联的情况
     *  如果有多表关联的情况，该方法会返回需要额外关联的表别名，可以再通过putJoinCondition方法追加JOIN表达式
     *
     * @param joinTableAlias 关联的表别名
     * @param tableAliasName 主表的别名
     * @param joinCondition JOIN表达式
     * @return 需要关联的表别名
     */
    private static String getShouldJoinTableAlias(String joinTableAlias, String tableAliasName, String joinCondition){
        String onCondition = joinCondition.substring(joinCondition.indexOf("on") + 3);
        // 第一个表别名
        String canJoinTableAliasA = onCondition.substring(0,onCondition.indexOf("."));
        // 第二个表别名
        String canJoinTableAliasB = onCondition.substring(onCondition.lastIndexOf("=") + 1, onCondition.lastIndexOf(".")).trim();
        // 如果两个别名没有一个是主表别名，那么说明要处理多表关联情况
        if (!canJoinTableAliasA.equals(tableAliasName) && !canJoinTableAliasB.equals(tableAliasName)) {
            String shouldJoinTableAlias;
            // 如果别名A是传入的表别名，则取别名B
            if (canJoinTableAliasA.equals(joinTableAlias)) {
                shouldJoinTableAlias = canJoinTableAliasB;
            } else {
                shouldJoinTableAlias = canJoinTableAliasA;
            }
            return shouldJoinTableAlias;
        } else {
            return null;
        }
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句
     * @param queryParamsName 配置文件资源名称
     * @param params 前端传入的表单参数
     * @return 完整可执行的querySQL和countSQL语句
     */
    public static JSONObject getQuerySQL(String queryParamsName, JSONObject params) throws Exception {
        long begin = System.currentTimeMillis();
        String path = QueryParamsMapper.getQueryParams(queryParamsName);
        if (path == null) {
            throw new WebException(500,"查询参数文件未找到: " + queryParamsName);
        }
        JSONObject queryParamsMap;
        try {
            queryParamsMap = (JSONObject) ResourceHelper.getString(ResourceType.QUERY_PARAMS,queryParamsName,path);
        } catch (FileNotFoundException e) {
            throw new WebException(500,path + ".文件无配置");
        } catch (IOException io) {
            throw new RuntimeException(io);
        }
        long end = System.currentTimeMillis();
        LOGGER.debug("读取查询参数文件，总耗时："+ (end - begin));
        // 获取主表的别名
        String tableAliasName = queryParamsMap.getString("tableAliasName");
        // 获取querySql和countSql片段
        StringBuilder querySql = new StringBuilder(queryParamsMap.getString("querySql"));
        StringBuilder countSql = new StringBuilder(queryParamsMap.getString("countSql"));
        // 获取配置文件预设的关联表集合
        JSONObject joinTableNameObject = queryParamsMap.getJSONObject("joinTableNameObject");
        // 获取查询时需要被关联的表集合
        JSONObject canJoinTableNameObject = queryParamsMap.getJSONObject("canJoinTableNameObject");
        // 获取查询的排序字段
        String orderBy = queryParamsMap.getString("orderBy");
        // 获取固定的查询表达式
        String conditionStr = queryParamsMap.getString("condition");
        // 根据前端传入的表单参数，结合动态生成表单查询条件所用的数据列集合，生成where表达式
        String whereCondition = getValue(queryParamsMap.getJSONObject("selectColumn"),
                params,tableAliasName,joinTableNameObject,canJoinTableNameObject,conditionStr);
        long end2 = System.currentTimeMillis();
        LOGGER.debug("组织表单查询条件，总耗时："+ (end2 - begin));
        // 取出需要JOIN的子表集合
        Iterator<String> canJoinTableIterator = canJoinTableNameObject.keys();
        while (canJoinTableIterator.hasNext()){
            JSONObject joinObject = canJoinTableNameObject.getJSONObject(canJoinTableIterator.next());
            String condition = joinObject.getString("value");
            // 追加需要JOIN的表达式到SQL中
            querySql.append("\n\t").append(condition);
            // 根据需要在countSQL中追加join表达式
            if (joinObject.getBoolean("useJoinByCount")){
                countSql.append("\n\t").append(condition);
            }
        }
        // 追加where表达式到SQL中
        querySql.append("\n").append("WHERE ").append(whereCondition);
        countSql.append("\n").append("WHERE ").append(whereCondition);
        // 追加orderBy表达式到SQL中
        querySql.append("\nORDER BY ").append(orderBy);
        JSONObject result = new JSONObject();
        result.put("querySql",querySql.toString());
        result.put("countSql",countSql.toString());
        long end3 = System.currentTimeMillis();
        LOGGER.debug("得到结果，总耗时："+ (end3 - begin));
        return result;
    }

    /**
     * 生成完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     * @param queryParamsName 配置文件资源名称
     * @param params 前端传入的表单参数
     * @return 完整可执行的querySQL和countSQL语句，以及查询列，表名，条件表达式的SQL片段
     */
    public static JSONObject getSignleStyleQuerySQL(String queryParamsName, JSONObject params) throws Exception {
        JSONObject queryParams = getQuerySQL(queryParamsName, params);
        String querySql = queryParams.getString("querySql");
        String countSql = queryParams.getString("countSql");
        JSONObject result = new JSONObject();
        result.put("items",querySql.substring(querySql.indexOf("SELECT") + 6,querySql.indexOf("FROM")));
        result.put("tableName",querySql.substring(querySql.indexOf("FROM") + 4,querySql.indexOf("WHERE")));
        result.put("condition",querySql.substring(querySql.indexOf("WHERE") + 5));
        result.put("countItems",countSql.substring(countSql.indexOf("SELECT") + 6,countSql.indexOf("FROM")));
        result.put("countTableName",countSql.substring(countSql.indexOf("FROM") + 4,countSql.indexOf("WHERE")));
        result.put("countCondition",countSql.substring(countSql.indexOf("WHERE") + 5));
        result.put("querySql", querySql);
        result.put("countSql", countSql);
        return result;
    }

    /**
     * 生成查询条件
     * @param queryMap 动态生成表单查询条件所用的数据列集合
     * @param params 前端传入的表单参数
     * @param tableAliasName 主表的别名
     * @param joinTableNameObject 获取配置文件预设的关联表集合
     * @param canJoinTableNameObject 获取查询时需要被关联的表集合
     * @param conditionStr 固定查询表达式
     * @return 查询条件的SQL表达式
     */
    public static String getValue(JSONObject queryMap, JSONObject params, String tableAliasName,
                                  JSONObject joinTableNameObject, JSONObject canJoinTableNameObject, String conditionStr) throws Exception {
        // 查询条件
        StringBuilder condition = new StringBuilder(conditionStr);
        // 遍历前端传入的表单参数
        Iterator<String> iterator = params.keys();
        while(iterator.hasNext()) {
            String key = iterator.next();
            String value = String.valueOf(params.get(key));
            // 如果前端传入的表单参数有值
            if (value.length() != 0 && (!value.equals("全部"))) {
                // 如果数据列集合中包含该列
                if (queryMap.has(key)) {
                    JSONObject queryParams = queryMap.getJSONObject(key);
                    // 获取查询类型
                    String queryTypeStr = queryParams.getString("queryType");
                    if(!QueryTypeEnum.is(queryTypeStr)) {
                        throw new WebException(500, "表单字段[" + key + "]无法生成查询条件，因为配置的查询类型不存在：" + queryTypeStr);
                    } else {
                        QueryTypeEnum queryType = QueryTypeEnum.toType(queryTypeStr);
                        // 获取该列名称（表别名.字段名）
                        String realKey = queryParams.getString("key");
                        // 拼接条件表达式
                        switch (queryType) {
                            case EQUALS:
                            case NO_EQUALS:
                            case LESS_THAN:
                            case LESS_THAN_EQUALS:
                            case GREATER_THAN:
                            case GREATER_THAN_EQUALS:
                                String mark = queryType.getValue();
                                if(value.equalsIgnoreCase("NULL")){
                                    condition.append("\n\tAND ").append(realKey).append(" IS NULL");
                                } else {
                                    condition.append("\n\tAND ").append(realKey).append(" ").append(mark).append(" '").append(value).append("'");
                                }
                                break;
                            case INNER_LIKE:
                                condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("%'");
                                break;
                            case LEFT_LIKE:
                                condition.append("\n\tAND ").append(realKey).append(" LIKE '%").append(value).append("'");
                                break;
                            case RIGHT_LIKE:
                                condition.append("\n\tAND ").append(realKey).append(" LIKE '").append(value).append("%'");
                                break;
                            case IN:
                                condition.append("\n\tAND ").append(realKey).append(" IN (").append(value).append(")");
                                break;
                            case NOT_IN:
                                condition.append("\n\tAND ").append(realKey).append(" NOT IN (").append(value).append(")");
                                break;
                            case BETWEEN:
                                JSONArray array = new JSONArray(value);
                                condition.append("\n\tAND ").append(realKey).append(" BETWEEN '").append(array.get(0)).append("' AND '").append(array.get(1)).append("'");
                                break;
                        }
                        // 追加JOIN表达式
                        // 在执行countSQL语句时,用于查询条件的子表数据列需要JOIN子表
                        // 获取这个数据字段对应的表别名
                        String joinTableAlias = realKey.substring(0, realKey.indexOf('.'));
                        putJoinCondition(joinTableAlias, tableAliasName, joinTableNameObject, canJoinTableNameObject, true);
                    }
                } else {
                    throw new WebException(500, "表单字段["+key+"]无法生成查询条件，因为没有配置这个字段的相关信息");
                }
            }
        }
        return condition.toString();
    }
}
