package com.aote.transaction;

import com.aote.ThreadResource;
import javassist.NotFoundException;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * session池，用于自己对hibernate的session进行管理，特别是前台控制事务的情况。
 */
@Component
public class SessionPool {
    @Autowired
    private SessionFactory factory;

    @Autowired
    private ThreadSession tSession;

    // 缓存的session，包括session及其事务
    private InnerSession[] sessions = new InnerSession[100];

    public static SessionPool getInstance() {
        return instance;
    }

    private static SessionPool instance;

    private SessionPool() {
        instance = this;
    }

    /**
     *  创建session，把新建的session添加到缓存中
     * @return
     */
    public int createSession() throws NotFoundException {
        Session session = this.factory.openSession();
        Transaction trans = session.beginTransaction();
        InnerSession innerSession = new InnerSession(session, trans);

        // 找到空闲位置，把session添加进去
        synchronized (this.sessions) {
            for(int i = 0; i < sessions.length; i++) {
                if (sessions[i] == null) {
                    sessions[i] = innerSession;
                    return i;
                }
            }
        }

        throw new NotFoundException("没有空闲位置");
    }

    /**
     * 获取第i项session
     * @param i 第i项session
     * @return session
     */
    public Session getSession(int i) {
        return this.sessions[i].session;
    }

    /**
     * 保存session及事务，方便事务提交或回滚
     */
    private class InnerSession {
        public Session session;
        public Transaction transaction;

        public InnerSession(Session session, Transaction transaction) {
            this.session = session;
            this.transaction = transaction;
        }
    }

    /**
     * 获得当前线程session，如果线程局部变量有sessionid，从sessionpool中取，否则，从sessionfactory中取。
     * @return
     */
    public Session getSession() {
        //如果线程中有session，直接取线程中的
        Session threadSession = tSession.get();
        if(threadSession != null){
            return threadSession;
        }
        // 如果请求头有session，根据请求头里的sessionid获得对应session，放到事务管理器中
        Integer sessionId = ThreadResource.SessionId.get();
        if (sessionId != null) {
            int i = sessionId;
            return this.getSession(i);
        }
        return factory.getCurrentSession();
    }
}
