package com.aote.rs;

import com.aote.exception.ExceptionLogManage;
import com.aote.sql.SqlServer;
import com.aote.util.*;
import com.aote.util.other.AESUtil;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import javax.ws.rs.*;

/**
 * 提供sql查询服务
 */
@Path("sql")
@Component
@Transactional
public class SqlService {
	static Logger log = Logger.getLogger(SqlService.class);

	@Autowired
	private SqlServer sqlServer;

	@Autowired
	private ExceptionLogManage exceptionLogManage;

	/**
	 * 清理SQL源缓存
	 * @return ok
	 */
	@GET
	@Path("/clear")
	public String clearSQLCache(){
		ResourceHelper.clearLogicCache();
		ResourceHelper.clearSqlCache();
		ResourceHelper.clearPathCache();
		return "ok";
	}

	/**
	 * 获取SQL语句的合计执行结果
	 *
	 * @param name
	 *            : sql语句名
	 * @param str
	 *            : sql语句执行参数
	 */
	@POST
	@Path("{name}/n")
	public String txgetTotalCnt(@PathParam("name") String name, String str)
			throws Exception {
		try {
			return sqlEncrypt(name, str, 1);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			exceptionLogManage.writeError(ex,name,"/rs/sql/"+name+"/n");
			throw ex;
		}
	}

	/**
	 * 执行sql,对sql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 */
	@POST
	@Path("{name}")
	public String txExecute(@PathParam("name") String name,
							@QueryParam("pageNo") int pageNo,
							@QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 2, pageNo, pageSize);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			exceptionLogManage.writeError(ex,name,"/rs/sql/"+name);
			throw ex;
		}
	}

	/**
	 * 获取HQL语句的合计执行结果
	 *
	 * @param name
	 *            : hql语句名
	 * @param str
	 *            : hql语句执行参数
	 */
	@POST
	@Path("hql/{name}/n")
	public String txHqlgetTotalCnt(@PathParam("name") String name, String str)
			throws Exception {
		try {
			return sqlEncrypt(name, str, 3);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name+"/n");
			throw ex;
		}
	}

	/**
	 * 执行hql,对hql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 */
	@POST
	@Path("hql/{name}")
	public String txHqlExecute(@PathParam("name") String name,
							   @QueryParam("pageNo") int pageNo,
							   @QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 4, pageNo, pageSize);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name);
			throw ex;
		}
	}

	/**
	 * 执行hql,对hql中的参数进行替换 pageNo - 页号，默认为1 pageSize - 每页个数，默认为1000
	 * 取消大于1000时限制1000的条件
	 */
	@POST
	@Path("hql/{name}/all")
	public String txHqlExecuteAll(@PathParam("name") String name,
								  @QueryParam("pageNo") int pageNo,
								  @QueryParam("pageSize") int pageSize, String str) throws Exception {
		try {
			return sqlEncrypt(name, str, 5, pageNo, pageSize);
		} catch (Exception ex) {
			log.error(ExceptionHelper.stackToString(ex));
			exceptionLogManage.writeError(ex,name,"/rs/sql/hql/"+name+"/all");
			throw ex;
		}
	}

	/**
	 * 封装sql
	 * @param sqlName sql名
	 * @param param	参数
	 * @param method 区分调用
	 * @param value	接收分页的两个参数
	 * @return
	 */
	private String sqlEncrypt(String sqlName, String param, int method, int ... value)throws Exception{
		JSONObject paramJson = new JSONObject(param);
		JSONObject configJson = JsonHelper.getConfig();
		//验证
		int checkEncry = Util.checkEncry(paramJson, configJson);
		String encryKey = null;
		if (checkEncry == 1){
			//判断解密方式
			String type = paramJson.getString("aoteEncrypt");
			if ("AES".equals(type)) {
				assert configJson != null;
				encryKey = configJson.getString("encryKey");
				assert encryKey != null;
				String data = paramJson.getString("data");
				param = AESUtil.decrypt(data, encryKey);
			} else {
				throw new RuntimeException("不支持的解密类型！" + type);
			}
		}
		//sql注入标志位
		boolean sqlInject = false;
		if(configJson != null && configJson.has("sqlInject")){
			sqlInject = configJson.getBoolean("sqlInject");
		}
		// 验证参数是否合法
		if (sqlInject) {
			ParamFilter.checkSqlJsonStr(param);
		}
		String result;
		if (method == 1) {
			result = sqlServer.queryTotal(sqlName, param).toString();
		} else if (method == 2) {
			result = sqlServer.query(sqlName, value[0], value[1], param).toString();
		} else if (method == 3) {
			result = sqlServer.queryHqlTotal(sqlName, param).toString();
		} else if (method == 4) {
			result = sqlServer.hqlQuery(sqlName, value[0], value[1], param).toString();
		} else {
			result = sqlServer.hqlQueryAll(sqlName, value[0], value[1], param).toString();
		}
		if (checkEncry == 1){
			result = AESUtil.encrypt(result,encryKey);
		}
		return result;
	}
}
