package com.aote.redis;

import com.aote.rs.mapper.WebException;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.data.redis.connection.jedis.JedisConnectionFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.data.redis.core.script.RedisScript;
import org.springframework.data.redis.serializer.GenericToStringSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.stereotype.Component;
import redis.clients.jedis.JedisPoolConfig;

import java.util.*;

/**
 * Redis操作工具
 * 具有分布式锁
 *
 * @author Mr.river
 */
@Component
public class RedisUtil implements RedisService{
    //redis操作类
    private static RedisTemplate<String, Object> redisTemplate;
    private static Logger log = Logger.getLogger(RedisUtil.class);

    //分布式锁脚本
    private static RedisScript<Boolean> lockScript;
    static {
        //初始化脚本
        String newSetIfAbsentScriptStr = " if 1 == redis.call('setnx',KEYS[1],ARGV[1]) then" +
                " redis.call('expire',KEYS[1],ARGV[2])" +
                " return 1;" +
                " else" +
                " return 0;" +
                " end;";
        lockScript = new DefaultRedisScript<>(newSetIfAbsentScriptStr, Boolean.class);
        //加载redis配置文件
        JSONObject redisConfigMap = null;
        if (RedisUtil.class.getClassLoader().getResourceAsStream("/redisConfig.json") != null) {
            redisConfigMap = JsonTools.readJsonFile("/redisConfig.json");
        }
        JedisPoolConfig poolConfig = new JedisPoolConfig();
        poolConfig.setMaxIdle(200);
        poolConfig.setMaxTotal(1024);
        poolConfig.setMaxWaitMillis(1000);
        poolConfig.setTestOnBorrow(true);
        poolConfig.setTestOnReturn(true);
        JedisConnectionFactory factory = new JedisConnectionFactory(poolConfig);
        if(redisConfigMap != null){
            factory.setHostName(redisConfigMap.getString("hostName"));
            if(redisConfigMap.has("password")){
                factory.setPassword(redisConfigMap.getString("password"));
            }
            factory.setPort(redisConfigMap.getInt("port"));
        } else {
            factory.setHostName("127.0.0.1");
            factory.setPort(6379);
        }
        factory.afterPropertiesSet();
        redisTemplate = new RedisTemplate<>();
        redisTemplate.setConnectionFactory(factory);
        //创建 序列化类
        GenericToStringSerializer<Object> genericToStringSerializer = new GenericToStringSerializer<>(Object.class);
        //序列化类，对象映射设置
        //设置 value 的转化格式和 key 的转化格式
        redisTemplate.setValueSerializer(genericToStringSerializer);
        redisTemplate.setKeySerializer(new StringRedisSerializer());
        redisTemplate.afterPropertiesSet();
    }

    @Override
    public Boolean has(String key) {
        return redisTemplate.hasKey(key);
    }

    @Override
    public void set(String key, Object value) {
        set(key,value,-1);
    }

    @Override
    public void set(String key, Object value, long timeoutSec) {
        //如果已有该key
        if(this.has(key)){
            //获取锁
            if(lock(key,10)){
                //执行修改
                if(timeoutSec == -1){
                    redisTemplate.opsForValue().set(key, value);
                } else {
                    redisTemplate.opsForValue().set(key, value, timeoutSec);
                }
                //释放锁
                unLock(key);
            } else {
                log.error("修改失败，该数据受锁保护中。");
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } else {
            //执行添加
            if(timeoutSec == -1){
                redisTemplate.opsForValue().set(key, value);
            } else {
                redisTemplate.opsForValue().set(key, value, timeoutSec);
            }
        }
    }

    @Override
    public void setHash(String hashKey, JSONObject value) {
        setHash(hashKey,value.toMap());
    }

    @Override
    public void setHash(String hashKey, Map<String, Object> value) {
        //如果已有该key
        if(this.has(hashKey)){
            //获取锁
            if(lock(hashKey,10)){
                //执行修改
                redisTemplate.opsForHash().putAll(hashKey,value);
                //释放锁
                unLock(hashKey);
            } else {
                log.error("修改失败，该数据受锁保护中。");
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } else {
            //执行添加
            redisTemplate.opsForHash().putAll(hashKey,value);
        }
    }

    @Override
    public JSONObject getHash(String hashKey) {
        return new JSONObject(redisTemplate.opsForHash().entries(hashKey));
    }

    @Override
    public Object getHash(String hashKey, String key) {
        return redisTemplate.opsForHash().get(hashKey,key);
    }

    @Override
    public void setHashKey(String hashKey, String key, Object value) {
        //如果已有该key
        if(this.hasHashKey(hashKey,key)){
            //获取锁
            if(hashLock(hashKey,key,10)){
                //执行修改
                redisTemplate.opsForHash().put(key,hashKey,value);
                //释放锁
                hashUnLock(hashKey,key);
            } else {
                log.error("修改失败，该数据受锁保护中。");
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } else {
            //执行添加
            redisTemplate.opsForHash().put(key,hashKey,value);
        }
    }

    @Override
    public void deleteHashKey(String hashKey, String key) {
        redisTemplate.opsForHash().delete(hashKey,key);
    }

    @Override
    public Boolean hasHashKey(String hashKey, String key) {
        return redisTemplate.opsForHash().hasKey(hashKey,key);
    }

    @Override
    public Boolean lock(String key, long timeoutSec) {
        return setIfAbsent(key+"-lock","lock",timeoutSec);
    }

    @Override
    public Boolean hashLock(String hashKey, String key, long timeoutSec) {
        return setIfAbsent(hashKey+"-"+key+"-lock","lock",timeoutSec);
    }

    @Override
    public void unLock(String key) {
        delete(key+"-lock");
    }

    @Override
    public void hashUnLock(String hashKey, String key) {
        delete(hashKey+"-"+key+"-lock");
    }

    @Override
    public void rename(String oldKey, String newKey) {
        redisTemplate.rename(oldKey, newKey);
    }

    @Override
    public Object get(String key) {
        return redisTemplate.opsForValue().get(key);
    }

    @Override
    public void delete(String key) {
        redisTemplate.delete(key);
    }

    @Override
    public void deleteList(Set<String> keys) {
        redisTemplate.delete(keys);
    }

    @Override
    public void deleteList(JSONArray keys) {
        Set<String> keySet = new HashSet<>(keys.length());
        for(int i = 0 ; i < keys.length() ; i++){
            keySet.add(keys.getString(i));
        }
        deleteList(keySet);
    }

    @Override
    public void deleteList(JSONObject keys) {
        deleteList(keys.keySet());
    }

    /**
     * setIfAbsent升级版，加了超时时间
     */
    public boolean setIfAbsent(String key,String value,long seconds){
        List<String> keys = new ArrayList<>();
        keys.add(key);
        return redisTemplate.<Boolean>execute(lockScript,keys,value,String.valueOf(seconds));
    }
}
