package com.aote.util;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.hibernate.HibernateException;
import org.hibernate.SessionFactory;
import org.hibernate.metadata.ClassMetadata;
import org.hibernate.type.*;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.springframework.util.StringUtils;

import java.io.IOException;
import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

public class JsonHelper {

	private static final ObjectMapper objectMapper;

	static {
		objectMapper = new ObjectMapper();
		objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES,false);
	}

	/**
	 * 将json转化为实体POJO
	 * @param json org.json
	 * @param obj 转换的实体类
	 * @return 实体POJO
	 */
	public static<T> T toParse(JSONObject json, Class<T> obj) {
		T t = null;
		try {
			t = objectMapper.readValue(json.toString(), obj);
		} catch (Exception e) {
			e.printStackTrace();
		}
		return t;
	}

	public static<T> List<T> toParseList(JSONArray json, Class<T> obj){
		List<T> t = null;
		try {
			t = objectMapper.readValue(json.toString(), getCollectionType(List.class, obj));
		} catch (Exception e) {
			e.printStackTrace();
		}
		return t;
	}

	public static<T> JSONArray toJsonArray(List<T> entity){
		try {
			return new JSONArray(objectMapper.writeValueAsString(entity));
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 将实体POJO转换为json
	 * @param entity 被转换的实体类
	 * @return org.json
	 */
	public static JSONObject toJSON(Object entity) {
		try {
			return new JSONObject(objectMapper.writeValueAsString(entity));
		} catch (Exception e) {
			e.printStackTrace();
		}
		return null;
	}

	/**
	 * 获取泛型的Collection Type
	 *
	 * @param collectionClass 泛型的Collection
	 * @param elementClasses  实体bean
	 * @return JavaType Java类型
	 */
	private static JavaType getCollectionType(Class<?> collectionClass, Class<?>... elementClasses) {
		return objectMapper.getTypeFactory().constructParametricType(collectionClass, elementClasses);
	}

	// 把参数格式的JSON转换成map
	@SuppressWarnings("unchecked")
	public static Map<String, Object> toMap(JSONObject object) {
		return object.toMap();
	}

	// 获取工程的config配置
	public static JSONObject getConfig(){
		if (ResourceHelper.class.getResourceAsStream("/config.json") == null) {
			return null;
		}
		try {
			return new JSONObject(ResourceHelper.getString("config.json"));
		} catch (IOException e) {
			throw new RuntimeException("获取config异常", e);
		}
	}
}
