package com.aote.redis;

import com.aote.config.SystemConfig;
import com.aote.rs.mapper.WebException;
import com.aote.util.other.JsonTools;
import org.apache.log4j.Logger;
import org.json.JSONObject;
import org.redisson.Redisson;
import org.redisson.api.RBucket;
import org.redisson.api.RLock;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.redisson.config.Config;
import org.redisson.config.SingleServerConfig;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Map;
import java.util.concurrent.TimeUnit;

/**
 * Redis操作工具
 *
 * @author Mr.river
 */
public class RedisUtil implements RedisService {
    private static volatile RedisUtil redisUtil;
    /**
     * redis操作类
     */
    private static final RedissonClient REDISSON_CLIENT;
    private static final Logger log = Logger.getLogger(RedisUtil.class);

    public static RedisUtil getInstance(){
        if(redisUtil == null) {
            synchronized (RedisUtil.class) {
                if (redisUtil == null && REDISSON_CLIENT != null) {
                    redisUtil = new RedisUtil();
                }
            }
        }
        return redisUtil;
    }

    private RedisUtil(){

    }

    static {
        if(SystemConfig.Redis.hasHostName()) {
            Config config = new Config();
            SingleServerConfig singleServerConfig = config.useSingleServer();
            singleServerConfig.setAddress("redis://"+ SystemConfig.Redis.getHostName());
            if (SystemConfig.Redis.hasPassword()) {
                singleServerConfig.setPassword(SystemConfig.Redis.getPassword());
            }
            REDISSON_CLIENT = Redisson.create(config);
            log.info("***redis实例已加载完成***");
        } else {
            REDISSON_CLIENT = null;
            log.warn("***redis实例未配置***");
        }
    }

    @Override
    public void set(String key, Object value) {
        set(key,value,-1);
    }

    @Override
    public void set(String key, Object value, long timeoutSec) {
        RBucket<String> rBucket = REDISSON_CLIENT.getBucket(key);
        if(timeoutSec > 0){
            rBucket.set(String.valueOf(value),timeoutSec, TimeUnit.SECONDS);
        } else {
            rBucket.set(String.valueOf(value));
        }
    }

    @Override
    public void setHash(String hashKey, JSONObject value) {
        setHash(hashKey, value.toMap());
    }

    @Override
    public void setHash(String hashKey, Map<String, Object> value) {
        setHash(hashKey,value,-1);
    }

    @Override
    public void setHash(String hashKey, Map<String, Object> value, long timeoutSec) {
        RMap<String, Object> rMap = REDISSON_CLIENT.getMap(hashKey);
        rMap.putAll(value);
        if(timeoutSec > 0){
            rMap.expire(timeoutSec, TimeUnit.SECONDS);
        }
    }

    @Override
    public JSONObject getHash(String hashKey) {
        return new JSONObject(REDISSON_CLIENT.getMap(hashKey));
    }

    @Override
    public Object getHash(String hashKey, String key) {
        return REDISSON_CLIENT.getMap(hashKey).get(key);
    }

    @Override
    public void setHashKey(String hashKey, String key, Object value) {
        REDISSON_CLIENT.getMap(hashKey).put(key,value);
    }

    @Override
    public void deleteHashKey(String hashKey, String key) {
        REDISSON_CLIENT.getMap(hashKey).remove(key);
    }

    @Override
    public Boolean hasHashKey(String hashKey, String key) {
        return REDISSON_CLIENT.getMap(hashKey).containsKey(key);
    }

    @Override
    public void lock(String key, Runnable runnable) {
        RLock lock = REDISSON_CLIENT.getLock("LOCK-"+key);
        boolean isLock;
        try{
            isLock = lock.tryLock(10,20,TimeUnit.SECONDS);
            if(isLock){
                runnable.success();
            } else {
                throw new WebException(505, "修改失败，该数据受锁保护中。");
            }
        } catch (InterruptedException e) {
            log.error("分布式锁"+lock.getName()+"获取失败");
            e.printStackTrace();
        } finally {
            if(lock.isHeldByCurrentThread()) {
                lock.unlock();
            }
        }
    }

    @Override
    public Object get(String key) {
        return REDISSON_CLIENT.getBucket(key).get();
    }

    @Override
    public void delete(String key) {
        REDISSON_CLIENT.getBucket(key).delete();
    }

    @Override
    public void deleteList(Iterable<Object> keys) {
        keys.forEach((item) -> delete(String.valueOf(item)));
    }

    @Override
    public void deleteList(JSONObject keys) {
        deleteList(new ArrayList<>(keys.keySet()));
    }

    @FunctionalInterface
    public interface Runnable {
        void success();
    }
}
