package com.aote.report;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.json.JSONObject;
import org.springframework.stereotype.Component;

import com.af.expression.Delegate;
import com.af.expression.Program;
import com.aote.logic.PluginMapper;
import com.aote.sql.SqlServer;
import com.aote.util.JsonHelper;

// 产生一个实际报表。
// 分两步，首先记录各单元格行、列、跨行数、跨列数，然后根据记录对所有单元个进行整理，产生table表格。
@Component
public class Report {
	public SqlServer sqlServer;

	// 报表中用到的sql查询以及查询参数
	public List<Sql> sqls = new ArrayList<Sql>();

	// 报表所有单元格以及块
	public List<Cell> cells = new ArrayList<Cell>();

	// 报表所对应的全局变量，包括sql语句执行结果
	public Map<String, Object> vars = new HashMap<String, Object>();

	public Report(Map<String, Object> vars) {
		this.vars = vars;
	}

	// 按JSON串构造报表
	public Report(String vars) throws Exception {
		JSONObject json = new JSONObject(vars);
		this.vars = JsonHelper.toMap(json);
	}

	// 用xml文件内容以及json格式参数构造报表
	public Report(String source, String vars) throws Exception {
		this.load(source);
		JSONObject json = new JSONObject(vars);
		JSONObject data = json.getJSONObject("data");
		this.vars = JsonHelper.toMap(data);
		// 附加用户注册的对象到业务逻辑中
		Map<String, Object> plugins = PluginMapper.getPlugins();
		for(String key : plugins.keySet()) {
			this.vars.put(key, plugins.get(key));
		}
	}

	public Report() {
	}

	// 产生HTML的table形式的报表结果
	public String render() throws Exception {
		// 处理sql语句
		this.procSqls();
		// 编译
		this.compile();
		// 把report注入各单元格，以便执行时，从report中获取参数
		for (Cell cell : cells) {
			cell.setReport(this);
		}

		// 产生单元格
		List<Cell> result = this.createCells();
		Document doc = DocumentHelper.createDocument();
		Element table = this.layout(doc, result);
		String str = table.asXML();
		return str;
	}

	// 加载xml格式报表
	private void load(String source) {
		SAXReader reader = new SAXReader();
		try {
			Document document = reader.read(new StringReader(source));
			Element root = document.getRootElement();
			// 加载所有sql语句
			for (Object obj : root.elements("sql")) {
				Element ele = (Element) obj;
				this.sqls.add(new Sql(ele));
			}
			// 加载所有独立单元格
			for (Object obj : root.elements("cell")) {
				Element ele = (Element) obj;
				this.cells.add(new Cell(ele));
			}
			// 加载所有报表块
			for (Object obj : root.elements("reportblock")) {
				Element ele = (Element) obj;
				this.cells.add(new ReportBlock(ele));
			}
		} catch (DocumentException ex) {
			throw new RuntimeException(ex);
		}
	}

	// 执行sql语句，把结果保存起来，方便单元格调用
	public void procSqls() throws Exception {
		for (Sql sql : sqls) {
			// sql的param按表达式执行
			Program proc = new Program(sql.param);
			Delegate delegate = proc.parse();
			JSONObject json = (JSONObject) delegate.invoke(this.vars);
			// 调用sql处理，把结果保存到sqlResult中
			Object ret = sqlServer.query(sql.sql, json);
			this.vars.put(sql.name, ret);
		}
	}

	// 对所有单元格的表达式进行编译
	public void compile() {
		for (Cell cell : cells) {
			cell.compile();
		}
	}

	// 产生单元格，根据绘制的、排好顺序的块，按顺序一块块处理。
	public List<Cell> createCells() {
		List<Cell> result = new ArrayList<Cell>();

		// 对单元格按行列进行排序，把同一行放在一起
		cells.sort((o1, o2) -> (o1.row * 10000 + o1.column)
				- (o2.row * 10000 + o2.column));
		int copyRow = 0;
		for (Cell cell : cells) {
			// 如果是报表块，让报表块产生单元格
			if (cell instanceof ReportBlock) {
				ReportBlock block = (ReportBlock) cell;
				block.setRow(copyRow);
				copyRow += block.createCell(result);
			}
			// 纯单元格
			else {
				if (!("false".equals(cell.isDown))) {
					cell.row += copyRow;
				}
				// 执行单元格内容表达式
				cell.content = cell.invoke().toString();
				result.add(cell);
			}
		}

		return result;
	}

	// 对所有单元格进行整理，产生table表格
	public Element layout(Document doc, List<Cell> cells) {
		// 产生table
		Element eTable = doc.addElement("table");
		eTable.addAttribute("class", "tableprint");
		// 对单元格按行列进行排序，把同一行放在一起
		cells.sort((o1, o2) -> (o1.row * 10000 + o1.column)
				- (o2.row * 10000 + o2.column));
		int currentRow = -1;
		// 正在添加的当前行
		Element eTr = null;
		for (Cell cell : cells) {
			// 当行号发生变化时，产生新的tr元素
			if (cell.row != currentRow) {
				eTr = eTable.addElement("tr");
				currentRow = cell.row;
			}
			// 添加td，设置内容
			Element eTd = eTr.addElement("td");
			if (cell.css != null && !cell.css.equals("")) {
				eTd.addAttribute("class", cell.css);
			}
			if (cell.width != null && !cell.width.equals("")) {
				eTd.addAttribute("width", cell.width);
			}
			if (cell.height != null && !cell.height.equals("")) {
				eTd.addAttribute("height", cell.height);
			}
			if (cell.columnSpan > 1) {
				eTd.addAttribute("colspan", cell.columnSpan + "");
			}
			if (cell.rowSpan > 1) {
				eTd.addAttribute("rowspan", cell.rowSpan + "");
			}
			// 是html类型，输出html
			if (cell.type != null && cell.type.equals("html")) {
				SAXReader reader = new SAXReader();
				try {
					Document document = reader.read(new StringReader(cell.content));
					Element root = document.getRootElement();
					eTd.add(root);
				} catch (RuntimeException e) {
					throw e;
				} catch (Exception e) {
					throw new RuntimeException(e);
				}
			} else {
				eTd.addText(cell.content);
			}
		}
		return eTable;
	}
}
