package com.af.plugins;

import com.aote.config.SystemConfig;
import com.aote.rs.mapper.WebException;
import org.apache.http.HttpResponse;
import org.apache.http.client.HttpClient;
import org.apache.http.client.methods.HttpDelete;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.util.EntityUtils;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.Iterator;

/**
 * REST风格请求插件2
 */

public class RestTools2 {

	/**
	 * 发送带JSON串请求参数的post请求
	 *
	 * @param path    请求路径
	 * @param value    请求参数
	 * @return 请求结果
	 */
	public static String post(String path, JSONObject value) {
		//指定请求参数的数据格式是JSON。
		JSONObject headers = new JSONObject();
		headers.put("Content-Type", "application/json");
		return post(path, value.toString(), headers.toString());
	}

	/**
	 * 发送带字符串请求参数的post请求
	 *
	 * @param path  请求路径
	 * @param value 请求参数
	 * @return 请求结果
	 */
	public static String post(String path, String value) {
		return post(path, value, null);
	}

	/**
	 * 发送带请求头的post请求
	 * 	--用于需要自定义请求头的情况--
	 *
	 * @param path    请求路径
	 * @param value    请求参数
	 * @param headers    请求头
	 * @return 请求结果
     */
	public static String post(String path, JSONObject value, JSONObject headers) {
		//指定请求参数的数据格式是JSON。
		headers.put("Content-Type", "application/json");
		return post(path, value.toString(), headers.toString());
	}

	/**
	 * 发送post请求
	 *
	 * @param path    请求路径
	 * @param value   请求参数
	 * @param headersStr 请求头
	 * @return 请求结果
     */
	public static String post(String path, String value, String headersStr) {
		try {
			// 从系统配置中获得path对应的地址
			path = SystemConfig.System.getProxyUrl();
			// POSTMethod
			HttpPost postMethod = new HttpPost(path);
			StringEntity se = new StringEntity(value, "UTF-8");
			postMethod.setEntity(se);
			//设置请求头
            if (headersStr != null && !"".equals(headersStr)) {
                JSONObject headers = new JSONObject(headersStr);
				Iterator keys = headers.keys();
				while (keys.hasNext()) {
					String key = (String) keys.next();
					String val = headers.getString(key);
					postMethod.setHeader(key, val);
				}
			}
			// 发送Post请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(postMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toString(response.getEntity(), "UTF8");
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				throw new WebException(code, data);
			}
		}
		catch (RuntimeException e) {
			throw e;
		}
		catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送无请求头的get请求
	 *
	 * @param path	请求路径
	 * @return String结果
	 */
	public static String get(String path) {
		return get(path, null);
	}

	/**
	 * 发送get请求
	 *
	 * @param path    请求路径
	 * @param headers 请求头
	 * @return 返回的String数据
	 */
	public static String get(String path, JSONObject headers) {
		try {
			// 从系统配置中获得path对应的地址
			path = SystemConfig.System.getProxyUrl();
			// GETMethod
			HttpGet getMethod = new HttpGet(path);
			//设置请求头
			if (headers != null) {
				Iterator keys = headers.keys();
				while (keys.hasNext()) {
					String key = (String) keys.next();
					String val = headers.getString(key);
					getMethod.setHeader(key, val);
				}
			}
			// 发送Get请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(getMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toString(response.getEntity(), "UTF8");
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				throw new WebException(code, data);
			}
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送无请求头的getJson请求
	 *
	 * @param path	请求路径
	 * @return String结果
	 */
	public static JSONObject getJson(String path) {
		return getJson(path, null);
	}

	/**
	 * 发送getJson请求
	 *
	 * @param path    请求路径
	 * @param headers 请求头
	 * @return json对象
	 */
	public static JSONObject getJson(String path, JSONObject headers) {
		try {
			// GETMethod
			HttpGet getMethod = new HttpGet(path);
			//设置请求头
			getMethod.setHeader("Content-Type", "application/json");
			if (headers != null) {
				Iterator keys = headers.keys();
				while (keys.hasNext()) {
					String key = (String) keys.next();
					String val = headers.getString(key);
					getMethod.setHeader(key, val);
				}
			}
			// 发送Get请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(getMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				String actual = EntityUtils.toString(response.getEntity(), "UTF8");
				Object array;
				if (actual.startsWith("{")) {
                    array = new JSONObject(actual);
                } else if (actual.startsWith("[")) {
                    array = new JSONArray(actual);
                } else {
                    throw new Exception("调用rest服务失败。");
                }
				JSONObject result = new JSONObject();
				result.put("code", code);
				result.put("data", array);
				return result;
			} else {
				JSONObject result = new JSONObject();
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				result.put("code", code);
				result.put("data", data);
				return result;
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 发送无请求头的delete请求
	 *
	 * @param path    请求路径
	 * @return String结果
	 */
	public static String delete(String path) {
		return delete(path, null);
	}

	/**
	 * 发送delete请求
	 *
	 * @param path    请求路径
	 * @param headers 请求头
	 * @return String结果
	 */
	public static String delete(String path, JSONObject headers) {
		// 从系统配置中获得path对应的地址
		path = SystemConfig.System.getProxyUrl();
		// DeleteMethod
		HttpDelete deleteMethod = new HttpDelete(path);
		//设置请求头
		if (headers != null) {
			Iterator keys = headers.keys();
			while (keys.hasNext()) {
				String key = (String) keys.next();
				String val = headers.getString(key);
				deleteMethod.setHeader(key, val);
			}
		}
		// 发送Post请求
		HttpClient httpClient = HttpClientBuilder.create().build();
		try {
			HttpResponse response = httpClient.execute(deleteMethod);
			int code = response.getStatusLine().getStatusCode();

			// 获取数据成功，返回数据
			if (code == 200) {
				return EntityUtils.toString(response.getEntity(), "UTF8");
			} else {
				String data = response.getStatusLine().getReasonPhrase();
				// 返回错误码
				throw new WebException(code, data);
			}
		} catch (RuntimeException e) {
			throw e;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 执行给定action
	 *
	 * @param param	参数列表
	 * @return json对象
	 */
	public static JSONObject action(JSONObject param) {
		String action = param.getString("action");
		String url = param.getString("url");
		String data = param.getString("data");

		try {
			HttpPost postMethod =new HttpPost(url);
			StringEntity se = new StringEntity(data, "UTF-8");
			postMethod.setEntity(se);

			// 发送Post请求
			HttpClient httpClient = HttpClientBuilder.create().build();
			HttpResponse response = httpClient.execute(postMethod);
			String actual = EntityUtils.toString(response.getEntity(), "UTF8");

			// 返回结果
			JSONObject result = new JSONObject();
			int code = response.getStatusLine().getStatusCode();
			result.put("code", code);
			if (code == 200) {
				Object array;
				if(actual.startsWith("{")) {
                    array =  new JSONObject(actual);
                } else if(actual.startsWith("[")) {
                    array = new JSONArray(actual);
                } else {
                    throw  new Exception("调用rest服务失败。");
                }
				result.put("data", array);
			} else {
				result.put("data", actual);
			}
			return result;
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 给请求路径拼接参数
	 *
	 * @param url   请求路径
	 * @param params 请求参数
	 * @return String结果
	 */
	public static String formatURL(String url, JSONObject params) {
		//设置表单长度30字节*N个请求参数
		int capacity = params.length() * 30;
		//拼接请求路径
		StringBuilder buffer = new StringBuilder(capacity);
		buffer.append(url).append("?");
		//取出JSON里面的请求参数，添加到路径中。
		Iterator keys = params.keys();
		while (keys.hasNext()){
			String key = (String) keys.next();
			String val = params.getString(key);
			buffer.append(key).append("=").append(val);
			if(keys.hasNext()){
				buffer.append("&");
			}
		}
		return buffer.toString();
	}
}
