package com.af.plugins;

import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.core.date.DateUtil;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;

/**
 * 日期操作插件
 */

public class DateTools {

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  格式:标准格林威治时间（GMT）
	 */
	public static Date getNow() {
		return DateUtil.date();
	}

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  格式:yyyy-MM-dd HH:mm:ss
	 */
	public static String getNow2() {
		return DateUtil.now();
	}

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  返回值long类型
	 */
	public static long getCurrentTimeMillis() {
		return System.currentTimeMillis();
	}

	/**
	 * 根据指定格式获取当前时间
	 * @param format format字符串
	 * @return 当前时间  返回指定格式的String字符串
	 */
	public static String getNow(String format) {
		return DateUtil.date().toString(format);
	}

	/**
	 * 返回当前时间的年份
	 *
	 * @return 当前时间的年份
	 */
	public static String getNowYear() {
		return String.valueOf(DateUtil.thisYear());
	}

	/**
	 * 返回当前时间的月份
	 *
	 * @return 当前时间的月份
	 */
	public static String getNowMonth() {
		return String.valueOf(getNowMonthAsInt());
	}

	/**
	 * 返回当前时间的月份
	 *
	 * @return 当前时间的月份
	 */
	public static int getNowMonthAsInt() {
		return DateUtil.thisMonth() + 1;
	}

	/**
	 * 返回当前时间的日
	 *
	 * @return 当前时间的日
	 */
	public static String getNowDay() {
		return String.valueOf(DateUtil.thisDayOfMonth());
	}

	/**
	 * 返回当前时间是星期几
	 *
	 * @return 星期几
	 */
	public static String getNowDayOfWeek() {
		int value = DateUtil.thisDayOfWeek() - 1;
		if(value == 0){
			value = 7;
		}
		return String.valueOf(value);
	}

	/**
	 * 获取指定时间的月份
	 *
	 * @param date 指定时间字符串
	 * @return 指定时间的月份
	 */
	public static int getMonth(String date) {
		return DateUtil.parse(date).month() + 1;
	}

	/**
	 * 获取指定时间的日份
	 *
	 * @param date 指定时间字符串
	 * @return 指定时间的日份
	 */
	public static int getDate(String date) {
		return DateUtil.parse(date).dayOfMonth();
	}

	/**
	 * 返回传入时间的年份和月份
	 *
	 * @param date 指定时间字符串
	 * @return 传入的年份和月份，格式为yyyyMM
	 */
	public static String getYearMonth(String date) {
		return DateUtil.parse(date).toString("yyyyMM");
	}

	/**
	 * 格式化一个日期为指定日期格式
	 *
	 * @param datetime 日期字符串
	 * @return 标准日期格式 返回值为指定的String格式
	 */
	public static String format(String datetime,String format) {
		return DateUtil.parse(datetime).toString(format);
	}


	/**
	 * 格式化一个yyyy-MM-dd格式的日期为标准日期格式
	 *
	 * @param date yyyy-MM-dd格式的日期字符串
	 * @return 标准日期格式  返回指定的String类型:yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String formatConvert(String date) throws ParseException {
		return formatDateTime(date, "yyyy-MM-dd");
	}

	/**
	 * 格式化一个日期为标准日期格式
	 *
	 * @param datetime 日期字符串
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 */
	public static String formatDateTime(String datetime) {
		return format(datetime,"yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 格式化一个Long类型的日期为标准日期格式
	 *
	 * @param date long类型的日期字符串
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 */
	public String formatLongToDate(String date) {
		long dateLong = new Long(date);
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		Date date1 = new Date(dateLong);
		return simpleDateFormat.format(date1);
	}

	/**
	 * 格式化一个指定格式的日期字符串为标准日期格式
	 *
	 * @param dateStr 日期字符串
	 * @param format  该字符串的格式
	 * @return 标准日期格式  返回值为指定的String格式::yyyy-MM-dd HH:mm:ss
	 */
	public static String formatDateTime(String dateStr, String format) {
		return DateUtil.parse(dateStr,format).toString();
	}

	/**
	 * 格式化一个指定格式的日期字符串为指定日期格式
	 * @param dateStr 日期字符串
	 * @param format 该字符串的格式
	 * @param newFormat 指定转换的日期格式
	 */
	public static String formatDateTime(String dateStr, String format ,String newFormat) {
		return DateUtil.parse(dateStr,format).toString(newFormat);
	}

	/**
	 * 比较两个日期的大小
	 * @param dateStr1 第一个日期字符串
	 * @param dateStr2 第二个日期字符串
	 * @return a>=b时为true
	 */
	public static boolean compareDate(String dateStr1, String dateStr2) {
		return DateUtil.compare(DateUtil.parse(dateStr1),DateUtil.parse(dateStr2)) >= 0;
	}

	/**
	 * 比较两个日期(指定日期格式)的大小
	 * @param dateStr1 第一个日期字符串
	 * @param dateStr2 第二个日期字符串
	 * @param formatStr 指定比较日期的格式
	 * @return dateStr1>=dateStr2时为true
	 */
	public static boolean compareFormatDate(String dateStr1, String dateStr2,String formatStr) {
		return DateUtil.compare(DateUtil.parse(dateStr1,formatStr),DateUtil.parse(dateStr2,formatStr)) >= 0;
	}

	/**
	 * 获取指定日期下个月的某一天（用于滞纳金的计算）
	 *
	 * @param date        标准日期格式的日期字符串
	 * @param day		该指定日期下个月的某一天
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String nextMonth(String date, String day) throws ParseException {

		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		Date target = simpleDateFormat.parse(date);
		Calendar now = Calendar.getInstance();
		now.setTime(target);
		now.set(now.get(Calendar.YEAR),now.get(Calendar.MONTH) + 1,
				Integer.parseInt(day), 0, 0, 0);
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		return formatter.format(now.getTime());
	}

    /**
     * 获取本月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getCurrentMonthStart() {
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()),
                "month",
                "yyyy-MM-dd"
        );
    }

    /**
     * 获取上个月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getLastMonthStart() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, -1);
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }

    /**
     * 获取上上个月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getFrontLastMonthStart() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, -2);
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }
    /**
     * 获取指定日期的开始时间
     *
     * @param date   指定日期
     * @param type   操作类型，可选值(year,month,date)
     * @param format 指定日期格式
     * @return 当前年/月/日的指定格式开始时间
     */
    public static String getStartDate(String date, String type, String format) {
        if (StrUtil.isBlank(date) || StrUtil.isBlank(type)) {
            throw new RuntimeException("获取开始时间: 您所传的参数有误，请检查参数是否正确！！");
        }
        DateTime dateTime = DateUtil.parse(date);
		if ("month".equalsIgnoreCase(type)) {
			return DateUtil.beginOfMonth(dateTime).toString(format);
		} else if ("year".equalsIgnoreCase(type)) {
			return DateUtil.beginOfYear(dateTime).toString(format);
		} else {
			return DateUtil.beginOfDay(dateTime).toString(format);
		}
    }
	public static String getStartDate(String date, String type) {
		return getStartDate(date, type, "yyyy-MM-dd HH:mm:ss");
	}

    /**
     * 获取下个月的最后时间
     *
     * @return 返回格式: "yyyy-MM-dd"
     */
    public static String getNextMonthEnd() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, +1);
        return getEndDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }
    /**
     * 获取指定日期的最后时间
     *
     * @param date 指定日期
     * @param type 操作类型，可选值(year,month,date)
	 * @param format  格式化日期
     * @return 当前年/月/日的指定格式最后时间
     */
    public static String getEndDate(String date, String type, String format) {
        if (StrUtil.isBlank(date) || StrUtil.isBlank(type)) {
            throw new RuntimeException("获取最后时间: 您所传的参数有误，请检查参数是否正确！！");
        }
		DateTime dateTime = DateUtil.parse(date);
		if ("month".equalsIgnoreCase(type)) {
			return DateUtil.endOfMonth(dateTime).toString(format);
		} else if ("year".equalsIgnoreCase(type)) {
			return DateUtil.endOfYear(dateTime).toString(format);
		} else {
			return DateUtil.endOfDay(dateTime).toString(format);
		}
    }
	public static String getEndDate(String date, String type) {
		return getEndDate(date, type, "yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 获取阶梯开始日期
	 *
	 * @param start 开始时间 格式为MM-dd 或 yyyy-HH-mm HH:mm:ss
	 * @param cycle 阶梯周期
	 * @param handdate 下发时间 格式为yyyy-MM-dd
	 * @return 返回阶梯开始日期 yyyy-MM-dd
	 * @throws Exception 异常
	 */
	public static String getStairStartDate(String start, Object cycle, String handdate)
			throws Exception {
		// 参数验证
		if ("".equals(start) || start == null) {
			throw new Exception("阶梯开始时间不能为空!!");
		}
		if (cycle == null) {
			throw new Exception("阶梯周期不能为空!!");
		}
		if ("".equals(handdate) || handdate == null) {
			throw new Exception("抄表时间不能为空!!");
		}

		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Calendar now = Calendar.getInstance();
		if(start.length() > 5) {
		    //如果长度大于5，则按传入yyyy-HH-mm HH:mm:ss格式进行截取
		    start = start.substring(5,10);
        }
		String[] str = start.split("-");
		// 开始月份
		int beginMonth = Integer.parseInt(str[0]);
		// 当前月份
		int nowMonth = getMonth(handdate);

		if (isLastYear(handdate)) {
			now.set(Calendar.YEAR, now.get(Calendar.YEAR) - 1);
		} else {
			now.set(Calendar.YEAR, now.get(Calendar.YEAR));
		}

		now.set(Calendar.DAY_OF_MONTH, Integer.parseInt(str[1]));


		// 计算月份
		int a = (nowMonth + (12 - beginMonth))%(Integer.parseInt(cycle.toString()));
//		//system.out.println("取余操作" + a);
		int month;
		if (isLowerDay(Integer.parseInt(str[1]))) {
			month = a == 0 ? a - (int)cycle : -a;
		} else {
			month = -a;
		}
//		//system.out.println("计算月份" + month);
		now.add(Calendar.MONTH, month);

		return format.format(now.getTime());
	}

	public static boolean isLowerDay(int d) {
		Calendar now = Calendar.getInstance();
		int day = now.get(Calendar.DATE);
		return day - d < 0;
	}


	/**
	 * 比较指定日期和当前日期的年份
	 *
	 * @param date        指定日期，格式为yyyy-MM-dd
	 * @return	比较结果，true：指定的日期年份比当前日期年份早
	 */
	private static boolean isLastYear(String date) {
		return DateUtil.parse(date).year() < DateUtil.date().year();
	}

	/**
	 * 判断指定参数是否为空
	 *
	 * @param param	指定参数
	 * @return	返回结果，true：参数为空
	 */
	public static boolean isNull(String param) {
		return "".equals(param) || param == null;
	}

	/**
	 * 计算两个指定日期的时间差
	 *
	 * @param before 较早的日期
	 * @param after 较晚的日期
	 * @return 判断的时间差结果
	 */
	public static String dateDisp(Date before, Date after) {
		long time = after.getTime() - before.getTime();
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		return dateFormat.format(after) + "-" + dateFormat.format(before) + " 时间差:" + time;
	}

	/**
	 * 根据操作类型获取当前日期的前后日期，如上周或下个月的这一天
	 *
	 * @param type   操作类型，可选值(YEAR,MONTH,DATE)
	 * @param param 差值，负数即从当前日期往前推
	 * @param format 返回的日期格式，默认为yyyy-MM-dd
	 * @return 当前日期的前后日期
	 */
	public static String getDiffDate(String type, String param, String format) {
		DateTime dateTime = DateUtil.date();
		return getDelayDate(dateTime,type,param,format);
	}

	/**
	 * 根据操作类型获取指定日期的前后日期，如上周或下个月的这一天
	 *
	 * @param date  指定日期，格式为标准日期格式
	 * @param type  操作类型，可选值(YEAR,MONTH,DAY,SECOND)
	 * @param param 差值，负数即从当前日期往前推
	 * @return 当前日期的前后日期，格式为标准日期格式
	 */
	public static String getDelayDate(String date, String type, String param) {
		DateTime dateTime = DateUtil.parse(date);
		return getDelayDate(dateTime,type,param,"yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 获取两个时间的相差间隔
	 * @param date1 时间1
	 * @param date2 时间2
	 * @return 相差间隔（毫秒，绝对值）
	 */
	public static String getDateBetween(String date1,String date2){
		return String.valueOf(DateUtil.betweenMs(DateUtil.parse(date1),DateUtil.parse(date2)));
	}

	/**
	 * 获取两个时间的天数相差间隔
	 * @param date1 时间1
	 * @param date2 时间2
	 * @param isReset
	 *   * 有时候我们计算相差天数的时候需要忽略时分秒。
	 * 	 * 比如：2016-02-01 23:59:59和2016-02-02 00:00:00相差一秒
	 * 	 * 如果isReset为<code>false</code>相差天数为0。
	 * 	 * 如果isReset为<code>true</code>相差天数将被计算为1
	 * @return 相差间隔（天数，绝对值）
	 */
	public static String getDateDayBetween(String date1,String date2,Boolean isReset){
		return String.valueOf(DateUtil.betweenDay(DateUtil.parse(date1),DateUtil.parse(date2),isReset));
	}

	private static String getDelayDate(DateTime dateTime, String type, String param,String format) {
		//获取处理后的时间
		switch (type.toUpperCase()) {
			case "YEAR":
				dateTime.offset(DateField.YEAR, Integer.parseInt(param));
				break;
			case "MONTH":
				dateTime.offset(DateField.MONTH, Integer.parseInt(param));
				break;
			case "DATE":
				dateTime.offset(DateField.DAY_OF_MONTH, Integer.parseInt(param));
				break;
			case "SECOND":
				dateTime.offset(DateField.SECOND, Integer.parseInt(param));
			default:
				throw new RuntimeException("不支持的日期类型！可选值(YEAR,MONTH,DATE,SECOND)");
		}
		return dateTime.toString(format);
	}
}
