package com.aote.plugins;

import com.aote.rs.ManagerService;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.CreateTopicsResult;
import org.apache.kafka.clients.admin.ListTopicsResult;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.TopicPartition;
import org.json.JSONArray;
import org.json.JSONObject;

import java.util.*;
import java.util.concurrent.ExecutionException;

public class Kafka {

    /**
     *  根据config.json重建topic
     */
    public static void createTopics(int numPartitions) {
        Properties props = new Properties();
        // 获取config的值
        JSONObject config = getConig();
        // 如果配置了kafka的server及topics，根据配置内容对topic进行处理
        if (!config.has("kafka")) {
            return;
        }
        JSONObject kafka = config.getJSONObject("kafka");
        if (!kafka.has("server") || !kafka.has("topics")) {
            return;
        }

        try {
            // 设置server
            String servers = kafka.getString("server");
            props.put("bootstrap.servers", servers);

            // 获取已经存在的topic
            AdminClient adminClient = AdminClient.create(props);
            Set<String> list = adminClient.listTopics().names().get();

            // 本次创建的topic，包括已经存在的
            Set<String> created = new HashSet<String>();
            // 需要本次新建topic
            ArrayList<NewTopic> topics = new ArrayList<>();

            // 对于配置文件里的每个topic
            JSONArray topicStr = kafka.getJSONArray("topics");
            for (Object o : topicStr) {
                String topic = (String) o;
                // 如果topic已经存在，则不创建
                created.add(topic);
                if (!list.contains(topic)) {
                    NewTopic newTopic = new NewTopic(topic, numPartitions, (short) 1);
                    topics.add(newTopic);
                }
            }

            // 创建新的topic
            adminClient.createTopics(topics);
            // 删除配置文件里没有的topic
            list.removeAll(created);
            adminClient.deleteTopics(list);
        } catch (RuntimeException e) {
            throw e;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 生产消息
     * @param topic 主题名称
     * @param news 一批消息，消息内容为JSON格式字符串
     */
    public static void produceMsg(String topic, JSONArray news) {
        Properties props = new Properties();
        props.put("acks", "all");
        props.put("retries", 0);
        props.put("batch.size", 16384);
        props.put("linger.ms", 1);
        props.put("buffer.memory", 33554432);
        props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        // 获取config中server的值
        String servers = getConig().getJSONObject("kafka").getString("server");
        props.put("bootstrap.servers", servers);
        Producer<String, String> producer = new KafkaProducer<>(props);
        // 发送消息
        try{
            for (int i = 0; i < news.length(); i++)
                producer.send(new ProducerRecord<>(topic, Integer.toString(i), news.getJSONObject(i).toString()));
        }catch (Exception e){
            throw new RuntimeException(e);
        }
        try {
            producer.close();
        } catch (Exception|Error e){
            //kafka引用的jar包异常， 暂不处理
        }
    }

    /**
     * 消费消息
     * @param topic 主题名称
     * @return 一批消息
     */
    public static ConsumerRecords<String, String> consumeMsg(String topic){
        Properties props = new Properties();
        props.put("group.id", "test");
        props.put("enable.auto.commit", "true");
        props.put("auto.commit.interval.ms", "1000");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        // 获取config的值
        String servers = getConig().getJSONObject("kafka").getString("server");
        props.put("bootstrap.servers", servers);
        final KafkaConsumer<String, String> consumer = new KafkaConsumer<>(props);
        List<String> topics = new ArrayList<>();
        topics.add(topic);
        consumer.subscribe(topics, new ConsumerRebalanceListener() {
            public void onPartitionsRevoked(Collection<TopicPartition> collection) {
            }
            public void onPartitionsAssigned(Collection<TopicPartition> collection) {
                //将偏移设置到最开始
                consumer.seekToBeginning(collection);
            }
        });
        ConsumerRecords<String, String> records = consumer.poll( 100);
        try {
            while (true) {
                for (ConsumerRecord<String, String> record : records)
                    System.out.printf("offset = %d, key = %s, value = %s%n", record.offset(), record.key(), record.value());
                Thread.sleep(1000);
            }
        } catch(Exception e) {
            throw new RuntimeException(e);
        }
        //return records;
    }

    private static JSONObject getConig() {
        ManagerService ms = new ManagerService();
        String config = ms.getConfig();
        return new JSONObject(config);
    }
}
