package com.af.plugins.xq;

import com.aote.ThreadResource;
import org.json.JSONObject;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.util.concurrent.ListenableFuture;
import org.springframework.web.client.AsyncRestTemplate;

/**
 * 异步Rest请求工具类
 *
 * @author Mr.river
 * @date 2019-06-24
 */
public abstract class AsyncTools {

    /**
     * 异步请求回调方法
     * @param result 请求结果（成功：true，失败：false）
     * @param httpCode HTTP状态码
     * @param response HTTP响应头信息
     * @param body 响应结果
     */
    public abstract void callBack(Boolean result, String httpCode, String response, String body);

    /**
     * 发送异步post请求
     *
     * @param path  请求路径
     * @param isCallBack 是否进行回调
     */
    public void postAsync(String path, Boolean isCallBack){
        postAsync(path,"","",isCallBack);
    }

    /**
     * 发送带请求体JSON的异步post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @param isCallBack 是否进行回调
     */
    public void postAsync(String path, JSONObject value, Boolean isCallBack) {
        postAsync(path, value.toString(), "",isCallBack);
    }

    /**
     * 发送带请求体字符串的异步post请求
     *
     * @param path  请求路径
     * @param value 请求参数
     * @param isCallBack 是否进行回调
     */
    public void postAsync(String path, String value, Boolean isCallBack) {
        postAsync(path,value,"",isCallBack);
    }

    /**
     * 发送带请求头和请求体的异步post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @param headers    请求头
     * @param isCallBack 是否进行回调
     */
    public void postAsync(String path, JSONObject value, JSONObject headers, Boolean isCallBack) {
        if(value != null){
            if(headers == null){
                headers = new JSONObject();
            }
            if(!headers.has("Content-Type")){
                //指定请求参数的数据格式是JSON。
                headers.put("Content-Type", "application/json");
            }
            postAsync(path,value.toString(),headers.toString(),isCallBack);
        } else {
            if(headers == null){
                postAsync(path,"","",isCallBack);
            } else {
                postAsync(path,"",headers.toString(),isCallBack);
            }
        }
    }

    /**
     * 发送带请求头和请求体的异步post请求
     *
     * @param path    请求路径
     * @param value    请求参数
     * @param headerStr 请求头字符串
     * @param isCallBack 是否进行回调
     */
    private void postAsync(String path, String value, String headerStr, Boolean isCallBack) {
        //设置请求头
        HttpHeaders httpHeaders = new HttpHeaders();
        setHeaders(headerStr,httpHeaders);
        //发送异步请求
        AsyncRestTemplate client = new AsyncRestTemplate();
        ListenableFuture<ResponseEntity<String>> listenableFuture =
                client.postForEntity(path, new HttpEntity<>(value,httpHeaders),String.class);
        if(isCallBack){
            listenableFuture.addCallback(
                result -> {
                    String httpCode = result.getStatusCode().toString();
                    String response = result.getStatusCode().getReasonPhrase();
                    String body = result.getBody();
                    this.callBack(true,httpCode,response,body);
                },
                ex -> {
                    String httpCode = ex.getMessage().substring(0,ex.getMessage().indexOf(" "));
                    String response = ex.getMessage().substring(ex.getMessage().indexOf(" "));
                    this.callBack(false,httpCode,response,null);
                }
            );
        }
    }

    /**
     * 设置请求头
     * @param headersStr 请求头字符串
     * @param base HttpHeader实例
     */
    private static void setHeaders(String headersStr, HttpHeaders base) {
        // 设置token
        String token = ThreadResource.Token.get();
        if (token != null) {
            base.add("Token", token);
        }

        // 设置其他请求头
        if (headersStr != null && !"".equals(headersStr)) {
            JSONObject headers = new JSONObject(headersStr);
            for(String headerKey : headers.keySet()){
                base.add(headerKey,headers.getString(headerKey));
            }
        }
    }
}
