package com.af.plugins;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.Calendar;
import java.util.Date;

/**
 * 日期操作插件
 */

public class DateTools {

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  格式:标准格林威治时间（GMT）
	 */
	public static Date getNow() {
		return new Date();
	}

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  格式:yyyy-MM-dd HH:mm:ss
	 */
	public static String getNow2() {
		return getNow("yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 获取当前时间
	 *
	 * @return 当前时间  返回值long类型
	 */
	public static long getCurrentTimeMillis() {
		return System.currentTimeMillis();
	}

	/**
	 * 根据指定格式获取当前时间
	 * @param format format字符串
	 * @return 当前时间  返回指定格式的String字符串
	 */
	public static String getNow(String format) {
		SimpleDateFormat formatter = new SimpleDateFormat(format);
		return formatter.format(new Date());
	}

	/**
	 * 返回当前时间的年份
	 *
	 * @return 当前时间的年份
	 */
	public static String getNowYear() {
		Calendar now = Calendar.getInstance();
		int year = now.get(Calendar.YEAR);
		return String.valueOf(year);
	}

	/**
	 * 返回当前时间的月份
	 *
	 * @return 当前时间的月份
	 */
	public static String getNowMonth() {
		Calendar now = Calendar.getInstance();
		int month = now.get(Calendar.MONTH);
		return String.valueOf(month + 1);
	}
	
	/**
	 * 返回当前时间的月份
	 *
	 * @return 当前时间的月份
	 */
	public static int getNowMonthAsInt() {
        return Integer.valueOf(getNowMonth());
	}

	/**
	 * 返回当前时间的日
	 *
	 * @return 当前时间的日
	 */
	public static String getNowDay() {
		Calendar now = Calendar.getInstance();
		int day = now.get(Calendar.DATE);
		return String.valueOf(day);
	}

	/**
	 * 返回当前时间是星期几
	 *
	 * @return 星期几
	 */
	public static String getNowDayOfWeek() {
		Calendar now = Calendar.getInstance();
		int dayOfWeek = now.get(Calendar.DAY_OF_WEEK);
		return String.valueOf(dayOfWeek - 1);
	}

	/**
	 * 获取指定时间的月份
	 *
	 * @param date 指定时间，格式为yyyy-MM-dd
	 * @return 指定时间的月份
	 * @throws ParseException 异常
	 */
	public static int getMonth(String date) throws ParseException {
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Date getDate = format.parse(date);
		Calendar cal = Calendar.getInstance();
		cal.setTime(getDate);
		return cal.get(Calendar.MONTH) + 1;
	}

	/**
	 * 返回传入时间的年份和月份
	 *
	 * @param date 传入的时间，格式为yyyy-MM-dd
	 * @return 传入的年份和月份，格式为yyyyMM
	 * @throws ParseException 异常
	 */
	public static String getYearMonth(String date) throws ParseException {
		//获取传入时间
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Date getDate = format.parse(date);
		Calendar cal = Calendar.getInstance();
		cal.setTime(getDate);
		//获取年份
		String year = String.valueOf(cal.get(Calendar.YEAR));
		//获取月份
		String month = String.valueOf((cal.get(Calendar.MONTH) + 1));
		//格式化结果
		format = new SimpleDateFormat("yyyyMM");
		getDate = format.parse(year + month);
		return format.format(getDate);
	}

	/**
	 * 格式化一个yyyy-MM-dd格式的日期为标准日期格式
	 *
	 * @param date yyyy-MM-dd格式的日期字符串
	 * @return 标准日期格式  返回指定的String类型:yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String formatConvert(String date) throws ParseException {
		return formatDateTime(date, "yyyy-MM-dd");
	}

	/**
	 * 格式化一个yyyyMMddHHmmss格式的日期为标准日期格式
	 *
	 * @param datetime yyyyMMddHHmmss格式的日期字符串
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String formatDateTime(String datetime) throws ParseException {
		return formatDateTime(datetime, "yyyyMMddHHmmss");
	}

	/**
	 * 格式化一个Long类型的日期为标准日期格式
	 *
	 * @param date long类型的日期字符串
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 */
	public String formatLongToDate(String date) {
		long dateLong = new Long(date);
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		Date date1 = new Date(dateLong);
		return simpleDateFormat.format(date1);
	}

	/**
	 * 格式化一个指定格式的日期字符串为标准日期格式
	 *
	 * @param dateStr 日期字符串
	 * @param format  该字符串的格式
	 * @return 标准日期格式  返回值为指定的String格式::yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String formatDateTime(String dateStr, String format) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		Date date = simpleDateFormat.parse(dateStr);
		simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		return simpleDateFormat.format(date);
	}

	/**
	 * 格式化一个指定格式的日期字符串为指定日期格式
	 * @param dateStr 日期字符串
	 * @param format 该字符串的格式
	 * @param newFormat 指定转换的日期格式
	 * @throws ParseException 异常
	 */
	public static String formatDateTime(String dateStr, String format ,String newFormat) throws ParseException {
		SimpleDateFormat simpleDateFormat = new SimpleDateFormat(format);
		Date date = simpleDateFormat.parse(dateStr);
		simpleDateFormat = new SimpleDateFormat(newFormat);
		return simpleDateFormat.format(date);
	}

	/**
	 * 比较两个日期的大小
	 * @param dateStr1 第一个日期字符串
	 * @param dateStr2 第二个日期字符串
	 * @return a>=b时为true
	 */
	public static boolean compareDate(String dateStr1, String dateStr2) throws ParseException {
		return compareFormatDate(dateStr1,dateStr2,"yyyy-MM-dd HH:mm:ss");
	}

	/**
	 * 比较两个日期(指定日期格式)的大小
	 * @param dateStr1 第一个日期字符串
	 * @param dateStr2 第二个日期字符串
	 * @param formatStr 指定比较日期的格式
	 * @return dateStr1>=dateStr2时为true
	 */
	public static boolean compareFormatDate(String dateStr1, String dateStr2,String formatStr) throws ParseException {
		SimpleDateFormat format = new SimpleDateFormat(formatStr);
		Date date1 = format.parse(dateStr1);
		Date date2 = format.parse(dateStr2);
		if (date1.before(date2)) {
			return false;
		} else if (date1.after(date2)) {
			return true;
		} else {
			return true;
		}
	}

	/**
	 * 获取指定日期下个月的某一天（用于滞纳金的计算）
	 *
	 * @param date        标准日期格式的日期字符串
	 * @param day		该指定日期下个月的某一天
	 * @return 标准日期格式 返回值为指定的String格式:yyyy-MM-dd HH:mm:ss
	 * @throws ParseException 异常
	 */
	public static String nextMonth(String date, String day) throws ParseException {

		SimpleDateFormat simpleDateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		Date target = simpleDateFormat.parse(date);
		Calendar now = Calendar.getInstance();
		now.setTime(target);
		now.set(now.get(Calendar.YEAR),now.get(Calendar.MONTH) + 1,
				Integer.parseInt(day), 0, 0, 0);
		SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		return formatter.format(now.getTime());
	}

    /**
     * 获取本月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getCurrentMonthStart() {
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(new Date()),
                "month",
                "yyyy-MM-dd"
        );
    }

    /**
     * 获取上个月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getLastMonthStart() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, -1);
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }

    /**
     * 获取上上个月开始时间
     *
     * @return 返回格式："yyyy-MM-dd"
     */
    public static String getFrontLastMonthStart() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, -2);
        return getStartDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }

	/**
	 * 获取指定日期的开始时间
	 *
	 * @param date 指定日期
     * @param type 操作类型，可选值(year,month,date)
     * @return 当前年/月/日的开始时间，默认格式为yyyy-MM-dd HH:mm:ss
     *
     */
    public static String getStartDate(String date, String type) {
        return getStartDate(date, type, "yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 获取指定日期的开始时间
     *
     * @param date   指定日期
     * @param type   操作类型，可选值(year,month,date)
     * @param format 指定日期格式
     * @return 当前年/月/日的指定格式开始时间
     */
    public static String getStartDate(String date, String type, String format) {
        if (isNull(date) || isNull(type)) {
            throw new RuntimeException("获取开始时间: 您所传的参数有误，请检查参数是否正确！！");
        }
        DateTimeFormatter formatter = DateTimeFormatter
                .ofPattern("yyyy-MM-dd HH:mm:ss");

        LocalDateTime getDate = LocalDateTime.parse(date, formatter);

        LocalDate firstDate;
        if ("month".equals(type.toLowerCase())) {
            firstDate = getDate.toLocalDate().with(
                    TemporalAdjusters.firstDayOfMonth());
        } else if ("year".equals(type.toLowerCase())) {
            firstDate = getDate.toLocalDate().with(
                    TemporalAdjusters.firstDayOfYear());
        } else {
            firstDate = getDate.toLocalDate();
        }

        LocalDateTime result = LocalDateTime.of(firstDate,
                LocalTime.of(0, 0, 0));

        return DateTimeFormatter.ofPattern(format).format(result);
    }

    /**
     * 获取下个月的最后时间
     *
     * @return 返回格式: "yyyy-MM-dd"
     */
    public static String getNextMonthEnd() {
        Calendar c = Calendar.getInstance();
        c.add(Calendar.MONTH, +1);
        return getEndDate(
                new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(c.getTime()),
                "month",
                "yyyy-MM-dd"
        );
    }

    /**
     * 获取指定日期的最后时间
     *
     * @param date 指定日期
     * @param type 操作类型，可选值(year,month,date)
     * @return 当前年/月/日的最后时间，默认格式为yyyy-MM-dd HH:mm:ss
     */
    public static String getEndDate(String date, String type) {
        return getEndDate(date, type, "yyyy-MM-dd HH:mm:ss");
    }

    /**
     * 获取指定日期的最后时间
     *
     * @param date 指定日期
     * @param type 操作类型，可选值(year,month,date)
	 * @param format  格式化日期
     * @return 当前年/月/日的指定格式最后时间
     */
    public static String getEndDate(String date, String type, String format) {
        if (isNull(date) || isNull(type)) {
            throw new RuntimeException("获取最后时间: 您所传的参数有误，请检查参数是否正确！！");
        }

        DateTimeFormatter formatter = DateTimeFormatter
                .ofPattern("yyyy-MM-dd HH:mm:ss");

        LocalDateTime getDate = LocalDateTime.parse(date, formatter);

        LocalDate firstDate;
        if ("month".equals(type.toLowerCase())) {
            firstDate = getDate.toLocalDate().with(
                    TemporalAdjusters.lastDayOfMonth());
        } else if ("year".equals(type.toLowerCase())) {
            firstDate = getDate.toLocalDate().with(
                    TemporalAdjusters.lastDayOfYear());
        } else {
            firstDate = getDate.toLocalDate();
        }

        LocalDateTime result = LocalDateTime.of(firstDate,
                LocalTime.of(23, 59, 59));

        return DateTimeFormatter.ofPattern(format).format(result);
    }

	/**
	 * 获取阶梯开始日期
	 *
	 * @param start 开始时间 格式为MM-dd
	 * @param cycle 阶梯周期
	 * @param handdate 下发时间 格式为MM-dd
	 * @return 返回阶梯开始日期 yyyy-MM-dd
	 * @throws Exception 异常
	 */
	public static String getStairStartDate(String start, Object cycle, String handdate)
			throws Exception {
		// 参数验证
		if ("".equals(start) || start == null) {
			throw new Exception("您所传的阶梯开始时间不能为空!!");
		}
		if (cycle == null) {
			throw new Exception("您所传的阶梯周期不能为空!!");
		}
		if ("".equals(handdate) || handdate == null) {
			throw new Exception("您所传的抄表单下发时间不能为空!!");
		}

		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Calendar now = Calendar.getInstance();
		String[] str = start.split("-");
		// 开始月份
		int beginMonth = Integer.parseInt(str[0]);
		// 当前月份
		int nowMonth = getMonth(handdate);

		if (isLastYear(handdate)) {
			now.set(Calendar.YEAR, now.get(Calendar.YEAR) - 1);
		} else {
			now.set(Calendar.YEAR, now.get(Calendar.YEAR));
		}
		now.set(Calendar.MONTH, beginMonth - 1);
		now.set(Calendar.DAY_OF_MONTH, Integer.parseInt(str[1]));
		now.add(Calendar.MONTH,
				(((new BigDecimal(nowMonth - beginMonth).divide(new BigDecimal(cycle.toString()),2,BigDecimal.ROUND_HALF_DOWN)).setScale(0, BigDecimal.ROUND_FLOOR)).multiply(
						new BigDecimal(cycle.toString()))).intValue());
		return format.format(now.getTime());
	}
	
	/**
	 * 比较指定日期和当前日期的年份
	 *
	 * @param date        指定日期，格式为yyyy-MM-dd
	 * @return	比较结果，true：指定的日期年份比当前日期年份早
	 */
	private static boolean isLastYear(String date) throws Exception {
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd");
		Date getDate = format.parse(date);

		Calendar cal = Calendar.getInstance();
		// 当前年份
		int currentYear = cal.get(Calendar.YEAR);
		// 获取传入日期的年份
		cal.setTime(getDate);
		int year = cal.get(Calendar.YEAR);
		return year < currentYear;
	}

	/**
	 * 判断指定参数是否为空
	 *
	 * @param param	指定参数
	 * @return	返回结果，true：参数为空
	 */
	private static boolean isNull(String param) {
		return "".equals(param) || param == null;
	}

	/**
	 * 计算两个指定日期的时间差
	 *
	 * @param before 较早的日期
	 * @param after 较晚的日期
	 * @return 判断的时间差结果
	 */
	public static String dateDisp(Date before, Date after) {
		long time = after.getTime() - before.getTime();
		SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		return dateFormat.format(after) + "-" + dateFormat.format(before) + " 时间差:" + time;
	}

	/**
	 * 根据操作类型获取当前日期的前后日期，如上周或下个月的这一天
	 *
	 * @param type   操作类型，可选值(YEAR,MONTH,DATE)
	 * @param number 差值，负数即从当前日期往前推
	 * @param format 返回的日期格式，默认为yyyy-MM-dd
	 * @return 当前日期的前后日期
	 */
	public static String getDiffDate(String type, String number, String format) {
		//获取当前时间
		Calendar calendar = Calendar.getInstance();
		//获取处理后的时间
		switch (type.toUpperCase()) {
			case "YEAR":
				calendar.add(Calendar.YEAR, Integer.parseInt(number));
				break;
			case "MONTH":
				calendar.add(Calendar.MONTH, Integer.parseInt(number));
				break;
			case "DATE":
			default:
				calendar.add(Calendar.DATE, Integer.parseInt(number));
				break;
		}
		Date date = calendar.getTime();
		//判断返回的日期格式
		format = format == null ? "yyyy-MM-dd" : format;

		SimpleDateFormat dateFormat = new SimpleDateFormat(format);
		return dateFormat.format(date);
	}

	/**
	 * 根据操作类型获取指定日期的前后日期，如上周或下个月的这一天
	 *
	 * @param date  指定日期，格式为标准日期格式
	 * @param type  操作类型，可选值(YEAR,MONTH,DATE)
	 * @param param 差值，负数即从当前日期往前推
	 * @return 当前日期的前后日期，格式为标准日期格式
	 * @throws ParseException 异常
	 */
	public static String getDelayDate(String date, String type, String param)
			throws ParseException {
		//获取指定的时间
		SimpleDateFormat format = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
		Date target = format.parse(date);
		Calendar cal = Calendar.getInstance();
		cal.setTime(target);
		//获取处理后的时间
		switch (type.toUpperCase()) {
			case "YEAR":
				cal.add(Calendar.YEAR, Integer.parseInt(param));
				break;
			case "MONTH":
				cal.add(Calendar.MONTH, Integer.parseInt(param));
				break;
			case "DAY":
				cal.add(Calendar.DAY_OF_MONTH, Integer.parseInt(param));
				break;
			default:
				throw new RuntimeException("不支持的日期类型！可选值(YEAR,MONTH,DATE)");
		}
		return format.format(cal.getTime());
	}
}