package com.aote.plugins.impexp.exportfile;

import java.io.FileNotFoundException;
import java.sql.Connection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.*;

import javax.sql.DataSource;

import com.af.plugins.impexp.FileManager;
import com.aote.ThreadResource;
import com.aote.module.ModuleMapper;
import com.aote.util.ResourceHelper;
import org.hibernate.criterion.IdentifierEqExpression;
import org.json.JSONArray;
import org.json.JSONObject;
import org.springframework.beans.factory.annotation.Autowired;

import com.aote.sql.SqlServer;
import com.aote.util.ExcelUtil;
import org.springframework.stereotype.Component;

/**
 * 导出Excel文件
 */
@Component
public class ExportFile implements IExportFile {

	// json形式的配置内容
	private JSONObject map = null;
	private static final int MAX_ROWS_PER_SHEET = 10001;
	private static final String defaultCondition = "{orderitem: 'id',condition: '1=1'}";

	@Autowired
	private DataSource dataSource;
	/**
	 * 导出数据
	 * @param condition 导出时的前台传递的数据
	 * @param configName 导出对应的配置内容
	 * @return
	 * @throws Exception
	 */
	@Override
	public JSONArray export(String condition, String configName) throws Exception {
		// 读取对应目录下的配置文件
		this.map = this.getConfig(configName);
		if(map == null) {
			return noTemplate(condition);
		} else {
			Set<String> templateNames = this.map.keySet();
			if (templateNames.size() == 0) {
				throw new Exception("此导出需要模板，请检查impexp/exp/路径下的相关配置！！");
			}
			return hasTemplate(templateNames, condition);
		}
	}

	/**
	 * 获取配置文件
	 * @param configName 配置文件名
	 * @return 配置文件
	 */
	private JSONObject getConfig(String configName) {
		if (configName == null || "".equals(configName)) {
			return null;
		}
		// 从线程变量里获取登录用户目录名
		String dir = ThreadResource.ComponentDir.get();
		// 如果登录用户不为空，
		if (dir != null) {
			return null;
		}

		//模块中的路径
		Map<String, Map<String, String>> map= ModuleMapper.getMap();
		for (String key : map.keySet()) {
			try {
				String configStr = ResourceHelper.getString(key + "/impexp/exp/" + configName);
				JSONObject result = new JSONObject(configStr);
				return result;
			} catch (Exception e) {
				// 未找到文件处理
				continue;
			}
		}
		throw new RuntimeException("导出的配置文件未找到！" + configName);
	}

	/**
	 *  获取模板文件
	 * @param configName
	 * @return
	 */
//	private JSONObject getTemplate(String configName) {
//
//	}

	/**
	 * 无模板导出
	 * @param condition: 前台传入的条件等内容，格式为：{templateName: 要保存的excel文件名}
	 * @return
	 * @throws Exception
	 */
	private JSONArray noTemplate(String condition) throws Exception {
		JSONArray result = new JSONArray();
		JSONObject json = new JSONObject();
		long tick = System.currentTimeMillis();

		// 从前台传入的内容中，获取excel文件名，templateName为excel文件名
		String filepath = getFilePath(new JSONObject(condition).getString("templateName")+".xlsx");
		json.put("filename", filepath);
		result.put(json);

		export(new JSONObject(condition).getString("sqlName"), condition, null, filepath);

		tick = System.currentTimeMillis() - tick;
		System.out.println("进行Excel导出耗时：" + (tick / 1000.0) + "秒");
		System.out.println("文件名：" + result);
		return result;
	}

	/**
	 * 有模板内容的导出
	 * @param templateNames: 模板文件名
	 * @param condition: 前台传入的数据
	 * @return
	 * @throws Exception
	 */
	private JSONArray hasTemplate(Set<String> templateNames, String condition)
			throws Exception {
		JSONArray result = new JSONArray();

		int i = 0;
		// 可以有多个模板，对每一个模板进行处理
		for (String template : templateNames) {
			String sqlName = map.getString(template);
			String filepath = getFilePath(template);
			JSONObject json = new JSONObject();
			json.put("filename", filepath);
			result.put(json);
			// 模板名 template 判断各公司有无特殊模板
			template = "excel/" + template;
			// sql名 sqlName
			long tick = System.currentTimeMillis();

			if (i == 0) {
				export(sqlName, condition, template, filepath);
			} else {
				export(sqlName, defaultCondition, template, filepath);
			}

			tick = System.currentTimeMillis() - tick;
			System.out.println("进行Excel导出耗时：" + (tick / 1000.0) + "秒");
			System.out.println("文件名：" + result);
		}

		return result;
	}

	/**
	 * 取得导出文件名称
	 * @param name
	 * @return
	 */
	private String getFilePath(String name) {
		String excelFileName = (name == null ? UUID.randomUUID() + ".xlsx" : name);
		String path = ExcelUtil.class.getClassLoader().getResource("config.json")
				.getPath();
		String rootPath = path.split("WEB-INF")[0];
		String filePath = rootPath + "excel/" + excelFileName;
		return filePath;
	}

	/**
	 * Real export stuff
	 * 
	 * @param name
	 * @param templateName
	 * @throws Exception
	 */
	private void export(String name, String body, String templateName,
			String filePath) throws Exception {

		JSONObject joParam = new JSONObject(body);

		String[][] footer = null;
		if (joParam.has("total")) {
			JSONArray totals = joParam.getJSONArray("total");
			footer = getFooter(totals);
		}

		// 获取原始sql语句
		JSONObject joVariable = joParam.getJSONObject("data");
		String sql = new SqlServer().call(name, joVariable);

		// open connection
		Connection connection = null;
		PreparedStatement preparedStatement = null;
		ResultSet rs = null;

		try {
			// fetch data begins
			connection = dataSource.getConnection();
			preparedStatement = connection.prepareStatement(sql);
			rs = preparedStatement.executeQuery();
			rs.setFetchSize(MAX_ROWS_PER_SHEET);
			if (templateName == null) {
                exportWithHedear(filePath, joParam, sql, rs, footer);
            } else {
                exportWithTemplate(templateName, filePath, rs, footer);
            }
		} finally {
			if (rs != null) {
                rs.close();
            }
			if (preparedStatement != null) {
                preparedStatement.close();
            }
			if (connection != null) {
                connection.close();
            }
		}
	}

	/**
	 * get footer lines
	 * 
	 * @param totals
	 * @return
	 */
	private String[][] getFooter(JSONArray totals) {
		int n = totals.length();
		String[][] footer = new String[n][];
		for (int i = 0; i < n; i++) {
			JSONObject jo = totals.getJSONObject(i);
			String[] names = JSONObject.getNames(jo);
			Arrays.sort(names, new Comparator<String>() {
				@Override
				public int compare(String o1, String o2) {
					return Integer.parseInt(o1) - Integer.parseInt(o2);
				}
			});
			footer[i] = new String[names.length];
			for (int j = 0; j < names.length; j++) {
				if (jo.isNull(names[j])) {
                    footer[i][j] = null;
                } else {
                    footer[i][j] = jo.get(names[j]) + "";
                }
			}
		}
		return footer;
	}

	/**
	 * export with template
	 * 
	 * @param templateName
	 * @param filePath
	 * @param footer
	 * @throws Exception
	 */
	private void exportWithTemplate(String templateName,String filePath, ResultSet rs, String[][] footer) throws Exception {
		// creating excel starts
		ExcelUtil eu = new ExcelUtil();
		eu.createBook(templateName, filePath);
		int nCol = rs.getMetaData().getColumnCount();
		int n = 1;
		List<List<Object>> rows = new ArrayList<List<Object>>();
		while (rs.next()) {
			if (n % MAX_ROWS_PER_SHEET == 0) {
				eu.createSheet();
				n = 1;
				eu.writeToSheet(rows, null, footer);
				rows = new ArrayList<List<Object>>();
			}
			List<Object> fieldList = new ArrayList<Object>();
			for (int i = 1; i <= nCol; i++) {
				Object object = rs.getObject(i);
				fieldList.add(object);
			}
			rows.add(fieldList);
			n++;
		}
		// not enough to reach the max row limit
		if (n > 1) {
			eu.createSheet();
			eu.writeToSheet(rows, null, footer);
		}
		eu.saveBook();
	}

	/**
	 * export with header info
	 * 
	 * @param filePath
	 * @param joParam
	 * @param sql
	 * @param rs
	 * @param footer
	 * @throws Exception
	 * @throws SQLException
	 */
	private void exportWithHedear(String filePath, JSONObject joParam,
			String sql, ResultSet rs, String[][] footer) throws Exception,
			SQLException {
		boolean fromHbm = false;

		// find out the headers
		JSONArray joField = joParam.getJSONArray("field");
		Map<String, String> colsMap = new LinkedHashMap<String, String>();
		findOutHeaders(joField, colsMap);
		String[] header = colsMap.values().toArray(new String[colsMap.size()]);
		String[] cols = colsMap.keySet().toArray(new String[colsMap.size()]);

		// creating excel starts
		ExcelUtil eu = new ExcelUtil();
		eu.createBook(null, filePath);

		int n = 1;
		List<List<Object>> rows = new ArrayList<List<Object>>();
		while (rs.next()) {
			if (n % MAX_ROWS_PER_SHEET == 0) {
				eu.createSheet();
				n = 1;
				eu.writeToSheet(rows, header, footer);
				rows = new ArrayList<List<Object>>();
			}
			List<Object> fieldList = new ArrayList<Object>();
			if (fromHbm) {
                fieldList.add(rs.getObject("id"));
            }
			for (int i = (fromHbm ? 1 : 0); i < colsMap.size(); i++) {
				Object object = rs.getObject(cols[i]);
				fieldList.add(object);
			}
			rows.add(fieldList);
			n++;
		}
		// not enough to reach the max row limit
		if (n > 1) {
			eu.createSheet();
			eu.writeToSheet(rows, header, footer);
		}
		eu.saveBook();
	}

	/**
	 * find out headers
	 * 
	 * @param joField
	 * @param headers
	 */
	private void findOutHeaders(JSONArray joField, Map<String, String> headers) {
		for (Object temp : joField) {
			String col = ((String) temp).split(":")[0];
			headers.put(col, ((String) temp).split(":")[1]); 
		}
	}

}
