package com.aote.webmeter.tools.iot.ctwing;

import com.af.expression.exception.ServiceException;
import com.af.plugins.HttpConnectionPoolUtil;
import com.af.plugins.HttpDeleteWithBody;
import com.aote.redis.RedisService;
import com.aote.redis.RedisUtil;
import com.aote.webmeter.enums.CTWingPlatformType;
import com.aote.webmeter.enums.IOTBusinessTypeEnum;
import com.aote.webmeter.enums.NotifyTypeEnum;
import com.aote.webmeter.enums.WebmeterPropertiesIOTEnum;
import com.aote.webmeter.tools.WebMeterInfo;
import org.apache.http.client.methods.HttpEntityEnclosingRequestBase;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.client.CloseableHttpClient;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManagerFactory;
import java.io.IOException;
import java.io.InputStream;
import java.security.*;
import java.security.cert.CertificateException;
import java.util.function.Supplier;

/**
 * 电信物联网API实现(OC/OC2/OC兼容AEP模式)
 *
 * @apiNote 仅作为对OC/AEP兼容OC模式的兼容
 */
@Component
public class CTWingApiService {
    /**
     * 错误标识
     */
    public static final String ERROR_PARAMS_KEY = "error_code";
    public static final String ERROR_PARAMS_KEY_2 = "resultcode";
    private static final ThreadLocal<CTWingPlatformType> PLATFORM_TYPE_THREAD_LOCAL = new ThreadLocal<>();
    private static final String SELF_CERT_PWD = "IoM@1234";
    private static final String TRUST_CA_PWD = "Huawei@123";
    /**
     * CTWingApiToken缓存key名
     */
    private static final String CTWing_API_TOKEN_CACHE_KEY = "ctwingApiToken@";
    private static final String CTWing_API_REFRESH_TOKEN_CACHE_KEY = "ctwingApiRefreshToken@";
    /**
     * 错误码：鉴权错误
     */
    private static final String AUTH_ERROR = "1010005";
    /**
     * 错误码：设备已被绑定
     */
    private static final String DEVICE_BINDED_1 = "100416";
    private static final String DEVICE_BINDED_2 = "100426";
    private static final String DEVICE_BINDED_3 = "1104";
    /**
     * 错误码：设备已被解绑
     */
    private static final String DEVICE_CANCEL_BINDED_1 = "100403";
    private static final String DEVICE_CANCEL_BINDED_2 = "100418";
    private static final Logger LOGGER = LoggerFactory.getLogger(CTWingApiService.class);
    /**
     * 发送请求的客户端单例
     */
    private static volatile CloseableHttpClient httpClient;

    /**
     * 获取HTTP客户端单例
     */
    private static CloseableHttpClient getHttpClient() {
        if (httpClient == null) {
            //多线程下同时调用getHttpClient容易导致重复创建httpClient对象的问题,所以加上了同步锁
            synchronized (CTWingApiService.class) {
                if (httpClient == null) {
                    httpClient = HttpConnectionPoolUtil.getHttpClient(initSslConfig());
                }
            }
        }
        return httpClient;
    }

    private static String getCTWingApiTokenRedisKey(String appid) {
        return CTWing_API_TOKEN_CACHE_KEY + appid;
    }

    private static String getCTWingApiRefreshTokenRedisKey(String appid) {
        return CTWing_API_REFRESH_TOKEN_CACHE_KEY + appid;
    }

    /**
     * 初始化SSL证书
     *
     * @return SSL连接工厂
     */
    private static SSLConnectionSocketFactory initSslConfig() {
        try (InputStream isTrustCa = CTWingApiService.class.getResourceAsStream("/iot/ctwing_ca.jks");
             InputStream isSelfCert = CTWingApiService.class.getResourceAsStream("/iot/ctwing_CertwithKey.pkcs12")) {
            KeyStore selfCert = KeyStore.getInstance("pkcs12");
            selfCert.load(isSelfCert, SELF_CERT_PWD.toCharArray());
            KeyManagerFactory kmf = KeyManagerFactory.getInstance("sunx509");
            kmf.init(selfCert, SELF_CERT_PWD.toCharArray());
            KeyStore caCert = KeyStore.getInstance("jks");
            caCert.load(isTrustCa, TRUST_CA_PWD.toCharArray());
            TrustManagerFactory tmf = TrustManagerFactory.getInstance("sunx509");
            tmf.init(caCert);
            SSLContext sc = SSLContext.getInstance("TLS");
            sc.init(kmf.getKeyManagers(), tmf.getTrustManagers(), null);
            return new SSLConnectionSocketFactory(sc, NoopHostnameVerifier.INSTANCE);
        } catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException |
                 UnrecoverableKeyException | KeyManagementException e) {
            throw new RuntimeException(e);
        }
    }

    public static CTWingPlatformType getPlatformType() {
        return PLATFORM_TYPE_THREAD_LOCAL.get();
    }

    public static void setPlatformType(CTWingPlatformType type) {
        PLATFORM_TYPE_THREAD_LOCAL.set(type);
    }

    public static void setPlatformType(String typeStr) {
        PLATFORM_TYPE_THREAD_LOCAL.set(CTWingPlatformType.toType(Integer.valueOf(typeStr)));
    }

    /**
     * 请求获取token信息
     *
     * @param appid  appid
     * @param secret secret
     * @return TOKEN
     */
    private String requestGetToken(String appid, String secret) {
        RedisService redisService = RedisUtil.getInstance();
        JSONObject result;
        //组织AppId和密钥
        JSONObject content = new JSONObject();
        content.put("appId", appid);
        content.put("secret", secret);
        //先尝试获取刷新鉴权密钥
        String refreshTokenKey = getCTWingApiRefreshTokenRedisKey(appid);
        Object refreshToken = redisService.get(refreshTokenKey);
        if (refreshToken != null) {
            content.put("refreshToken", refreshToken);
            result = post(IOTBusinessTypeEnum.REFRESH_AUTH, content, null);
            if (result.has("msg")) {
                //如果刷新鉴权失败，则清理刷新鉴权密钥，重新鉴权
                redisService.delete(refreshTokenKey);
                return requestGetToken(appid, secret);
            } else {
                //取到新密钥和刷新密钥
                String newAccessToken = result.getString("accessToken");
                String newRefreshToken = result.getString("refreshToken");
                redisService.set(getCTWingApiTokenRedisKey(appid), newAccessToken, result.getInt("expiresIn") - 60);
                //如果取到的刷新密钥和之前的不一致，直接进行更新
                if (!refreshToken.equals(newRefreshToken)) {
                    redisService.set(refreshTokenKey, newRefreshToken, (3600 * 24) - 60);
                }
                return newAccessToken;
            }
        } else {
            result = post(IOTBusinessTypeEnum.AUTH, content, null);
            if (result.has("msg")) {
                //如果鉴权都失败了，那只能抛出异常了
                Object data = result.opt("data");
                String msg;
                if (data != null) {
                    msg = new JSONObject(data.toString()).optString("error_desc");
                } else {
                    msg = result.optString("msg", null);
                }
                throw new ServiceException("天翼NB-IOT平台鉴权失败, appId: " + appid + ", error: " + msg);
            }
            String newRefreshToken = result.getString("refreshToken");
            //存储刷新鉴权用的密钥，然后再次调用一遍该方法进行刷新鉴权，避免取到旧的密钥
            redisService.set(refreshTokenKey, newRefreshToken, (3600 * 24) - 60);
            return requestGetToken(appid, secret);
        }
    }

    /**
     * 获取缓存最后尝试鉴权时间的key
     * @param appId appId
     * @return key
     */
    private String getTokenLastAttemptTimeCacheKey(String appId) {
        return CTWing_API_TOKEN_CACHE_KEY + "LastAttemptTime" + appId;
    }

    /**
     * 获取TOKEN
     *
     * @param appid  appid
     * @param secret secret
     * @param force 表示跳过缓存命中，但仍受 refresh cooldown 限制，防止集群多实例刷新 token。
     * @return TOKEN值
     */
    private String getAuthToken(String appid, String secret, boolean force) {
        RedisService redisService = RedisUtil.getInstance();
        String key = getCTWingApiTokenRedisKey(appid);

        Supplier<String> getTokenByCache = () -> {
            Object cachedToken = redisService.get(key);
            if (cachedToken != null) {
                LOGGER.info("{}:获取缓存 token", appid);
                return cachedToken.toString();
            }
            return null;
        };

        // 1. 非强制策略下先取缓存
        if (!force) {
            String tokenCache = getTokenByCache.get();
            if (tokenCache != null) {
                return tokenCache;
            }
        }

        final String[] resultToken = { null };

        try {
            redisService.lock(key, () -> {

                long now = System.currentTimeMillis();

                // 2. 非强制策略下再次确认缓存
                if (!force) {
                    String tokenCache = getTokenByCache.get();
                    if (tokenCache != null) {
                        resultToken[0] = tokenCache;
                        return;
                    }
                }

                // 3. 检查鉴权冷却
                String timeKey = getTokenLastAttemptTimeCacheKey(appid);
                Object lastTime = redisService.get(timeKey);
                long last = lastTime == null ? 0 : Long.parseLong(lastTime.toString());
                if (now - last < 2 * 60_000) {
                    // 冷却期间如果已经存在token，说明可能已经有线程得到了新Token，直接返回
                    String tokenCache = getTokenByCache.get();
                    if (tokenCache != null) {
                        resultToken[0] = tokenCache;
                        return;
                    } else {
                        throw new RuntimeException(appid + ": 没有得到任何的Token，且处于鉴权冷却期，等待2分钟");
                    }
                }

                LOGGER.info("天翼NB-IOT平台：{}进行鉴权操作", appid);

                try {
                    // 4. 调用鉴权接口
                    String newToken = requestGetToken(appid, secret);
                    resultToken[0] = newToken;
                } finally {
                    redisService.set(timeKey, String.valueOf(System.currentTimeMillis()), 300);
                }
            });
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }

        return resultToken[0];
    }

    private String getAuthToken(String appid, String secret) {
        return getAuthToken(appid, secret, false);
    }

    /**
     * 发送命令
     *
     * @param serviceId 服务ID
     * @param method    命令名称
     * @param deviceId  设备ID
     * @param jo        命令参数JSON
     * @return 请求结果JSON
     */
    public JSONObject sendInstruct(String serviceId, String method, String deviceId, Object jo) {
        CTWingPlatformType platformType = CTWingApiService.getPlatformType();
        // 透传模式
        if (jo instanceof String) {
            JSONObject object = new JSONObject();
            object.put("rawData", jo.toString());
            jo = object;
            if (platformType != CTWingPlatformType.AEP || serviceId == null) {
                serviceId = "RawData";
            }
        }
        String callbackUrl;
        if (platformType == CTWingPlatformType.AEP) {
            callbackUrl = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.AEP_CALLBACK_URL.getValue());
        } else if (platformType == CTWingPlatformType.OC) {
            callbackUrl = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC_CALLBACK_URL.getValue());
        } else {
            callbackUrl = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC2_CALLBACK_URL.getValue());
        }
        Integer maxRetransmit = 3;

        JSONObject paramCommand = new JSONObject();
        paramCommand.put("serviceId", serviceId);
        paramCommand.put("method", method);
        paramCommand.put("paras", jo);

        JSONObject content = new JSONObject();
        content.put("deviceId", deviceId);
        content.put("command", paramCommand);
        content.put("callbackUrl", callbackUrl);
        content.put("expireTime", 0);
        content.put("maxRetransmit", maxRetransmit);

        JSONObject response = post(IOTBusinessTypeEnum.SEND_COMMAND, content, null);
        JSONObject result = new JSONObject();
        if (response.has("msg")) {
            result.put("code", -1).put("msg", response.get("msg"));
        } else {
            result.put("code", 0).put("commandId", response.get("commandId"));
        }
        return result;
    }

    /**
     * 发送订阅请求
     *
     * @param notifyTypeEnum 订阅类型
     * @param callBackUrl    回调地址
     * @return 请求结果JSON
     */
    public JSONObject subscribe(NotifyTypeEnum notifyTypeEnum, String callBackUrl) {
        JSONObject content = new JSONObject();
        content.put("notifyType", notifyTypeEnum.getValue());
        content.put("callbackUrl", callBackUrl);

        JSONObject result = post(IOTBusinessTypeEnum.SUBSCRIPTIONS, content, null);

        if (result.has("msg")) {
            LOGGER.error("IOT接口调用错误！，信息：{}", result);
            return null;
        } else {
            return result;
        }
    }

    /**
     * 删除所有订阅
     *
     * @return 请求结果JSON
     */
    public JSONObject deleteAllSubscribe() {
        JSONObject result = post(IOTBusinessTypeEnum.DELETE_ALL_SUBSCRIPTIONS, null, null);

        if (result.has("msg")) {
            LOGGER.error("IOT接口调用错误！，信息：{}", result);
            return null;
        } else {
            return result;
        }
    }

    /**
     * 注册设备
     */
    public JSONObject regDevice(String nodeId, String name) throws IOException {
        JSONObject content = new JSONObject();
        content.put("nodeId", nodeId);
        content.put("verifyCode", nodeId);
        content.put("timeout", 0);
        content.put("deviceName", name);

        JSONObject deviceInfo = new JSONObject();
        deviceInfo.put("manufacturerId", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.MANUFACTURER_ID));
        deviceInfo.put("manufacturerName", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.MANUFACTURER_NAME));
        deviceInfo.put("deviceType", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.DEVICE_TYPE));
        deviceInfo.put("model", WebMeterInfo.getString(WebmeterPropertiesIOTEnum.METER_MODEL));
        deviceInfo.put("protocolType", "CoAP");

        content.put("deviceInfo", deviceInfo);
        JSONObject result = new JSONObject();
        JSONObject resultData = post(IOTBusinessTypeEnum.CREATE_DEVICE, content);
        if (resultData.has("msg")) {
            JSONObject errorEntity = new JSONObject(resultData.get("data").toString());
            String errorKey = errorEntity.get(CTWingApiService.ERROR_PARAMS_KEY).toString();
            if (DEVICE_BINDED_1.equals(errorKey) || DEVICE_BINDED_2.equals(errorKey) || DEVICE_BINDED_3.equals(errorKey)) {
                result.put("code", -1).put("msg", "尝试注册时已绑定");
            } else {
                result.put("code", -1).put("msg", "注册IOT设备失败：" + errorEntity);
            }
        } else {
            result.put("code", 0).put("msg", resultData.getString("deviceId"));
        }
        return result;
    }

    /**
     * 注销设备
     */
    public JSONObject removeDevice(String deviceId) {
        JSONObject result = new JSONObject();
        JSONObject resultData = post(IOTBusinessTypeEnum.REMOVE_DEVICE, null, deviceId);
        if (resultData.has("msg")) {
            JSONObject errorEntity = new JSONObject(resultData.get("data").toString());
            String code = errorEntity.get(CTWingApiService.ERROR_PARAMS_KEY).toString();
            if (DEVICE_CANCEL_BINDED_1.equals(code) || DEVICE_CANCEL_BINDED_2.equals(code)) {
                result.put("code", 0).put("msg", "尝试删除时已不存在");
            } else {
                result.put("code", -1).put("msg", "删除IOT设备失败：" + errorEntity);
            }
        } else {
            result.put("code", 0).put("msg", "成功");
        }
        return result;
    }

    private JSONObject post(IOTBusinessTypeEnum typeEnum, JSONObject contentObj) {
        return post(typeEnum, contentObj, null);
    }

    private JSONObject post(IOTBusinessTypeEnum typeEnum, JSONObject contentObj, String urlAppend) {
        CTWingPlatformType platformType = getPlatformType();
        String appid;
        String secret;
        if (platformType == CTWingPlatformType.AEP) {
            appid = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.AEP_APIKEY);
            secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.AEP_SECRET);
        } else if (platformType == CTWingPlatformType.OC) {
            appid = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC_APIKEY);
            secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC_SECRET);
        } else {
            appid = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC2_APIKEY);
            secret = WebMeterInfo.getString(WebmeterPropertiesIOTEnum.OC2_SECRET);
        }
        String header = null;
        if (typeEnum != IOTBusinessTypeEnum.AUTH) {
            JSONObject headerObj = new JSONObject();
            if (typeEnum != IOTBusinessTypeEnum.REFRESH_AUTH) {
                String accessToken = getAuthToken(appid, secret);
                headerObj.put("app_key", appid);
                headerObj.put("Authorization", "Bearer" + " " + accessToken);
            }
            headerObj.put("Content-Type", "application/json");
            header = headerObj.toString();
        }

        String url = null;
        CloseableHttpClient client;
        HttpEntityEnclosingRequestBase request = null;
        switch (typeEnum) {
            case AUTH:
                url = CTWingApi.getAuthUrl();
                request = new HttpPost();
                break;
            case REFRESH_AUTH:
                url = CTWingApi.getRefreshAuthUrl();
                request = new HttpPost();
                break;
            case CREATE_DEVICE:
                url = CTWingApi.getDeviceCreateUrl();
                request = new HttpPost();
                break;
            case MODIFY_DEVICE:
                url = CTWingApi.getDeviceUpdateUrl() + "/" + urlAppend;
                request = new HttpPost();
                break;
            case REMOVE_DEVICE:
                url = CTWingApi.getDeviceUpdateUrl() + "/" + urlAppend;
                request = new HttpDeleteWithBody();
                break;
            case SEND_COMMAND:
                url = CTWingApi.getSendInstructUrl();
                request = new HttpPost();
                break;
            case SUBSCRIPTIONS:
                url = CTWingApi.getSubscriptionsUrl();
                request = new HttpPost();
                break;
            case DELETE_ALL_SUBSCRIPTIONS:
                url = CTWingApi.getSubscriptionsUrl();
                request = new HttpDeleteWithBody();
                break;
        }
        if (url.startsWith("https")) {
            client = CTWingApiService.getHttpClient();
        } else {
            client = HttpConnectionPoolUtil.getHttpClient();
        }

        if (contentObj == null){
            contentObj = new JSONObject();
        }
        String responseBody;
        JSONObject result;
        String log = "api: " + url + "，header: " + header + "，body: " + contentObj;
        try {
            if (typeEnum == IOTBusinessTypeEnum.AUTH) {
                responseBody = HttpConnectionPoolUtil.requestFormUrlEncoded(url, contentObj, request, client);
            } else {
                responseBody = HttpConnectionPoolUtil.request(url, contentObj.toString(), header, request, client);
            }
            log = log + "，response: " + responseBody;
            // 处理删除设备没有响应值的情况
            if (responseBody == null) {
                return new JSONObject();
            }
            result = new JSONObject(responseBody);
            // V4逻辑兼容V3版本
            if (result.has("errorEntity")) {
                result.put("msg", "error");
                result.put("data", result.get("errorEntity"));
            }
        } catch (IOException e) {
            result = new JSONObject();
            result.put("msg", e.getMessage());
            log = log + "，response: " + e.getMessage();
        } finally {
            LOGGER.info(log);
        }
        if (typeEnum != IOTBusinessTypeEnum.AUTH
                && typeEnum != IOTBusinessTypeEnum.REFRESH_AUTH
                && result.has("msg")) {
            if (result.has("data")) {
                String entity = result.get("data").toString();
                JSONObject errorEntity = new JSONObject(entity);
                String code;
                if (errorEntity.has(ERROR_PARAMS_KEY)) {
                    code = errorEntity.get(ERROR_PARAMS_KEY).toString();
                } else {
                    code = errorEntity.get(ERROR_PARAMS_KEY_2).toString();
                }
                // 如果出现鉴权错误，重新进行鉴权操作
                if (AUTH_ERROR.equals(code)) {
                    String auth = getAuthToken(appid, secret, true);
                    if (auth != null) {
                        return post(typeEnum, contentObj, urlAppend);
                    }
                }
            } else {
                result.put("code", -1).put("msg", result.getString("msg"));
            }
        }
        return result;
    }
}
