package com.af.v4.system.runtime.controller;

import com.af.v4.system.common.core.constant.CodeNormsConstants;
import com.af.v4.system.common.core.constant.HttpStatus;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.utils.StringUtils;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.plugins.core.ConvertTools;
import com.af.v4.system.common.redis.RedisService;
import com.af.v4.system.common.resource.constant.CacheKey;
import com.af.v4.system.common.resource.mapper.LogicMapper;
import com.af.v4.system.runtime.annotation.OpenApi;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import java.nio.charset.StandardCharsets;
import java.util.*;

/**
 * Logic控制器
 *
 * @author Mr.river
 * @since 1.0.0
 */
@RestController
@RequestMapping("/logic")
public class LogicController {
    private static final Logger LOGGER = LoggerFactory.getLogger(LogicController.class);
    private static final String RESPONSE_HEADER = "header";
    private static final String RESPONSE_CONTENT = "content";
    private static final String RESPONSE_VALUE = "value";
    private final LogicService logicService;
    private final LogicMapper logicMapper;
    private final RedisService redisService;

    public LogicController(LogicService logicService, LogicMapper logicMapper, RedisService redisService) {
        this.logicService = logicService;
        this.logicMapper = logicMapper;
        this.redisService = redisService;
    }

    @Log(title = "调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Object run(@PathVariable("logicName") String logicName,
                      @RequestBody(required = false) String values,
                      HttpServletResponse response,
                      HttpServletRequest request) {
        String params = getRequestParams(values, request);
        return runMain(logicName, params, request, response);
    }

    @Log(title = "异步调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/async/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    public Object runAsync(@PathVariable("logicName") String logicName,
                           @RequestBody(required = false) String values,
                           HttpServletRequest request) {
        String params = getRequestParams(values, request);
        logicService.runAsync(logicName, params).whenComplete((o, e) -> {
            if (o != null) {
                LOGGER.info("异步调用结果：{}", o);
            }
        });
        return R.ok("ok");
    }

    @Log(title = "无token调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/openapi/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    @OpenApi
    public Object runByNoToken(@PathVariable("logicName") String logicName,
                               @RequestBody(required = false) String values,
                               HttpServletResponse response,
                               HttpServletRequest request) {
        return run(logicName, values, response, request);
    }

    @Log(title = "无token异步调用Logic", businessType = BusinessType.LOGIC)
    @RequestMapping(value = "/openapi/async/{logicName}", produces = MediaType.APPLICATION_JSON_VALUE)
    @OpenApi
    public Object runAsyncByNoToken(@PathVariable("logicName") String logicName,
                                    @RequestBody(required = false) String values,
                                    HttpServletRequest request) {
        return runAsync(logicName, values, request);
    }

    /**
     * 获取请求参数
     *
     * @param values  请求体内容
     * @param request HttpServletRequest
     * @return 请求参数
     */
    private String getRequestParams(String values, HttpServletRequest request) {
        JSONObject paramsJson;
        if (StringUtils.isEmpty(values)) {
            return new JSONObject().toString();
        }
        if (values.startsWith("{") && values.endsWith("}")) {
            paramsJson = new JSONObject(values);
        } else {
            paramsJson = new JSONObject();
            paramsJson.put("str", values);
            //加入请求头数据
            Enumeration<String> headerNames = request.getHeaderNames();
            if (headerNames != null) {
                JSONObject header = new JSONObject();
                while (headerNames.hasMoreElements()) {
                    String temp = headerNames.nextElement();
                    header.put(temp, request.getHeader(temp));
                }
                paramsJson.put("header", header);
            }
        }
        //加入请求地址参数
        Map<String, String[]> parameterMap = request.getParameterMap();
        parameterMap.forEach((key, value) -> {
            if (value.length > 0 && !paramsJson.has(key)) {
                paramsJson.put(key, value[0]);
            }
        });
        return paramsJson.toString();
    }

    /**
     * Logic主执行入口
     *
     * @param logicName logic名称
     * @param values    参数
     * @param response  HttpServletResponse
     * @return 执行结果
     */
    private Object runMain(String logicName, String values, HttpServletRequest request, HttpServletResponse response) {
        String requestId = request.getHeader("X-Request-ID");
        if (requestId != null) {
            LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "请求唯一标识：{}", requestId);
        }
        // 获取Logic资源
        LogicMapper.LogicResource resource = logicMapper.getResource(logicName);
        boolean isEnableCache = resource.isCache();
        if (!isEnableCache) {
            Object runResult = logicService.run(logicName, values);
            return buildResponse(logicName, runResult, response);
        }
        // 获取自定义缓存Key
        String cacheKeyValue = resource.getCacheKey();
        String paramsStr;
        if (!values.startsWith("{")) {
            LOGGER.warn(CodeNormsConstants.DEBUG_PREFIX + "Logic[{}}]的请求参数不满足获取自定义缓存Key，已跳过获取缓存", logicName);
            Object runResult = logicService.run(logicName, values);
            return buildResponse(logicName, runResult, response);
        }
        JSONObject params = new JSONObject(values);
        if (Optional.ofNullable(cacheKeyValue).isPresent() && params.has(cacheKeyValue)) {
            // 如果调用Logic的参数中含有该key，则取该key对应的值，否则将传入参数的base值作为缓存key
            paramsStr = params.get(cacheKeyValue).toString();
        } else {
            paramsStr = ConvertTools.base64Encode(new TreeMap<>(params.toMap()).toString().getBytes(StandardCharsets.UTF_8));
        }
        String cacheKey = CacheKey.getLogicDataCacheKey(logicName, paramsStr);
        Object cacheResult = redisService.get(cacheKey);
        if (cacheResult != null) {
            LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[{}]命中缓存，参数：{}", logicName, values);
            return buildResponse(logicName, cacheResult, response);
        }
        try {
            long cacheTime = resource.getTime();
            Object result = redisService.lock(cacheKey, cacheTime <= 0 ? 0 : 60, 300, () -> {
                if (cacheTime <= 0) {
                    return logicService.run(logicName, values);
                }
                Object newCacheResult = redisService.get(cacheKey);
                if (newCacheResult != null) {
                    LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[{}]命中缓存，参数：{}", logicName, values);
                    return newCacheResult;
                }
                newCacheResult = logicService.run(logicName, values);
                if (newCacheResult != null) {
                    redisService.set(cacheKey, newCacheResult, cacheTime);
                    LOGGER.info(CodeNormsConstants.DEBUG_PREFIX + "Logic[{}]缓存成功，存活时间：{}秒", logicName, cacheTime);
                }
                return newCacheResult;
            });
            return buildResponse(logicName, result, response);
        } catch (InterruptedException e) {
            String demoteLogicName = resource.getDemoteLogicName();
            if (demoteLogicName != null) {
                //由于锁占用，业务降级处理
                return buildResponse(logicName, logicService.run(demoteLogicName, values), response);
            } else {
                throw new ServiceException("重复请求", HttpStatus.CONFLICT);
            }
        }
    }

    private Object buildResponse(String logicName, Object runResult, HttpServletResponse response) {
        // 如果Logic返回值中包含header和content关键字，则认为需要给调用方指定的响应头
        if (runResult instanceof JSONObject object && object.has(RESPONSE_HEADER) && object.has(RESPONSE_CONTENT)) {
            JSONObject paramHeader = object.getJSONObject(RESPONSE_HEADER);
            for (String key : paramHeader.keySet()) {
                response.setHeader(key, String.valueOf(paramHeader.get(key)));
            }
            runResult = object.get(RESPONSE_CONTENT);
        }
        // 转换响应结果为标准类型
        Object result;
        if (runResult instanceof JSONObject json) {
            result = json.toMap();
        } else if (runResult instanceof JSONArray array) {
            result = array.toList();
        } else if (runResult instanceof Map<?, ?> || runResult instanceof Collection<?>) {
            result = runResult;
        } else {
            if (runResult == null) {
                result = null;
            } else {
                Map<String, Object> map = new HashMap<>(1);
                map.put(RESPONSE_VALUE, runResult);
                result = map;
            }
        }

        // 如果为第三方回调接口，响应体不进行封装处理
        LogicMapper.LogicResource resource = logicMapper.getResource(logicName);
        if (resource.isCallbackApi()) {
            return result;
        }

        // 如果返回的是标准的响应格式，则根据响应码，返回对应的标准响应体
        if (result instanceof Map<?, ?> map) {
            if (map.containsKey("status")) {
                String status = map.get("status").toString();
                if (status.equals("200") && map.containsKey("data")) {
                    return R.ok(map.get("data"));
                } else if (!status.equals("200") && map.containsKey("msg")) {
                    return R.fail(status, (String) map.get("msg"));
                }
            }
        }

        return R.ok(result);
    }
}
