package com.af.v4.system.runtime.upgrade;

import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.liuli.utils.ApplicationUtils;
import com.af.v4.system.common.redis.RedisService;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.TreeMap;

@Component
public final class UpgradePathManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(UpgradePathManager.class);
    private static final TreeMap<Integer, UpgradePathExecutor> UPGRADE_PATH_EXECUTORS = new TreeMap<>();
    private final RedisService redisService;
    private final ApplicationUtils applicationUtils;
    private final BaseUpgradeAppBootstrap bootstrap;

    public UpgradePathManager(RedisService redisService, ApplicationUtils applicationUtils, BaseUpgradeAppBootstrap bootstrap) {
        this.redisService = redisService;
        this.applicationUtils = applicationUtils;
        this.bootstrap = bootstrap;
    }

    // 注册升级路径执行器
    public static void register(UpgradePathExecutor executor) {
        UPGRADE_PATH_EXECUTORS.put(executor.targetVersionCode(), executor);
    }

    // 执行升级路径
    public void executeUpgrade() {
        LOGGER.info("检查服务升级...");
        String cacheKey = "System@AppVersion-" + applicationUtils.getApplicationName();
        String fromVersion = redisService.get(cacheKey);
        int fromVersionCode;
        if (fromVersion == null) {
            fromVersionCode = ApplicationUtils.getAppVersionCode();
        } else {
            fromVersionCode = ApplicationUtils.buildVersionCode(fromVersion);
        }
        String toVersion = ApplicationUtils.getAppVersion();
        int toVersionCode = ApplicationUtils.getAppVersionCode();
        if (fromVersionCode >= toVersionCode) {
            LOGGER.info("没有需要升级的内容.");
            redisService.set(cacheKey, toVersion);
            return;
        }
        Map<Integer, UpgradePathExecutor> upgradePathExecutorMap = UPGRADE_PATH_EXECUTORS.subMap(fromVersionCode, false, toVersionCode, true);
        if (upgradePathExecutorMap.isEmpty()) {
            LOGGER.info("没有需要升级的内容.");
            redisService.set(cacheKey, toVersion);
            return;
        } else {
            LOGGER.info("待升级数据包数量：{}", upgradePathExecutorMap.size());
        }
        for (UpgradePathExecutor executor : upgradePathExecutorMap.values()) {
            LOGGER.info("开始进行服务升级 ===============> {} to {}", fromVersion, executor.targetVersion());
            try {
                executor.upgrade();
            } catch (Exception e) {
                LOGGER.error("服务升级失败，请检查！", e);
                try {
                    //此处睡眠是为了避免异步日志未输出完就抛出异常
                    Thread.sleep(1000);
                } catch (InterruptedException ex) {
                    throw new RuntimeException(ex);
                }
                throw new ServiceException("服务升级失败，程序已强制中止.");
            }
            redisService.set(cacheKey, executor.targetVersion());
            LOGGER.info("{} => {} done.", fromVersion, executor.targetVersion());
            fromVersion = executor.targetVersion();
        }
    }

    @PostConstruct
    public void init() {
        if (applicationUtils.getEnvType() != EnvType.LOCAL) {
            for (UpgradePathExecutor executor : bootstrap.getAllExecutor()) {
                UpgradePathManager.register(executor);
            }
            this.executeUpgrade();
        }
    }
}
