package com.af.v4.system.runtime.controller;

import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.exception.LogicException;
import com.af.v4.system.common.core.proxy.excel.IExcelServiceProxy;
import com.af.v4.system.common.datasource.DynamicDataSource;
import com.af.v4.system.common.jpa.service.EntityService;
import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.log.annotation.Log;
import com.af.v4.system.common.log.enums.BusinessType;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.common.plugins.core.CommonTools;
import com.af.v4.system.common.plugins.date.DateTools;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * CRUD动态表单配置控制器
 *
 * @author Mr.river
 */
@RestController
@RequestMapping("/crud")
public class CrudController {

    private static final Logger LOGGER = LoggerFactory.getLogger(CrudController.class);

    private final EntityService entityService;

    private final LogicService logicService;

    private final LiuLiConfigService liuLiConfigService;

    private final IExcelServiceProxy excelService;

    public CrudController(EntityService entityService, LogicService logicService, LiuLiConfigService liuLiConfigService, IExcelServiceProxy excelService) {
        this.entityService = entityService;
        this.logicService = logicService;
        this.liuLiConfigService = liuLiConfigService;
        this.excelService = excelService;
    }

    /**
     * 通过excel导入数据
     *
     * @param file            二进制文件
     * @param queryParamsName 查询配置名称
     * @param userDataString  基本数据
     * @param otherParamsString 其他参数
     * @return 执行结果
     */
    @Log(title = "通过excel导入数据", businessType = BusinessType.IMPORT)
    @PostMapping(value = "/importDataByExcel", produces = MediaType.APPLICATION_JSON_VALUE)
    public R<Object> importDataByExcel(@RequestParam("file") MultipartFile file,
                                       @RequestParam("queryParamsName") String queryParamsName,
                                       @RequestParam(value = "userData", required = false) String userDataString,
                                       @RequestParam(value = "otherParams", required = false) String otherParamsString) {
        try {
            JSONObject config = liuLiConfigService.get(queryParamsName, true);
            JSONObject resultObj = new JSONObject();
            // 得到导入模板数据
            JSONArray excelImportTemplate = config.optJSONArray("excelImportTemplate");
            if (excelImportTemplate == null || excelImportTemplate.isEmpty()) {
                return R.fail("导入失败，没有配置导入列");
            }
            JSONObject apiSlot = config.optJSONObject("apiSlot");
            JSONArray titles = new JSONArray();
            JSONArray columns = new JSONArray();
            JSONObject userData = new JSONObject();
            if (userDataString != null && !userDataString.isEmpty()) {
                try {
                    userData = new JSONObject(userDataString);
                } catch (JSONException ignored) {
                    LOGGER.warn("操作人信息解析错误。");
                }
            }
            JSONObject otherParams = new JSONObject();
            if (otherParamsString != null && !otherParamsString.isEmpty()) {
                try {
                    otherParams = new JSONObject(otherParamsString);
                } catch (JSONException ignored) {
                    LOGGER.warn("otherParams 解析错误。");
                }
            }
            // 得到所有的标题和字段
            excelImportTemplate.forEach(item -> {
                JSONObject itemObj = ((JSONObject) item);
                titles.putAll(itemObj.get("title"));
                columns.putAll(itemObj.get("column"));
            });
            // 导入excel得到数据，数据是以下标为key，数据值为value的map集合
            List<Map<Integer, String>> dataList = excelService.importExcel(file.getInputStream(), titles);
            resultObj.put("successLen", dataList.size());

            // 预处理 columnJson
            JSONObject slotTypeJSON = new JSONObject();
            Optional.ofNullable(config.optJSONArray("columnJson"))
                    .ifPresent(arr -> arr.forEach(o -> {
                        JSONObject item = (JSONObject) o;
                        slotTypeJSON.put(CommonTools.substringAroundFirst(item.getString("dataIndex"), "_", true, true), item.optString("slotType", "ellipsis"));
                    }));
            // 字段类型校验失败 提示信息
            StringBuilder fieldTypeValidationErrorMessage = new StringBuilder();

            // 创建一个空的JSON数组，用于存储处理后的数据
            JSONArray datas = new JSONArray();

            // 遍历从Excel读取的原始数据列表
            for (int d = 0; d < dataList.size(); d++) {
                Map<Integer, String> data = dataList.get(d);
                // 为每一行数据创建一个JSON对象
                JSONObject entity = new JSONObject();
                // 创建一个临时对象用于存储嵌套JSON结构的数据
                JSONObject realJSONColumns = new JSONObject();

                // 遍历配置的列定义
                for (int i = 0; i < columns.length(); i++) {
                    String key = columns.getString(i);

                    // 处理包含"."的列名（表示嵌套JSON结构）
                    if (key.contains(".")) {
                        // 分割列名，如"address.city" -> ["address", "city"]
                        String[] keys = key.split("\\.");
                        String realColumnName = keys[0];  // 父字段名（如"address"）
                        String childColumnName = keys[1]; // 子字段名（如"city"）

                        // 如果父字段不存在，则创建一个新的JSON对象
                        if (!realJSONColumns.has(realColumnName)) {
                            realJSONColumns.put(realColumnName, new JSONObject());
                        }
                        // 将值放入嵌套结构中
                        realJSONColumns.getJSONObject(realColumnName).put(childColumnName, data.get(i));
                    } else {
                        String value = data.get(i); // 数据
                        // 普通字段直接放入entity对象
                        entity.put(columns.getString(i), value);

                        // 根据 slotType 判断每个字段 校验数据 是否符合类型 af-liuli SlotTypeEnum枚举
                        String slotType = slotTypeJSON.optString(key, "ellipsis");
                        switch (slotType) {
                            case "ellipsis":
                                break;
                            case "date":
                            case "dateTime": {
                                // 校验值为 日期类型
                                if (!DateTools.isValidDateTime(value)) {
                                    String errMessage = "第%d行[%s]列;".formatted(d + 1, titles.opt(i));
                                    fieldTypeValidationErrorMessage.append(errMessage);
                                }
                            }
                            default: {
                                break;
                            }
                        }
                    }
                }

                // 将处理好的嵌套结构转换为字符串并放入entity
                for (String key : realJSONColumns.keySet()) {
                    entity.put(key, realJSONColumns.getJSONObject(key).toString());
                }

                // 将处理完的一行数据加入结果集
                datas.put(entity);
            }
            if (!fieldTypeValidationErrorMessage.isEmpty()) {
                LOGGER.error("导入数据失败: {}", fieldTypeValidationErrorMessage);
                return R.fail(STR."\{fieldTypeValidationErrorMessage}数据有误!");
            }
            // 使用自定义校验 或者 自定义导入逻辑时 需要组织数据
            JSONObject SlotData = new JSONObject();
            if (apiSlot != null && !apiSlot.isEmpty()) {
                SlotData.put("config", config).put("datas", datas).put("userData", userData).put("otherParams", otherParams);
            }
            // 导入前插槽
            if (apiSlot != null && apiSlot.has("excelImportTemplateBefore")) {
                resultObj = new JSONObject(logicService.run(apiSlot.getString("excelImportTemplateBefore"), SlotData).toString());
                if (resultObj.get("status").toString().equals("1")) {
                    return R.fail(resultObj.optString("msg", "导入失败，数据校验失败"));
                }
            }
            // 是否配置了自定义导入逻辑插槽
            if (apiSlot != null && apiSlot.has("excelImportTemplateLogic")) {
                resultObj = new JSONObject(logicService.run(apiSlot.getString("excelImportTemplateLogic"), SlotData).toString());
            } else {
                // 定义一个映射，用于存储要检查的字段及它们在finalUserData中的对应键名
                Map<String, String> fieldMapping = new HashMap<>();
                fieldMapping.put("f_orgid", "orgId");
                fieldMapping.put("f_orgname", "orgName");
                fieldMapping.put("f_depid", "depId");
                fieldMapping.put("f_depname", "depName");
                fieldMapping.put("f_operatorid", "currUserId");
                fieldMapping.put("f_operator", "currUserName");
                JSONObject finalUserData = userData;
                excelImportTemplate.forEach(importTable -> {
                    JSONObject importTableObj = ((JSONObject) importTable);
                    // 得到表名
                    String tableName = importTableObj.getString("tableName");
                    LOGGER.info("导入表名：{}", tableName);
                    // 批量保存数据
                    datas.forEach(data -> {
                        try {
                            JSONObject savaData = (JSONObject) data;
                            // 遍历映射，进行字段填充
                            for (Map.Entry<String, String> entry : fieldMapping.entrySet()) {
                                if (!savaData.has(entry.getKey())) {
                                    savaData.put(entry.getKey(), finalUserData.get(entry.getValue()));
                                }
                            }
                            DynamicDataSource.withDataSource(config.optString("dataSourceName", "master"), () -> {
                                entityService.partialSave(tableName, savaData);
                            });
                        } catch (Exception e) {
                            throw new RuntimeException(e);
                        }
                    });
                });
            }
            // 导入后插槽
            if (apiSlot != null && apiSlot.has("excelImportTemplateAfter")) {
                logicService.run(apiSlot.getString("excelImportTemplateAfter"), SlotData);
            }
            return R.ok(resultObj.toMap(), "导入数据成功");
        } catch (LogicException le) {
            LOGGER.error("导入数据失败：{}", le.getStack());
            return R.fail(le.getMessage());
        } catch (Exception e) {
            LOGGER.error("导入数据失败", e);
            return R.fail("导入数据失败");
        }
    }
}
