package com.af.v4.v3.workflow.perform;

import com.af.v4.system.common.core.utils.SpringUtils;
import org.springframework.stereotype.Component;

import java.io.Serial;
import java.io.Serializable;
import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 系统运行时的流程定义管理器
 */
@Component
public class ProcessDefManager implements Serializable {
    /**
     *
     */
    @Serial
    private static final long serialVersionUID = 7046680794343910842L;

    /**
     * 流程定义名称与流程定义对照表
     */
    private final Map<String, ProcessDef> nameAndProcesses = new ConcurrentHashMap<>();

    public static ProcessDefManager getInstance() {
        return SpringUtils.getBean(ProcessDefManager.class);
    }

    /**
     * 根据登录用户，取流程目录，没有找到，返回空
     * 找到返回流程定义
     *
     * @param name 流程文件名
     * @return def  流程定义
     */
    private static ProcessDef getProcessesByUser(String name) {
        try {
            Diagram diagram = DiagramManager.getInstance().getUserDiagramByName(name);
            if (diagram == null) {
                return null;
            }
            return new ProcessDef(diagram);
        } catch (RuntimeException e) {
            throw e;
        } catch (Throwable e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 装入所有的流程定义
     */
    public void load() {
        // 取得所有的流程图
        DiagramManager manager = DiagramManager.getInstance();
        Collection<Diagram> v;
        try {
            v = manager.getDiagrams();
        } catch (Exception e) {
            throw new RuntimeException("ProcessDefManager.load()出错，装载流程图定义出错");
        }

        for (Diagram diagram : v) {
            // 对每一个流程图，产生流程定义
            try {
                nameAndProcesses.put(diagram.getName(), new ProcessDef(diagram));
            } catch (Exception e) {
                throw new RuntimeException("ProcessDefManager.load()出错，产生流程定义出错");
            }
        }
    }

    /**
     * 装入指定的流程定义
     */
    public void reLoadProcessDefByName(Diagram d) {
        // 得到旧的流程定义
        ProcessDef def = this.getProcessDef(d.getName());
        // 存在
        if (def != null) {
            def.setDiagram(d);
        }
        // 重新构造
        else {
            def = new ProcessDef(d);
        }
        nameAndProcesses.put(d.getName(), def);
    }

    /**
     * 根据名字得到流程定义
     *
     * @param name 流程名称
     * @return 流程定义，不存在，返回null
     */
    public ProcessDef getProcessDef(String name) {
        // 如果登录用户有特殊流程，返回
        ProcessDef result = getProcessesByUser(name);
        if (result != null) {
            return result;
        }
        return nameAndProcesses.get(name);

    }

    /**
     * 得到所有流程定义
     *
     * @return 流程定义列表
     */
    public Collection<ProcessDef> getProcesses() {
        return nameAndProcesses.values();
    }

    /**
     * 得到所有流程定义迭代器
     *
     * @return 流程定义迭代器
     */
    public Iterator<ProcessDef> getProcessesIterator() {
        return nameAndProcesses.values().iterator();
    }

}
