package com.af.v4.system.runtime.utils;

import com.af.v4.system.common.datasource.dialects.Dialect;
import com.af.v4.system.common.datasource.enums.DbType;

import java.util.HashMap;
import java.util.Map;
import java.util.regex.Pattern;

/**
 * 数据类型工具类
 * 用于处理不同数据库之间的数据类型转换
 */
public class DataTypeUtils {

    // 数据库类型映射表
    private static final Map<DbType, Map<Pattern, String>> TYPE_MAPPINGS = new HashMap<>();

    static {
        initializeTypeMappings();
    }

    /**
     * 初始化数据类型映射表
     */
    private static void initializeTypeMappings() {
        // Oracle 数据类型映射
        Map<Pattern, String> oracleMapping = new HashMap<>();
        oracleMapping.put(Pattern.compile("datetime"), "timestamp");
        oracleMapping.put(Pattern.compile("getdate\\(\\)"), "sysdate");
        oracleMapping.put(Pattern.compile("varchar\\("), "varchar2(");
        oracleMapping.put(Pattern.compile("numeric\\("), "NUMBER(");
        oracleMapping.put(Pattern.compile("\\bint\\b"), "number(10)");
        oracleMapping.put(Pattern.compile("bigint"), "number(19)");
        oracleMapping.put(Pattern.compile("\\bbit\\b"), "number(1)");
        oracleMapping.put(Pattern.compile("text"), "clob");
        TYPE_MAPPINGS.put(DbType.oracle, oracleMapping);

        // PostgreSQL 数据类型映射
        Map<Pattern, String> postgresMapping = new HashMap<>();
        postgresMapping.put(Pattern.compile("datetime"), "timestamp");
        postgresMapping.put(Pattern.compile("getdate\\(\\)"), "current_timestamp");
        postgresMapping.put(Pattern.compile("\\bbit\\b"), "boolean");
        postgresMapping.put(Pattern.compile("uniqueidentifier"), "uuid");
        postgresMapping.put(Pattern.compile("nvarchar"), "varchar");
        postgresMapping.put(Pattern.compile("ntext"), "text");
        TYPE_MAPPINGS.put(DbType.postgresql, postgresMapping);
        TYPE_MAPPINGS.put(DbType.opengauss, postgresMapping); // OpenGauss 使用与 PostgreSQL 相同的映射

        // MySQL 数据类型映射
        Map<Pattern, String> mysqlMapping = new HashMap<>();
        mysqlMapping.put(Pattern.compile("getdate\\(\\)"), "current_timestamp");
        mysqlMapping.put(Pattern.compile("nvarchar"), "varchar");
        mysqlMapping.put(Pattern.compile("uniqueidentifier"), "char(36)");
        mysqlMapping.put(Pattern.compile("ntext"), "longtext");
        TYPE_MAPPINGS.put(DbType.mysql, mysqlMapping);
        TYPE_MAPPINGS.put(DbType.doris, mysqlMapping); // Doris 使用与 MySQL 相同的映射

        // SQL Server 数据类型映射
        Map<Pattern, String> sqlServerMapping = new HashMap<>();
        sqlServerMapping.put(Pattern.compile("timestamp"), "datetime");
        sqlServerMapping.put(Pattern.compile("sysdate"), "getdate()");
        sqlServerMapping.put(Pattern.compile("current_timestamp"), "getdate()");
        sqlServerMapping.put(Pattern.compile("varchar2"), "nvarchar");
        sqlServerMapping.put(Pattern.compile("clob"), "ntext");
        sqlServerMapping.put(Pattern.compile("boolean"), "bit");
        TYPE_MAPPINGS.put(DbType.sqlserver, sqlServerMapping);

        // 达梦数据库映射 (与 Oracle 类似)
        Map<Pattern, String> dmMapping = new HashMap<>(oracleMapping);
        TYPE_MAPPINGS.put(DbType.dm, dmMapping);

        // ClickHouse 数据类型映射
        Map<Pattern, String> clickhouseMapping = new HashMap<>();
        clickhouseMapping.put(Pattern.compile("datetime"), "DateTime");
        clickhouseMapping.put(Pattern.compile("getdate\\(\\)"), "now()");
        clickhouseMapping.put(Pattern.compile("\\bbit\\b"), "UInt8");
        clickhouseMapping.put(Pattern.compile("\\bint\\b"), "Int32");
        clickhouseMapping.put(Pattern.compile("bigint"), "Int64");
        clickhouseMapping.put(Pattern.compile("nvarchar|varchar"), "String");
        clickhouseMapping.put(Pattern.compile("ntext|text|clob"), "String");
        TYPE_MAPPINGS.put(DbType.clickhouse, clickhouseMapping);
    }

    /**
     * 根据不同数据库类型转换数据类型
     *
     * @param originalType 原始数据类型
     * @param dbType       数据库类型
     * @return 转换后的数据类型
     */
    public static String convertDataType(String originalType, DbType dbType) {
        if (originalType == null || originalType.isEmpty()) {
            return originalType;
        }

        // 获取数据库对应的映射表
        Map<Pattern, String> mappings = TYPE_MAPPINGS.get(dbType);
        if (mappings == null) {
            throw new UnsupportedOperationException(STR."不支持的数据库类型: \{dbType}");
        }

        // 应用映射规则
        String result = originalType;
        for (Map.Entry<Pattern, String> entry : mappings.entrySet()) {
            result = entry.getKey().matcher(result).replaceAll(entry.getValue());
        }

        return result;
    }

    /**
     * 根据数据库方言获取适当大小写的表名
     */
    public static String getTableNameWithCase(String tableName, Dialect dialect) {
        if (dialect.isOracleFamily()) {
            return tableName.toUpperCase();
        } else if (dialect.isPostgresFamily()) {
            return tableName.toLowerCase();
        } else {
            return tableName;
        }
    }

    /**
     * 根据数据库方言获取适当大小写的字段名
     */
    public static String getFieldNameWithCase(String fieldName, Dialect dialect) {
        if (dialect.isOracleFamily()) {
            return fieldName.toUpperCase();
        } else if (dialect.isPostgresFamily()) {
            return fieldName.toLowerCase();
        } else {
            return fieldName;
        }
    }
}
