package com.af.v4.system.runtime.upgrade;

import com.af.v4.system.common.core.enums.EnvType;
import com.af.v4.system.common.core.exception.ServiceException;
import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.redis.RedisService;
import jakarta.annotation.PostConstruct;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.TreeMap;

@Component
public final class UpgradePathManager {
    private static final Logger LOGGER = LoggerFactory.getLogger(UpgradePathManager.class);
    private final RedisService redisService;
    private final ApplicationService applicationService;
    private final BaseUpgradeAppBootstrap upgradeAppBootstrap;
    private final BaseUpgradeAppBootstrap upgradeCoreBootstrap;

    public UpgradePathManager(
            RedisService redisService,
            ApplicationService applicationService,
            @Qualifier("upgradeAppBootstrap") BaseUpgradeAppBootstrap upgradeAppBootstrap,
            @Qualifier("upgradeCoreBootstrap") BaseUpgradeAppBootstrap upgradeCoreBootstrap
    ) {
        this.redisService = redisService;
        this.applicationService = applicationService;
        this.upgradeAppBootstrap = upgradeAppBootstrap;
        this.upgradeCoreBootstrap = upgradeCoreBootstrap;
    }

    public enum UpgradeType {
        /**
         * 核心升级
         */
        CORE_UPGRADE,
        /**
         * 应用升级
         */
        APP_UPGRADE
    }

    /**
     * 获取升级标识key
     * @param upgradeType 升级类型
     * @return 升级标识key
     */
    private String getUpgradeKey(UpgradeType upgradeType) {
        return switch (upgradeType) {
            case CORE_UPGRADE -> "System@CoreVersion";
            case APP_UPGRADE -> "System@AppVersion-" + applicationService.getApplicationName();
        };
    }
    /**
     * 获取版本号
     * @param upgradeType 升级类型
     * @return 版本号
     */
    private Integer getVersionCode(UpgradeType upgradeType) {
        return switch (upgradeType) {
            case CORE_UPGRADE -> ApplicationService.getSystemV4VersionCode();
            case APP_UPGRADE -> ApplicationService.getAppVersionCode();
        };
    }

    /**
     * 获取版本字符串
     * @param upgradeType 升级类型
     * @return 版本字符串
     */
    private String getVersion(UpgradeType upgradeType) {
        return switch (upgradeType) {
            case CORE_UPGRADE -> ApplicationService.getSystemV4Version();
            case APP_UPGRADE -> ApplicationService.getAppVersion();
        };
    }

    /**
     * 构建升级路径
     * @param upgradeType 升级类型
     * @return 待升级的升级执行器和来源版本
     */
    private UpgradePath buildUpgradePath(UpgradeType upgradeType) {
        LOGGER.info("[{}] 检查升级...", applicationService.getApplicationName());
        String upgradeKey = this.getUpgradeKey(upgradeType);
        final String fromVersion = redisService.get(upgradeKey);
        int fromVersionCode;
        if (fromVersion == null) {
            fromVersionCode = 0;
        } else {
            fromVersionCode = ApplicationService.buildVersionCode(fromVersion);
        }
        String toVersion = getVersion(upgradeType);
        int toVersionCode = getVersionCode(upgradeType);
        if (fromVersionCode >= toVersionCode) {
            LOGGER.info("没有需要升级的内容.");
            redisService.set(upgradeKey, toVersion);
            return null;
        }
        final Map<Integer, UpgradePathExecutor> upgradePathExecutorMap = new TreeMap<>();
        BaseUpgradeAppBootstrap bootstrap = upgradeType == UpgradeType.CORE_UPGRADE ? upgradeCoreBootstrap : upgradeAppBootstrap;
        for (UpgradePathExecutor executor : bootstrap.getAllExecutor()) {
            if (executor.targetVersionCode() <= fromVersionCode) {
                continue;
            }
            upgradePathExecutorMap.put(executor.targetVersionCode(), executor);
        }
        if (upgradePathExecutorMap.isEmpty()) {
            LOGGER.info("没有需要升级的内容.");
            redisService.set(upgradeKey, toVersion);
            return null;
        } else {
            LOGGER.info("待升级数据包数量：{}", upgradePathExecutorMap.size());
        }
        return new UpgradePath(upgradePathExecutorMap, fromVersion);
    }

    // 执行升级路径
    public void executeUpgrade(UpgradeType upgradeType) {
        String upgradeKey = getUpgradeKey(upgradeType);
        redisService.syncLock(upgradeKey, () -> {
            // 构建升级路径
            UpgradePath upgradePath = buildUpgradePath(upgradeType);
            if (upgradePath == null) {
                return null;
            }
            // 开始升级
            upgrade(upgradeType, upgradePath);
            return null;
        });
    }

    public void upgrade(UpgradeType upgradeType, UpgradePath upgradePath) {
        // 升级标识
        String upgradeKey = this.getUpgradeKey(upgradeType);
        Map<Integer, UpgradePathExecutor> upgradePathExecutorMap = upgradePath.executors;
        String fromVersion = upgradePath.fromVersion;
        for (UpgradePathExecutor executor : upgradePathExecutorMap.values()) {
            LOGGER.info("[{}][{}] 开始进行升级 ===============> {} to {}",
                    upgradeType.name(),
                    applicationService.getApplicationName(),
                    fromVersion,
                    executor.targetVersion());
            try {
                // 执行升级
                executor.upgrade();
                // 更新升级状态
                redisService.set(upgradeKey, executor.targetVersion());
                LOGGER.info("{} => {} done.", fromVersion, executor.targetVersion());
                fromVersion = executor.targetVersion();
            } catch (Exception e) {
                LOGGER.error("服务升级失败，请检查！", e);
                try {
                    //此处睡眠是为了避免异步日志未输出完就抛出异常
                    Thread.sleep(1000);
                } catch (InterruptedException ex) {
                    throw new RuntimeException(ex);
                }
                throw new ServiceException("服务升级失败，程序已强制中止.");
            }
        }
    }

    @PostConstruct
    public void init() {
        if (applicationService.getEnvType() != EnvType.LOCAL) {
            // 对核心框架进行升级
            this.executeUpgrade(UpgradeType.CORE_UPGRADE);
            // 对应用集群进行升级
            this.executeUpgrade(UpgradeType.APP_UPGRADE);
        }
    }

    /**
     * 升级路径
     * @param executors 执行器
     * @param fromVersion 当前版本
     */
    public record UpgradePath(Map<Integer, UpgradePathExecutor> executors, String fromVersion) {
    }
}
