package com.af.v4.system.runtime.service;

import com.af.v4.system.api.RemoteAppService;
import com.af.v4.system.api.factory.DynamicFeignClientFactory;
import com.af.v4.system.common.core.domain.R;
import com.af.v4.system.common.core.service.ApplicationService;
import org.json.JSONObject;
import org.springframework.stereotype.Service;

import java.util.Map;

/**
 * 琉璃-应用服务
 *
 * @author Mr.river
 * @since 2.2.1
 */
@Service
public class AppService {

    private final ApplicationService applicationService;
    private final DynamicFeignClientFactory<RemoteAppService> remoteAppServiceFactory;

    public AppService(ApplicationService applicationService, DynamicFeignClientFactory<RemoteAppService> remoteAppServiceFactory) {
        this.applicationService = applicationService;
        this.remoteAppServiceFactory = remoteAppServiceFactory;
    }

    /**
     * 获取本地应用信息
     *
     * @return 应用信息
     */
    public JSONObject getAppInfo() {
        return applicationService.getValues();
    }

    /**
     * 本地应用版本号比较
     *
     * @param version 传入的要比较的版本号
     * @return 如果传入版本号等于应用版本号，则值为0；
     * 如果传入版本号小于应用版本号，则值小于0；
     * 如果传入版本号大于应用版本号，则值大于0
     */
    public Integer compareAppVersion(String version) {
        int currVersionCode = ApplicationService.buildVersionCode(version);
        int targetVersionCode = ApplicationService.getAppVersionCode();
        return Integer.compare(currVersionCode, targetVersionCode);
    }

    /**
     * 本地琉璃框架版本号比较
     *
     * @param version 传入的要比较的版本号
     * @return 如果传入版本号等于琉璃框架版本号，则值为0；
     * 如果传入版本号小于琉璃框架版本号，则值小于0；
     * 如果传入版本号大于琉璃框架版本号，则值大于0
     */
    public Integer compareLiuLiVersion(String version) {
        int currVersionCode = ApplicationService.buildVersionCode(version);
        int targetVersionCode = ApplicationService.getSystemV4VersionCode();
        return Integer.compare(currVersionCode, targetVersionCode);
    }

    /**
     * 获取远程应用信息
     *
     * @param serviceName 服务名
     * @return 调用结果，如果接口调用失败，则返回null
     */
    public JSONObject getRemoteAppInfo(String serviceName) {
        RemoteAppService remoteAppService = remoteAppServiceFactory.getFeignClient(RemoteAppService.class, serviceName);
        R<Map<String, Object>> result = remoteAppService.getAppInfo(serviceName);
        return result.getCode() == 200 ? new JSONObject(result.getData()) : null;
    }

    /**
     * 远程应用版本号比较
     *
     * @param serviceName 服务名
     * @param version     传入的要比较的版本号
     * @return 如果传入版本号等于应用版本号，则值为0；
     * 如果传入版本号小于应用版本号，则值小于0；
     * 如果传入版本号大于应用版本号，则值大于0；
     * 如果接口调用失败，则返回null
     */
    public Integer compareRemoteAppVersion(String serviceName, String version) {
        RemoteAppService remoteAppService = remoteAppServiceFactory.getFeignClient(RemoteAppService.class, serviceName);
        R<Integer> result = remoteAppService.compareAppVersion(serviceName, version);
        return result.getCode() == 200 ? result.getData() : null;
    }

    /**
     * 远程琉璃框架版本号比较
     *
     * @param serviceName 服务名
     * @param version     传入的要比较的版本号
     * @return 如果传入版本号等于琉璃框架版本号，则值为0；
     * 如果传入版本号小于琉璃框架版本号，则值小于0；
     * 如果传入版本号大于琉璃框架版本号，则值大于0；
     * 如果接口调用失败，则返回null
     */
    public Integer compareRemoteLiuLiVersion(String serviceName, String version) {
        RemoteAppService remoteAppService = remoteAppServiceFactory.getFeignClient(RemoteAppService.class, serviceName);
        R<Integer> result = remoteAppService.compareLiuLiVersion(serviceName, version);
        return result.getCode() == 200 ? result.getData() : null;
    }
}
