package com.af.v4.system.runtime.upgrade;

import com.af.v4.system.common.core.service.ApplicationService;
import com.af.v4.system.common.core.utils.SpringUtils;
import com.af.v4.system.common.jpa.service.SqlService;
import com.af.v4.system.common.liuli.config.service.LiuLiConfigService;
import com.af.v4.system.common.logic.service.LogicService;
import com.af.v4.system.runtime.dto.UpgradeField;
import com.af.v4.system.runtime.dto.UpgradeTable;
import org.json.JSONArray;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;

/**
 * 升级路径执行器
 */
@Component
public abstract class UpgradePathExecutor {

    protected final LogicService logicService;
    protected final SqlService sqlService;
    protected final ApplicationService applicationService;
    protected final LiuLiConfigService liuLiConfigService;

    protected UpgradePathExecutor() {
        this.logicService = SpringUtils.getBean(LogicService.class);
        this.sqlService = SpringUtils.getBean(SqlService.class);
        this.applicationService = SpringUtils.getBean(ApplicationService.class);
        this.liuLiConfigService = SpringUtils.getBean(LiuLiConfigService.class);
    }

    @Transactional(rollbackFor = Exception.class)
    public abstract void upgrade();

    public abstract String targetVersion();

    public final int targetVersionCode() {
        return ApplicationService.buildVersionCode(targetVersion());
    }

    /**
     * 批量清理琉璃缓存
     *
     * @param cacheKeys 缓存键列表
     */
    protected final void removeCacheKeys(List<String> cacheKeys) {
        String namespaceName = applicationService.getApplicationName();
        liuLiConfigService.removeCacheKeys(namespaceName, "prod", cacheKeys);
    }

    /**
     * 批量创建字段，如果表不存在则创建表
     *
     * @param fields 字段列表
     */
    protected final void addFields(List<UpgradeField> fields) {
        for (UpgradeField field : fields) {
            String searchSql = field.getSearchSql();
            JSONArray searchResult = sqlService.querySQL("UpgradeField", searchSql);
            if (searchResult.isEmpty()) {
                String addSql = field.getAddSql();
                sqlService.execSQL("UpgradeField", addSql);
            }
        }
    }

    /**
     * 批量创建表
     *
     * @param tables 表定义列表
     */
    protected final void createTables(List<UpgradeTable> tables) {
        for (UpgradeTable table : tables) {
            String searchSql = table.getSearchSql();
            JSONArray result = sqlService.querySQL("CreateTable", searchSql);
            if (result.isEmpty()) {
                String createSql = table.getCreateSql();
                sqlService.execSQL("CreateTable", createSql);
            }
        }
    }

}
